/*
    Tiny_Tetris_SDL :  A tiny and simply Tetris with SDL
    Copyright (C) 2008-2009 Loc Bartoletti
    http://coder.tuxfamily.org/

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
     the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    Loc Bartoletti
    coder@tuxfamily.org
*/
#include "graph.h"


/*!
* \file graph.c
* \brief Graphicals functions
* \author L. Bartoletti
* \version 1.3
* \date 08 March 2009
*/

/*!
* \fn void dessineInterface(SDL_Surface *global_ecran, int fullscreen)
* \brief Draw Tetronimo
* \param *global_ecran main screen
* \param fullscreen enable or disable variable for fullscreen mode
*/
void dessineInterface(SDL_Surface *global_ecran, int fullscreen)
{

    /** \code

    /** in case of fullscreen make correction */
    int fullscreen_bord_x, fullscreen_bord_y;
    if (fullscreen == 1)
    {
        fullscreen_bord_x = FULLSCREEN_BORD_X;
        fullscreen_bord_y = FULLSCREEN_BORD_Y;

        /* and gimme a title */
        /* title */
        /* Create Surface */
        SDL_Surface *txt = NULL;
        /* Create font and load it */
        TTF_Font *police = TTF_OpenFont("data/Soviet2.ttf", 32);
        if (police == NULL)
        {
#if(!WIN32)
            police = TTF_OpenFont("/usr/local/share/Tiny_Tetris_SDL/data/Soviet2.ttf", 32);
            if (police==NULL)
            {
                fprintf(stderr, "Impossible de charger la police.\n");
                exit(0);
            }
#else
            fprintf(stderr, "Impossible de charger la police.\n");
            exit(0);
#endif
        }
        /* Write title */
        char *texte = malloc(sizeof(char*) * strlen("TINY TETRIS SDL"));
        if (texte == NULL)
        {
            fprintf(stderr, "erreur lors de l'allocation de texte\n");
            exit(0);
        }
        sprintf(texte, "TINY TETRIS SDL");
        SDL_Color couleurTexteJaune = {255, 255, 0, 0};
        SDL_Rect position_txt;
        txt = TTF_RenderText_Solid(police, texte, couleurTexteJaune);
        position_txt.x = global_ecran->w/2 - txt->w/2;
        position_txt.y = txt->h;
        position_txt.h = txt->h;
        position_txt.w = txt->w;
        SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 255, 0, 0));
        SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);


        /* Draw St Basil */
        SDL_Surface *stbasil = NULL;
        stbasil = IMG_Load("data/stbasil.png");
        if (stbasil!=NULL)
        {
#if(!WIN32)
            stbasil = IMG_Load("data/stbasil.png");
            if (stbasil!=NULL)
            {
                SDL_Rect pos;
                pos.x = fullscreen_bord_x/2  - stbasil->w/2;
                pos.y = global_ecran->h/2 - stbasil->h/2;
                SDL_BlitSurface(stbasil, NULL, global_ecran, &pos);
            }
#else
            SDL_Rect pos;
            pos.x = fullscreen_bord_x/2  - stbasil->w/2;
            pos.y = global_ecran->h/2 - stbasil->h/2;
            SDL_BlitSurface(stbasil, NULL, global_ecran, &pos);
#endif
        }


    }
    else
    {
        fullscreen_bord_x = 0;
        fullscreen_bord_y = 0;
    }

    /* Draw the tetromino */
    SDL_Rect graph;
    /* vertical left line */
    graph.x = BORD_X + fullscreen_bord_x;
    graph.y = BORD_Y + fullscreen_bord_y;
    graph.w = BLOC;
    graph.h = BLOC*NB_LIGNE;
    SDL_FillRect(global_ecran, &graph, SDL_MapRGB(global_ecran->format, 204, 204, 204));

    /* vertical right line */
    graph.x = BORD_X + NB_COL*BLOC + BLOC + fullscreen_bord_x;
    SDL_FillRect(global_ecran, &graph, SDL_MapRGB(global_ecran->format, 204, 204, 204));

    /* horizontal down line */
    graph.x = BORD_X + fullscreen_bord_x;
    graph.y = BLOC*NB_LIGNE + BORD_Y + fullscreen_bord_y;
    graph.w = BLOC + BLOC*NB_COL + BLOC;
    graph.h = BLOC;
    SDL_FillRect(global_ecran, &graph, SDL_MapRGB(global_ecran->format, 204, 204, 204));

    /* frame of future block */
    /* vertical left line */
    graph.x = BORD_X + NB_COL*BLOC + ESPACE + BLOC + fullscreen_bord_x;
    graph.y = BORD_Y + fullscreen_bord_y;
    graph.w = BLOC;
    graph.h = BLOC * 7; /* 7 = frame height */
    SDL_FillRect(global_ecran, &graph, SDL_MapRGB(global_ecran->format, 204, 204, 204));

    /* vertical right line */
    graph.x += graph.h;
    SDL_FillRect(global_ecran, &graph, SDL_MapRGB(global_ecran->format, 204, 204, 204));

    /* horizontal up line */
    graph.x = BORD_X + NB_COL*BLOC + ESPACE + BLOC + fullscreen_bord_x;
    graph.y = BORD_Y + fullscreen_bord_y;
    graph.w = graph.h;
    graph.h = BLOC;
    SDL_FillRect(global_ecran, &graph, SDL_MapRGB(global_ecran->format, 204, 204, 204));

    /* horizontal down line */
    graph.y = BORD_Y + BLOC*6 + fullscreen_bord_y;
    SDL_FillRect(global_ecran, &graph, SDL_MapRGB(global_ecran->format, 204, 204, 204));





    /** \endcode */
}

/*!
* \fn void posProchBloc(SDL_Surface *ecran, SDL_Surface *nBloc, int fullscreen)
* \brief Draw the next bloc in its box
* \param *ecran main screen
* \param *nBloc SDL_Surface of the next bloc
* \param fullscreen enable or disable variable for fullscreen mode
*/
void posProchBloc(SDL_Surface *ecran, SDL_Surface *nBloc, int fullscreen)
{

    /** \code
    /** in case of fullscreen make correction */
    int fullscreen_bord_x, fullscreen_bord_y;
    if (fullscreen == 1)
    {
        fullscreen_bord_x = FULLSCREEN_BORD_X;
        fullscreen_bord_y = FULLSCREEN_BORD_Y;
    }
    else
    {
        fullscreen_bord_x = 0;
        fullscreen_bord_y = 0;
    }
    /* draw the next bloc */

    SDL_Rect posNBloc;

    /* set coordinate of the position of block in frame */
    posNBloc.x = (BORD_X + NB_COL*BLOC + ESPACE + BLOC) + ((BLOC * 7)/2) - (nBloc->w/2) + fullscreen_bord_x;
    posNBloc.y = BORD_Y + ((BLOC * 7)/2) - (nBloc->h/2) + fullscreen_bord_y;

    /* and draw it */
    SDL_BlitSurface(nBloc, NULL, ecran, &posNBloc);


    /** \endcode */
}

/*!
* \fn SDL_Surface *chargeIMG(int numIMG)
* \brief Function made to load image of the bloc
* \param numIMG the number of the image in data folder
*/
SDL_Surface *chargeIMG(int numIMG)
{


    /** \code

    /* Load the current block */
    SDL_Surface *retIMG = NULL;

    /* each image is stocked in data/ */
    /* add the number of the filename */
    char *filename = malloc(sizeof(char*) * (strlen("data/")+3+strlen(".png")));
    if (filename == NULL)
    {
        fprintf(stderr, "erreur d'allocation mmoire pour l'image : %s\n", filename);
        exit(0);
    }
    sprintf(filename, "data/%d.png", numIMG);

    /* load it */
    retIMG = IMG_Load(filename);
    if (retIMG == NULL)
    {
#if(!WIN32)
        char *filename = malloc(sizeof(char*) * (strlen("/usr/local/share/Tiny_Tetris_SDL/data/")+3+strlen(".png")));
        if (filename == NULL)
        {
            fprintf(stderr, "erreur d'allocation mmoire pour l'image : %s\n", filename);
            exit(0);
        }
        sprintf(filename, "/usr/local/share/Tiny_Tetris_SDL/data/%d.png", numIMG);
        retIMG = IMG_Load(filename);
        if (retIMG == NULL)
        {
            fprintf(stderr, "erreur de chargement de l'image : %s\n", filename);
            exit(0);
        }
#else
        fprintf(stderr, "erreur de chargement de l'image : %s\n", filename);
        exit(0);
#endif
    }
    /* return the SDL_Surface */
    return retIMG;


    /** \endcode */
}

/*!
* \fn Uint32 getpixel(SDL_Surface *surface, int x, int y)
* \brief SDL Function
* \param *surface the SDL_Surface where get pixel
* \param x X position of the pixel
* \param y Y position of the pixel
*/
Uint32 getpixel(SDL_Surface *surface, int x, int y)
{

    /** \code
    /* See SDL for documentation */
    int bpp = surface->format->BytesPerPixel;
    /* Here p is the address to the pixel we want to retrieve */
    Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * bpp;

    switch (bpp)
    {
    case 1:
        return *p;

    case 2:
        return *(Uint16 *)p;

    case 3:
        if (SDL_BYTEORDER == SDL_BIG_ENDIAN)
            return p[0] << 16 | p[1] << 8 | p[2];
        else
            return p[0] | p[1] << 8 | p[2] << 16;

    case 4:
        return *(Uint32 *)p;

    default:
        return 0;       /* shouldn't happen, but avoids warnings */
    }
    /** \endcode */
}


/*!
* \fn void putpixel(SDL_Surface *surface, int x, int y, Uint32 pixel)
* \brief SDL Function
* \param *surface the SDL_Surface where put pixel
* \param x X position of the pixel
* \param y Y position of the pixel
* \param pixel the pixel to put on surface
*/
void putpixel(SDL_Surface *surface, int x, int y, Uint32 pixel)
{
    /** \code
    /* See SDL for documentation */
    int bpp = surface->format->BytesPerPixel;
    /* Here p is the address to the pixel we want to set */
    Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * bpp;

    switch (bpp)
    {
    case 1:
        *p = pixel;
        break;

    case 2:
        *(Uint16 *)p = pixel;
        break;

    case 3:
        if (SDL_BYTEORDER == SDL_BIG_ENDIAN)
        {
            p[0] = (pixel >> 16) & 0xff;
            p[1] = (pixel >> 8) & 0xff;
            p[2] = pixel & 0xff;
        }
        else
        {
            p[0] = pixel & 0xff;
            p[1] = (pixel >> 8) & 0xff;
            p[2] = (pixel >> 16) & 0xff;
        }
        break;

    case 4:
        *(Uint32 *)p = pixel;
        break;
    }
    /** \endcode */
}

/*!
* \fn void affichage(SDL_Surface *global_ecran, int niveau, int nbLignes, long points, int fullscreen)
* \brief Draw informations
* \param *global_ecran main screen
* \param niveau level
* \param nbLignes number of lines for next level calcul
* \param points the number of points
* \param fullscreen enable or disable variable for fullscreen mode
*/
void affichage(SDL_Surface *global_ecran, int niveau, int nbLignes, long points, int fullscreen)
{


    /** \code

    /** in case of fullscreen make correction */
    int fullscreen_bord_x, fullscreen_bord_y;
    if (fullscreen == 1)
    {
        fullscreen_bord_x = FULLSCREEN_BORD_X;
        fullscreen_bord_y = FULLSCREEN_BORD_Y;
    }
    else
    {
        fullscreen_bord_x = 0;
        fullscreen_bord_y = 0;
    }
    /* Create Surface */
    SDL_Surface *txt = NULL;
    /* Create font and load it */
    TTF_Font *police = TTF_OpenFont("data/Soviet2.ttf", 12);
    if (police == NULL)
    {
#if(!WIN32)
        police = TTF_OpenFont("/usr/local/share/Tiny_Tetris_SDL/data/Soviet2.ttf", 12);
        if (police==NULL)
        {
            fprintf(stderr, "Impossible de charger la police.\n");
            exit(0);
        }
#else
        fprintf(stderr, "Impossible de charger la police.\n");
        exit(0);
#endif
    }
    /* Create variables */
    char *texte = malloc(sizeof(char*) * 255);
    if (texte == NULL)
    {
        fprintf(stderr, "erreur lors de l'allocation de texte\n");
        exit(0);
    }
    SDL_Color couleurTexteBlanc = {255, 255, 255, 0};
    SDL_Rect position_txt;

    /* Set position */
    position_txt.x = BORD_X + NB_COL*BLOC + ESPACE + BLOC + fullscreen_bord_x;
    position_txt.y = BORD_Y + BLOC * 7 + ESPACE + fullscreen_bord_y;

    /* Write Level */
    sprintf(texte,"Level :");
    txt = TTF_RenderText_Solid(police, texte, couleurTexteBlanc);
    position_txt.h = txt->h;
    position_txt.w = global_ecran->w - position_txt.x;
    SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 0, 0, 0));
    SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);

    /* Write the number of the level */
    sprintf(texte,"%d", niveau);
    position_txt.y += txt->h + 1;
    txt = TTF_RenderText_Solid(police, texte, couleurTexteBlanc);
    position_txt.h = txt->h;
    position_txt.w = global_ecran->w - position_txt.x;
    SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 0, 0, 0));
    SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);

    /* Write Points */
    sprintf(texte,"Points :");
    position_txt.y += txt->h + 1;
    txt = TTF_RenderText_Solid(police, texte, couleurTexteBlanc);
    position_txt.h = txt->h;
    position_txt.w = global_ecran->w - position_txt.x;
    SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 0, 0, 0));
    SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);

    /* Write the points */
    sprintf(texte,"%ld", points);
    position_txt.y += txt->h + 1;
    txt = TTF_RenderText_Solid(police, texte, couleurTexteBlanc);
    position_txt.h = txt->h;
    position_txt.w = global_ecran->w - position_txt.x;
    SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 0, 0, 0));
    SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);


    /* Write Lines before next level : */
    sprintf(texte,"Lines before next level :");
    position_txt.y += txt->h + 1;
    txt = TTF_RenderText_Solid(police, texte, couleurTexteBlanc);
    position_txt.h = txt->h;
    position_txt.w = global_ecran->w - position_txt.x;
    SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 0, 0, 0));
    SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);

    /* Write the number of lines before next level : */
    /* nbLignes is the number of lines fulled in the level */
    /* so the number of lines before next level is 10 - nbLignes */
    sprintf(texte,"%d", 10-nbLignes);
    position_txt.y += txt->h + 1;
    txt = TTF_RenderText_Solid(police, texte, couleurTexteBlanc);
    position_txt.h = txt->h;
    position_txt.w = global_ecran->w - position_txt.x;
    SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 0, 0, 0));
    SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);


    /* Free it */
    TTF_CloseFont(police);
    SDL_FreeSurface(txt);
    texte = NULL;



    /** \endcode */
}

/*!
* \fn int verifGameOver(SDL_Surface *global_ecran, SDL_Surface *curBloc, SDL_Rect posBloc)
* \brief check if the game is end
* \param *global_ecran main screen
* \param *curBloc the bloc in instance
* \param posBloc position of the bloc
*/
int verifGameOver(SDL_Surface *global_ecran, SDL_Surface *curBloc, SDL_Rect posBloc)
{


    /** \code


    /* Variables for check pixel color on screen and current block */
    Uint32 pixBloc, pixEcran;
    Uint8 rEcran, gEcran, bEcran;
    Uint8 rBloc, gBloc, bBloc;
    int posX, posY;
    int x, y;
    posX = posBloc.x;
    posY = posBloc.y;
    int posBX, posBY;
    posBX = 0;
    posBY = 0;
    int result = 1;
    int sEcran, sBloc;


    /* check each pixel of block on the screen */
    /* if a block pixel colored is on a colored screen you're dead */
    for (x = 0; x < curBloc->w; x++)
    {
        for (y = 0; y < curBloc->h; y++)
        {
            pixEcran = getpixel(global_ecran, posX+x, posY+y);
            pixBloc = getpixel(curBloc, posBX+x, posBY+y);
            SDL_GetRGB(pixEcran, global_ecran->format, &rEcran, &gEcran, &bEcran);
            SDL_GetRGB(pixBloc, curBloc->format, &rBloc, &gBloc, &bBloc);

            sEcran = rEcran + gEcran + bEcran;
            sBloc = rBloc + gBloc + bBloc;


            if ((sBloc != 0) && (sEcran != 0))
            {
                result = 0;
            }
        }
    }


    /* return if you're still alive */
    return result;


    /** \endcode */
}

/*!
* \fn void gameOver(SDL_Surface *global_ecran, long points, int lignes, int niveau)
* \brief Draw the "game over" and make test if there is a new best score
* \param *global_ecran main screen
* \param points the number of points winned
* \param lignes total number of lines fulled
* \param niveau max level in game
*/
void gameOver(SDL_Surface *global_ecran, long points, int lignes, int niveau)
{

    /** \code


    /* Write the death */


    /* Create Surface */
    SDL_Surface *txt = NULL;
    /* Create font and load it */
    TTF_Font *police = TTF_OpenFont("data/Soviet2.ttf", 32);
    if (police == NULL)
    {
#if(!WIN32)
        police = TTF_OpenFont("/usr/local/share/Tiny_Tetris_SDL/data/Soviet2.ttf", 32);
        if (police==NULL)
        {
            fprintf(stderr, "Impossible de charger la police.\n");
            exit(0);
        }
#else
        fprintf(stderr, "Impossible de charger la police.\n");
        exit(0);
#endif
    }
    /* Write Game Over */
    char *texte = malloc(sizeof(char*) * strlen("GAME OVER !"));
    if (texte == NULL)
    {
        fprintf(stderr, "erreur lors de l'allocation de texte\n");
        exit(0);
    }
    sprintf(texte, "GAME OVER !");
    SDL_Color couleurTexteJaune = {255, 255, 0, 0};
    SDL_Rect position_txt;
    txt = TTF_RenderText_Solid(police, texte, couleurTexteJaune);
    position_txt.x = global_ecran->w/2 - txt->w/2 ;
    position_txt.y = global_ecran->h/2 - txt->h/2 ;
    position_txt.h = txt->h;
    position_txt.w = txt->w;
    SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 255, 0, 0));
    SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);


    /* open the file which contain the best score */
    FILE *fp;
    if ( (fp = fopen("data/score.tet", "r")) == NULL)
    {
#if(!WIN32)
        char *pathscore = malloc((strlen(getenv("HOME"))+strlen("/.Tiny_Tetris_SDL/score.tet"))*sizeof(char*));
        sprintf(pathscore, "%s/.Tiny_Tetris_SDL/score.tet", getenv("HOME"));
        if ( (fp = fopen(pathscore, "r")) == NULL)
        {
            printf("Impossible d'ouvrir le fichier score.tet\n");
            exit (EXIT_FAILURE);
        }
        free(pathscore);
#else
        printf("Impossible d'ouvrir le fichier score.tet\n");
        exit (EXIT_FAILURE);
#endif
    }


    /* make test */
    /* 1st line : best Score */
    long bestLigne, bestScore, bestLevel;
    char ligne[10];
    char * result = fgets(ligne, sizeof ligne, fp);
    if (result == NULL)
    {
        fprintf(stderr, "rrr\n");
        exit(0);
    }
    bestScore = strtol(ligne, NULL, 10);

    /* 2nd line : line of best score */
    result =fgets(ligne, sizeof ligne, fp);
    if (result == NULL)
    {
        fprintf(stderr, "rrr\n");
        exit(0);
    }
    bestLigne = strtol(ligne, NULL, 10);

    /* 3rd line : level of best score */
    result =fgets(ligne, sizeof ligne, fp);
    if (result == NULL)
    {
        fprintf(stderr, "rrr\n");
        exit(0);
    }
    bestLevel = strtol(ligne, NULL, 10);
    fclose(fp);


    /* check if best score is down */
    if (bestScore < points)
    {
        /* write the new best score */
        FILE *fpw;
        if ( (fpw = fopen("data/score.tet2", "w")) == NULL)
        {
#if(!WIN32)
            char *pathscore = malloc((strlen(getenv("HOME"))+strlen("/.Tiny_Tetris_SDL/score.tet2"))*sizeof(char*));
            sprintf(pathscore, "%s/.Tiny_Tetris_SDL/score.tet2", getenv("HOME"));
            if ( (fpw = fopen(pathscore, "w")) == NULL)
            {
                printf("Impossible d'ouvrir le fichier score.tet\n");
                exit (EXIT_FAILURE);
            }
            free(pathscore);
#else
            printf("Impossible d'ouvrir le fichier score.tet\n");
            exit (EXIT_FAILURE);
#endif
        }
        fprintf(fpw, "%ld\n%d\n%d\n",points, lignes, niveau);

        /* write a congratulation on screen */
        TTF_CloseFont(police);
        police = TTF_OpenFont("data/Soviet2.ttf", 12);
        if (police == NULL)
        {
#if(!WIN32)
            police = TTF_OpenFont("/usr/local/share/Tiny_Tetris_SDL/data/Soviet2.ttf", 12);
            if (police==NULL)
            {
                fprintf(stderr, "Impossible de charger la police.\n");
                exit(0);
            }
#else
            fprintf(stderr, "Impossible de charger la police.\n");
            exit(0);
#endif
        }
        texte = malloc(sizeof(char*) * strlen("But you have the best Score"));
        if (texte == NULL)
        {
            fprintf(stderr, "erreur lors de l'allocation de texte\n");
            exit(0);
        }
        sprintf(texte, "But you have the best Score");
        SDL_Color couleurTexteJaune = {255, 255, 0, 0};
        SDL_Rect position_txt;

        txt = TTF_RenderText_Solid(police, texte, couleurTexteJaune);


        position_txt.y = global_ecran->h - global_ecran->h/4;
        position_txt.x = global_ecran->w/2 - txt->w/2 ;
        position_txt.h = txt->h;
        position_txt.w = txt->w;
        SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 255, 0, 0));

        SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);

        fclose(fpw);

        if (remove("data/score.tet")!=0)
        {
            char *pathscore = malloc((strlen(getenv("HOME"))+strlen("/.Tiny_Tetris_SDL/score.tet"))*sizeof(char*));
            sprintf(pathscore, "%s/.Tiny_Tetris_SDL/score.tet", getenv("HOME"));
            remove(pathscore);
            free(pathscore);
        }
        if (rename("data/score.tet2", "data/score.tet")!=0)
        {
            char *pathscore = malloc((strlen(getenv("HOME"))+strlen("/.Tiny_Tetris_SDL/score.tet"))*sizeof(char*));
            char *pathscore2 = malloc((strlen(getenv("HOME"))+strlen("/.Tiny_Tetris_SDL/score.tet2"))*sizeof(char*));
            sprintf(pathscore, "%s/.Tiny_Tetris_SDL/score.tet", getenv("HOME"));
            sprintf(pathscore2, "%s/.Tiny_Tetris_SDL/score.tet2", getenv("HOME"));
            rename(pathscore2,pathscore);
            free(pathscore);
            free(pathscore2);
        }
    }


    /* write if you want restart */
    TTF_CloseFont(police);
    police = TTF_OpenFont("data/Soviet2.ttf", 12);
    if (police == NULL)
    {
#if(!WIN32)
        police = TTF_OpenFont("/usr/local/share/Tiny_Tetris_SDL/data/Soviet2.ttf", 12);
        if (police==NULL)
        {
            fprintf(stderr, "Impossible de charger la police.\n");
            exit(0);
        }
#else
        fprintf(stderr, "Impossible de charger la police.\n");
        exit(0);
#endif
    }
    texte = malloc(sizeof(char*) * strlen("RESTART ? Press 'R' Key."));
    if (texte == NULL)
    {
        fprintf(stderr, "erreur lors de l'allocation de texte\n");
        exit(0);
    }
    sprintf(texte, "RESTART ? Press 'R' Key.");

    txt = TTF_RenderText_Solid(police, texte, couleurTexteJaune);


    position_txt.y = global_ecran->h - global_ecran->h/8;
    position_txt.x = global_ecran->w/2 - txt->w/2 ;
    position_txt.h = txt->h;
    position_txt.w = txt->w;
    SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 255, 0, 0));

    SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);


    /* free it */
    TTF_CloseFont(police);
    SDL_FreeSurface(txt);
    texte = NULL;

}

/*!
* \fn void affichBest(SDL_Surface *global_ecran)
* \brief Get informations and draw the best score
* \param *global_ecran main screen
*/
void affichBest(SDL_Surface *global_ecran)
{


    /** \code

    /* load best score */
    FILE *fp;
    if ( (fp = fopen("data/score.tet", "r")) == NULL)
    {
#if(!WIN32)
        char *pathscore = malloc(strlen((getenv("HOME"))+strlen("/.Tiny_Tetris_SDL/score.tet"))*sizeof(char*));
        sprintf(pathscore, "%s/.Tiny_Tetris_SDL/score.tet", getenv("HOME"));
        if ( (fp = fopen(pathscore, "r")) == NULL)
        {
            sprintf(pathscore, "mkdir %s/.Tiny_Tetris_SDL", getenv("HOME"));
            system(pathscore);
            sprintf(pathscore, "%s/.Tiny_Tetris_SDL/score.tet", getenv("HOME"));
            if ( (fp = fopen(pathscore, "w")) == NULL)
            {
                printf("Impossible d'ouvrir le fichier score.tet\n");
                exit (EXIT_FAILURE);
            }
            else
            {
                fprintf(fp, "0\n0\n0\n");
                fclose(fp);
                if ( (fp = fopen(pathscore, "r")) == NULL)
                {
                    printf("Impossible d'ouvrir le fichier score.tet\n");
                    exit (EXIT_FAILURE);
                }

            }
        }
        free(pathscore);
#else
        printf("Impossible d'ouvrir le fichier score.tet\n");
        exit (EXIT_FAILURE);
#endif
    }

    /* 1st line : best Score */
    long bestLigne, bestScore, bestLevel;
    char ligne[10];
    char *result = fgets(ligne, sizeof ligne, fp);
    if (result == NULL)
    {
        fprintf(stderr, "rrr\n");
        exit(0);
    }
    bestScore = strtol(ligne, NULL, 10);

    /* 2nd line : line of best score */
    result =fgets(ligne, sizeof ligne, fp);
    if (result == NULL)
    {
        fprintf(stderr, "rrr\n");
        exit(0);
    }
    bestLigne = strtol(ligne, NULL, 10);

    /* 3rd line : level of best score */
    result =fgets(ligne, sizeof ligne, fp);
    if (result == NULL)
    {
        fprintf(stderr, "rrr\n");
        exit(0);
    }
    bestLevel = strtol(ligne, NULL, 10);
    fclose(fp);


    /* Write the best score on screen */
    SDL_Surface *txt = NULL;
    TTF_Font *police = TTF_OpenFont("data/Soviet2.ttf", 12);
    if (police == NULL)
    {
#if(!WIN32)
        police = TTF_OpenFont("/usr/local/share/Tiny_Tetris_SDL/data/Soviet2.ttf", 12);
        if (police==NULL)
        {
            fprintf(stderr, "Impossible de charger la police.\n");
            exit(0);
        }
#else
        fprintf(stderr, "Impossible de charger la police.\n");
        exit(0);
#endif
    }
    char *texte = malloc(sizeof(char*) * strlen("BEST SCORE : WITH LINES AND LEVEL : ")+10+10+10);
    if (texte == NULL)
    {
        fprintf(stderr, "erreur lors de l'allocation de texte\n");
        exit(0);
    }
    sprintf(texte, "BEST SCORE : %ld WITH %ld LINES AND LEVEL : %ld", bestScore, bestLigne,bestLevel);
    SDL_Color couleurTexteBlanc = {255, 255, 255, 0};
    SDL_Rect position_txt;

    txt = TTF_RenderText_Solid(police, texte, couleurTexteBlanc);


    position_txt.x = global_ecran->w/2 - txt->w/2;
    position_txt.y = global_ecran->h - txt->h - ESPACE;
    SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 0, 0, 0));

    SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);

    /** \endcode */
}


/*!
* \fn void pause(SDL_Surface *global_ecran, int choix)
* \brief Draw and undraw "pause" message
* \param *global_ecran main screen
* \param choix yes (1) or not (0)
* \param fullscreen enable or disable variable for fullscreen mode
*/
void pause(SDL_Surface *global_ecran, int choix, int fullscreen)
{


    /** \code
    /** in case of fullscreen make correction */
    int fullscreen_bord_x, fullscreen_bord_y;
    if (fullscreen == 1)
    {
        fullscreen_bord_x = FULLSCREEN_BORD_X;
        fullscreen_bord_y = FULLSCREEN_BORD_Y;
    }
    else
    {
        fullscreen_bord_x = 0;
        fullscreen_bord_y = 0;
    }

    /* prepare the pause text */
    SDL_Surface *txt = NULL;
    TTF_Font *police = TTF_OpenFont("data/Soviet2.ttf", 12);
    if (police == NULL)
    {
#if(!WIN32)
        police = TTF_OpenFont("/usr/local/share/Tiny_Tetris_SDL/data/Soviet2.ttf", 12);
        if (police==NULL)
        {
            fprintf(stderr, "Impossible de charger la police.\n");
            exit(0);
        }
#else
        fprintf(stderr, "Impossible de charger la police.\n");
        exit(0);
#endif
    }

    char *texte = malloc(sizeof(char*) * strlen("PAUSE"));
    if (texte == NULL)
    {
        fprintf(stderr, "erreur lors de l'allocation de texte\n");
        exit(0);
    }
    sprintf(texte, "PAUSE");
    SDL_Color couleurTexteBlanc = {255, 255, 255, 0};
    SDL_Rect position_txt;

    txt = TTF_RenderText_Solid(police, texte, couleurTexteBlanc);


    position_txt.x = BORD_X + NB_COL*BLOC + ESPACE + BLOC + BLOC * 7 + ESPACE + fullscreen_bord_x;
    position_txt.y = BORD_Y + fullscreen_bord_y;
    position_txt.h = txt->h;
    position_txt.w = txt->w;

    switch (choix)
    {

        /* check the choice. 0 : unpause ; 1 : pause */
    case 0:
        /* unwrite pause */
        SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 0, 0, 0));
        break;


    case 1:
        /* write pause */
        SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);
        break;

    default:
        break;
    }


    /* free it */
    TTF_CloseFont(police);
    SDL_FreeSurface(txt);
    texte = NULL;

    /** \endcode */
}


void writeTexte(SDL_Surface *global_ecran, const char* texte, SDL_Color couleur_texte, int posX, int posY)
{
    /* Create Surface */
    SDL_Surface *txt = NULL;
    /* Create font and load it */
    TTF_Font *police = TTF_OpenFont("data/Soviet2.ttf", 32);
    if (police == NULL)
    {
#if(!WIN32)
        police = TTF_OpenFont("/usr/local/share/Tiny_Tetris_SDL/data/Soviet2.ttf", 32);
        if (police==NULL)
        {
            fprintf(stderr, "Impossible de charger la police.\n");
            exit(0);
        }
#else
        fprintf(stderr, "Impossible de charger la police.\n");
        exit(0);
#endif
    }

    SDL_Rect position_txt;
    txt = TTF_RenderText_Solid(police, texte, couleur_texte);
    position_txt.x = posX;
    position_txt.y = posY;
    position_txt.h = txt->h;
    position_txt.w = txt->w;
    SDL_FillRect(global_ecran, &position_txt, SDL_MapRGB(global_ecran->format, 255, 0, 0));
    SDL_BlitSurface(txt, NULL, global_ecran, &position_txt);

}
