/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hertzbeat.warehouse.store.history.tsdb.iotdb;

import com.google.common.collect.Maps;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.apache.hertzbeat.common.entity.arrow.RowWrapper;
import org.apache.hertzbeat.common.entity.dto.Value;
import org.apache.hertzbeat.common.entity.message.CollectRep;
import org.apache.hertzbeat.common.util.JsonUtil;
import org.apache.hertzbeat.warehouse.store.history.tsdb.AbstractHistoryDataStorage;
import org.apache.hertzbeat.warehouse.store.history.tsdb.iotdb.IotDbProperties;
import org.apache.iotdb.rpc.IoTDBConnectionException;
import org.apache.iotdb.rpc.StatementExecutionException;
import org.apache.iotdb.session.pool.SessionDataSetWrapper;
import org.apache.iotdb.session.pool.SessionPool;
import org.apache.iotdb.tsfile.file.metadata.enums.TSDataType;
import org.apache.iotdb.tsfile.read.common.Field;
import org.apache.iotdb.tsfile.read.common.RowRecord;
import org.apache.iotdb.tsfile.write.record.Tablet;
import org.apache.iotdb.tsfile.write.schema.MeasurementSchema;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

@Component
@ConditionalOnProperty(prefix="warehouse.store.iot-db", name={"enabled"}, havingValue="true")
public class IotDbDataStorage
extends AbstractHistoryDataStorage {
    private static final Logger log = LoggerFactory.getLogger(IotDbDataStorage.class);
    private static final String BACK_QUOTE = "`";
    private static final String NEVER_EXPIRE = "-1";
    private static final String STORAGE_GROUP = "root.hertzbeat";
    private static final String SHOW_DATABASE = "show databases %s";
    private static final String CREATE_DATABASE = "create database %s";
    private static final String SET_TTL = "set ttl to %s %s";
    private static final String CANCEL_TTL = "unset ttl to %s";
    private static final String SHOW_DEVICES = "SHOW DEVICES %s";
    private static final String SHOW_STORAGE_GROUP = "show storage group";
    private static final String QUERY_HISTORY_SQL = "SELECT %s FROM %s WHERE Time >= now() - %s order by Time desc";
    private static final String QUERY_HISTORY_INTERVAL_WITH_INSTANCE_SQL = "SELECT FIRST_VALUE(%s), AVG(%s), MIN_VALUE(%s), MAX_VALUE(%s) FROM %s GROUP BY ([now() - %s, now()), 4h)";
    private SessionPool sessionPool;
    private long queryTimeoutInMs;

    public IotDbDataStorage(IotDbProperties iotDbProperties) {
        this.serverAvailable = this.initIotDbSession(iotDbProperties);
    }

    private boolean initIotDbSession(IotDbProperties properties) {
        SessionPool.Builder builder = new SessionPool.Builder();
        builder.host(properties.host());
        if (properties.rpcPort() != null) {
            builder.port(properties.rpcPort().intValue());
        }
        if (properties.username() != null) {
            builder.user(properties.username());
        }
        if (properties.password() != null) {
            builder.password(properties.password());
        }
        if (properties.nodeUrls() != null && !properties.nodeUrls().isEmpty()) {
            builder.nodeUrls(properties.nodeUrls());
        }
        if (properties.zoneId() != null) {
            builder.zoneId(properties.zoneId());
        }
        this.queryTimeoutInMs = properties.queryTimeoutInMs();
        this.sessionPool = builder.build();
        boolean available = this.checkConnection();
        if (!available) {
            log.error("IotDB session pool init error with check connection");
            return false;
        }
        available = this.createDatabase();
        if (!available) {
            log.error("IotDB session pool init error with create database");
            return false;
        }
        this.initTtl(properties.expireTime());
        log.info("IotDB session pool init success");
        return true;
    }

    private boolean checkConnection() {
        try {
            this.sessionPool.executeNonQueryStatement(SHOW_STORAGE_GROUP);
            return true;
        }
        catch (Exception e) {
            log.error(e.getMessage(), (Throwable)e);
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private boolean createDatabase() {
        SessionDataSetWrapper dataSet = null;
        try {
            String showDatabaseSql = String.format(SHOW_DATABASE, STORAGE_GROUP);
            dataSet = this.sessionPool.executeQueryStatement(showDatabaseSql);
            if (!dataSet.hasNext()) {
                String createDatabaseSql = String.format(CREATE_DATABASE, STORAGE_GROUP);
                this.sessionPool.executeNonQueryStatement(createDatabaseSql);
            }
            if (dataSet == null) return true;
        }
        catch (IoTDBConnectionException | StatementExecutionException e) {
            boolean bl;
            try {
                log.error("create database error, error: {}", (Object)e.getMessage());
                bl = false;
                if (dataSet == null) return bl;
            }
            catch (Throwable throwable) {
                if (dataSet == null) throw throwable;
                this.sessionPool.closeResultSet(dataSet);
                throw throwable;
            }
            this.sessionPool.closeResultSet(dataSet);
            return bl;
        }
        this.sessionPool.closeResultSet(dataSet);
        return true;
    }

    private void initTtl(String expireTime) {
        if (expireTime == null || expireTime.isEmpty()) {
            return;
        }
        try {
            if (NEVER_EXPIRE.equals(expireTime)) {
                String cancelTtlSql = String.format(CANCEL_TTL, STORAGE_GROUP);
                this.sessionPool.executeNonQueryStatement(cancelTtlSql);
            } else {
                String sstTtlSql = String.format(SET_TTL, STORAGE_GROUP, expireTime);
                this.sessionPool.executeNonQueryStatement(sstTtlSql);
            }
        }
        catch (IoTDBConnectionException | StatementExecutionException e) {
            log.error("IoTDB init ttl error, expireTime: {}, error: {}", (Object)expireTime, (Object)e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void saveData(CollectRep.MetricsData metricsData) {
        if (!this.isServerAvailable() || metricsData.getCode() != CollectRep.Code.SUCCESS) {
            return;
        }
        if (metricsData.getValues().isEmpty()) {
            log.info("[warehouse iotdb] flush metrics data {} is null, ignore.", (Object)metricsData.getInstance());
            return;
        }
        ArrayList schemaList = new ArrayList();
        HashMap tabletMap = Maps.newHashMapWithExpectedSize((int)8);
        try {
            metricsData.getFields().forEach(field -> {
                MeasurementSchema schema = new MeasurementSchema();
                schema.setMeasurementId(field.getName());
                byte type = (byte)field.getType();
                if (type == 0) {
                    schema.setType(TSDataType.DOUBLE);
                } else if (type == 1) {
                    schema.setType(TSDataType.TEXT);
                }
                schemaList.add(schema);
            });
            long now = System.currentTimeMillis();
            RowWrapper rowWrapper = metricsData.readRow();
            while (rowWrapper.hasNextRow()) {
                int rowIndex;
                rowWrapper = rowWrapper.nextRow();
                HashMap labels = Maps.newHashMapWithExpectedSize((int)8);
                rowWrapper.cellStream().forEach(cell -> {
                    if (cell.getMetadataAsBoolean("label").booleanValue() && !"&nbsp;".equals(cell.getValue())) {
                        labels.put(cell.getField().getName(), cell.getValue());
                    }
                });
                String label = JsonUtil.toJson((Object)labels);
                String deviceId = this.getDeviceId(metricsData.getApp(), metricsData.getMetrics(), metricsData.getInstance(), label, true);
                if (tabletMap.containsKey(label)) {
                    ++now;
                } else {
                    tabletMap.put(label, new Tablet(deviceId, schemaList));
                }
                Tablet tablet = (Tablet)tabletMap.get(label);
                ++tablet.rowSize;
                tablet.addTimestamp(rowIndex, now);
                rowWrapper.cellStream().forEach(cell -> {
                    if ("&nbsp;".equals(cell.getValue())) {
                        tablet.addValue(cell.getField().getName(), rowIndex, null);
                        return;
                    }
                    Byte type = cell.getMetadataAsByte("type");
                    if (type == 0) {
                        tablet.addValue(cell.getField().getName(), rowIndex, (Object)Double.parseDouble(cell.getValue()));
                    } else if (type == 1) {
                        tablet.addValue(cell.getField().getName(), rowIndex, (Object)cell.getValue());
                    }
                });
            }
            for (Tablet tablet : tabletMap.values()) {
                this.sessionPool.insertTablet(tablet, true);
            }
        }
        catch (Exception e) {
            log.error(e.getMessage(), (Throwable)e);
        }
        finally {
            for (Tablet tablet : tabletMap.values()) {
                tablet.reset();
            }
            tabletMap.clear();
        }
    }

    @Override
    public Map<String, List<Value>> getHistoryMetricData(String instance, String app, String metrics, String metric, String history) {
        HashMap<String, List<Value>> instanceValuesMap = new HashMap<String, List<Value>>(8);
        if (!this.isServerAvailable()) {
            log.error("\n\t---------------IotDb Init Failed---------------\n\t--------------Please Config IotDb--------------\n\t----------Can Not Use Metric History Now----------\n");
            return instanceValuesMap;
        }
        String deviceId = this.getDeviceId(app, metrics, instance, null, true);
        String selectSql = "";
        try {
            List<String> devices = this.queryAllDevices(deviceId);
            if (devices.isEmpty()) {
                log.warn("no iot device found for deviceId: {}", (Object)deviceId);
                return instanceValuesMap;
            }
            for (String device : devices) {
                selectSql = String.format(QUERY_HISTORY_SQL, this.addQuote(metric), device, history);
                String labels = this.extractLabelsFromDevice(device, deviceId);
                this.handleHistorySelect(selectSql, labels, instanceValuesMap);
            }
        }
        catch (IoTDBConnectionException | StatementExecutionException e) {
            log.error("select error history sql: {}", (Object)selectSql);
            log.error(e.getMessage(), e);
        }
        return instanceValuesMap;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleHistorySelect(String selectSql, String labels, Map<String, List<Value>> instanceValuesMap) throws IoTDBConnectionException, StatementExecutionException {
        SessionDataSetWrapper dataSet = null;
        try {
            dataSet = this.sessionPool.executeQueryStatement(selectSql, this.queryTimeoutInMs);
            log.debug("iot select sql: {}", (Object)selectSql);
            while (dataSet.hasNext()) {
                RowRecord rowRecord = dataSet.next();
                long timestamp = rowRecord.getTimestamp();
                double value = ((Field)rowRecord.getFields().get(0)).getDoubleV();
                String strValue = BigDecimal.valueOf(value).setScale(4, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
                List valueList = instanceValuesMap.computeIfAbsent(labels, k -> new LinkedList());
                valueList.add(new Value(strValue, timestamp));
            }
        }
        finally {
            if (dataSet != null) {
                this.sessionPool.closeResultSet(dataSet);
            }
        }
    }

    @Override
    public Map<String, List<Value>> getHistoryIntervalMetricData(String instance, String app, String metrics, String metric, String history) {
        HashMap<String, List<Value>> instanceValuesMap = new HashMap<String, List<Value>>(8);
        if (!this.isServerAvailable()) {
            log.error("\n\t---------------IotDb Init Failed---------------\n\t--------------Please Config IotDb--------------\n\t----------Can Not Use Metric History Now----------\n");
            return instanceValuesMap;
        }
        String deviceId = this.getDeviceId(app, metrics, instance, null, true);
        List<String> devices = this.queryAllDevices(deviceId);
        if (devices.isEmpty()) {
            log.warn("no iot device found for deviceId: {}", (Object)deviceId);
            return instanceValuesMap;
        }
        for (String device : devices) {
            String selectSql = String.format(QUERY_HISTORY_INTERVAL_WITH_INSTANCE_SQL, this.addQuote(metric), this.addQuote(metric), this.addQuote(metric), this.addQuote(metric), device, history);
            String labels = this.extractLabelsFromDevice(device, deviceId);
            this.handleHistoryIntervalSelect(selectSql, labels, instanceValuesMap);
        }
        return instanceValuesMap;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleHistoryIntervalSelect(String selectSql, String instance, Map<String, List<Value>> instanceValuesMap) {
        SessionDataSetWrapper dataSet = null;
        try {
            dataSet = this.sessionPool.executeQueryStatement(selectSql, this.queryTimeoutInMs);
            log.debug("iot select sql: {}", (Object)selectSql);
            while (dataSet.hasNext()) {
                RowRecord rowRecord = dataSet.next();
                if (rowRecord.hasNullField()) continue;
                long timestamp = rowRecord.getTimestamp();
                double origin = ((Field)rowRecord.getFields().get(0)).getDoubleV();
                String originStr = BigDecimal.valueOf(origin).setScale(4, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
                double avg = ((Field)rowRecord.getFields().get(1)).getDoubleV();
                String avgStr = BigDecimal.valueOf(avg).setScale(4, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
                double min = ((Field)rowRecord.getFields().get(2)).getDoubleV();
                String minStr = BigDecimal.valueOf(min).setScale(4, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
                double max = ((Field)rowRecord.getFields().get(3)).getDoubleV();
                String maxStr = BigDecimal.valueOf(max).setScale(4, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
                Value value = Value.builder().origin(originStr).mean(avgStr).min(minStr).max(maxStr).time(Long.valueOf(timestamp)).build();
                List valueList = instanceValuesMap.computeIfAbsent(instance, k -> new LinkedList());
                valueList.add(value);
            }
        }
        catch (IoTDBConnectionException | StatementExecutionException e) {
            log.error("select error history interval sql: {}", (Object)selectSql);
            log.error(e.getMessage(), e);
        }
        finally {
            if (dataSet != null) {
                this.sessionPool.closeResultSet(dataSet);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<String> queryAllDevices(String deviceId) {
        ArrayList<String> devices = new ArrayList<String>();
        ArrayList<String> sqls = new ArrayList<String>();
        sqls.add(String.format(SHOW_DEVICES, deviceId));
        sqls.add(String.format(SHOW_DEVICES, deviceId + ".*"));
        for (String sql : sqls) {
            SessionDataSetWrapper dataSet = null;
            try {
                dataSet = this.sessionPool.executeQueryStatement(sql, this.queryTimeoutInMs);
                while (dataSet.hasNext()) {
                    RowRecord rowRecord = dataSet.next();
                    devices.add(((Field)rowRecord.getFields().get(0)).getStringValue());
                }
            }
            catch (IoTDBConnectionException | StatementExecutionException e) {
                log.error("query show devices sql error. sql: {}", (Object)sql);
                log.error(e.getMessage(), e);
            }
            finally {
                if (dataSet == null) continue;
                this.sessionPool.closeResultSet(dataSet);
            }
        }
        return devices;
    }

    private String getDeviceId(String app, String metrics, String instance, String labels, boolean useQuote) {
        if (instance.contains(".") || instance.contains(":") || instance.contains("[")) {
            instance = instance.replace(".", "_").replace(":", "_").replace("[", "_").replace("]", "_");
        }
        String deviceId = "root.hertzbeat." + (useQuote ? this.addQuote(app) : app) + "." + (useQuote ? this.addQuote(metrics) : metrics) + "." + this.addQuote(instance);
        if (!(labels == null || labels.isEmpty() || labels.equals("&nbsp;") || "{}".equals(labels))) {
            deviceId = deviceId + "." + this.addQuote(labels);
        }
        return deviceId;
    }

    private String extractLabelsFromDevice(String device, String baseDeviceId) {
        if (device.length() > baseDeviceId.length() + 1) {
            String labelsPart = device.substring(baseDeviceId.length() + 1);
            return labelsPart.replace(BACK_QUOTE, "");
        }
        return "";
    }

    private String addQuote(String text) {
        if (text == null || text.isEmpty() || text.startsWith(BACK_QUOTE) && text.endsWith(BACK_QUOTE)) {
            return text;
        }
        text = text.replace("*", "-");
        text = String.format("`%s`", text);
        return text;
    }

    public void destroy() {
        if (this.sessionPool != null) {
            this.sessionPool.close();
        }
    }
}

