/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.computer.core.bsp;

import com.google.common.annotations.VisibleForTesting;
import io.etcd.jetcd.ByteSequence;
import io.etcd.jetcd.Client;
import io.etcd.jetcd.KV;
import io.etcd.jetcd.KeyValue;
import io.etcd.jetcd.Watch;
import io.etcd.jetcd.kv.DeleteResponse;
import io.etcd.jetcd.kv.GetResponse;
import io.etcd.jetcd.options.DeleteOption;
import io.etcd.jetcd.options.GetOption;
import io.etcd.jetcd.options.WatchOption;
import io.etcd.jetcd.watch.WatchEvent;
import io.etcd.jetcd.watch.WatchResponse;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import org.apache.hugegraph.computer.core.common.exception.ComputerException;
import org.apache.hugegraph.concurrent.BarrierEvent;
import org.apache.hugegraph.util.E;
import org.apache.hugegraph.util.Log;
import org.slf4j.Logger;

public class EtcdClient {
    private static final Logger LOG = Log.logger(EtcdClient.class);
    private static final Charset ENCODING = StandardCharsets.UTF_8;
    private final Client client;
    private final Watch watch;
    private final KV kv;

    public EtcdClient(String endpoints, String namespace) {
        E.checkArgumentNotNull((Object)endpoints, (String)"The endpoints can't be null", (Object[])new Object[0]);
        E.checkArgumentNotNull((Object)namespace, (String)"The namespace can't be null", (Object[])new Object[0]);
        ByteSequence namespaceSeq = ByteSequence.from((byte[])namespace.getBytes(ENCODING));
        this.client = Client.builder().endpoints(new String[]{endpoints}).namespace(namespaceSeq).build();
        this.watch = this.client.getWatchClient();
        this.kv = this.client.getKVClient();
    }

    public void put(String key, byte[] value) {
        E.checkArgument((key != null ? 1 : 0) != 0, (String)"The key can't be null.", (Object[])new Object[0]);
        E.checkArgument((value != null ? 1 : 0) != 0, (String)"The value can't be null.", (Object[])new Object[0]);
        try {
            this.kv.put(ByteSequence.from((String)key, (Charset)ENCODING), ByteSequence.from((byte[])value)).get();
        }
        catch (InterruptedException e) {
            throw new ComputerException("Interrupted while putting with key='%s'", (Throwable)e, new Object[]{key});
        }
        catch (ExecutionException e) {
            throw new ComputerException("Error while putting with key='%s'", (Throwable)e, new Object[]{key});
        }
    }

    public byte[] get(String key) {
        return this.get(key, false);
    }

    public byte[] get(String key, boolean throwException) {
        E.checkArgumentNotNull((Object)key, (String)"The key can't be null", (Object[])new Object[0]);
        try {
            ByteSequence keySeq = ByteSequence.from((String)key, (Charset)ENCODING);
            GetResponse response = (GetResponse)this.kv.get(keySeq).get();
            if (response.getCount() > 0L) {
                List kvs = response.getKvs();
                assert (kvs.size() == 1);
                return ((KeyValue)kvs.get(0)).getValue().getBytes();
            }
            if (throwException) {
                throw new ComputerException("Can't find value for key='%s'", new Object[]{key});
            }
            return null;
        }
        catch (InterruptedException e) {
            throw new ComputerException("Interrupted while getting with key='%s'", (Throwable)e, new Object[]{key});
        }
        catch (ExecutionException e) {
            throw new ComputerException("Error while getting with key='%s'", (Throwable)e, new Object[]{key});
        }
    }

    public byte[] get(String key, long timeout, long logInterval) {
        E.checkArgumentNotNull((Object)key, (String)"The key can't be null", (Object[])new Object[0]);
        E.checkArgument((timeout > 0L ? 1 : 0) != 0, (String)"The timeout must be > 0, but got: %s", (Object[])new Object[]{timeout});
        E.checkArgument((logInterval > 0L ? 1 : 0) != 0, (String)"The logInterval must be > 0, but got: %s", (Object[])new Object[]{logInterval});
        ByteSequence keySeq = ByteSequence.from((String)key, (Charset)ENCODING);
        try {
            GetResponse response = (GetResponse)this.kv.get(keySeq).get();
            if (response.getCount() > 0L) {
                List kvs = response.getKvs();
                return ((KeyValue)kvs.get(0)).getValue().getBytes();
            }
            long revision = response.getHeader().getRevision();
            return this.waitAndGetFromPutEvent(keySeq, revision, timeout, logInterval);
        }
        catch (InterruptedException e) {
            throw new ComputerException("Interrupted while getting with key='%s'", (Throwable)e, new Object[]{key});
        }
        catch (ExecutionException e) {
            throw new ComputerException("Error while getting with key='%s'", (Throwable)e, new Object[]{key});
        }
    }

    private byte[] waitAndGetFromPutEvent(ByteSequence keySeq, long revision, long timeout, long logInterval) throws InterruptedException {
        WaitEvent barrierEvent = new WaitEvent();
        Consumer<WatchResponse> consumer = watchResponse -> {
            List events = watchResponse.getEvents();
            Iterator iterator = events.iterator();
            if (iterator.hasNext()) {
                WatchEvent event = (WatchEvent)iterator.next();
                if (WatchEvent.EventType.PUT.equals((Object)event.getEventType())) {
                    KeyValue keyValue = event.getKeyValue();
                    if (keySeq.equals((Object)keyValue.getKey())) {
                        byte[] result = event.getKeyValue().getValue().getBytes();
                        barrierEvent.signalAll(result);
                        return;
                    }
                    assert (false);
                    throw new ComputerException("Expect event key '%s', found '%s'", new Object[]{keySeq.toString(ENCODING), keyValue.getKey().toString(ENCODING)});
                }
                assert (false);
                throw new ComputerException("Unexpected event type '%s'", new Object[]{event.getEventType()});
            }
        };
        WatchOption watchOption = WatchOption.newBuilder().withRevision(revision).withNoDelete(true).build();
        try (Watch.Watcher watcher = this.watch.watch(keySeq, watchOption, consumer);){
            byte[] byArray = (byte[])barrierEvent.await(timeout, logInterval, () -> LOG.info("Wait for key '{}' with timeout {}ms", (Object)keySeq.toString(ENCODING), (Object)timeout));
            return byArray;
        }
    }

    public List<byte[]> getWithPrefix(String prefix) {
        E.checkArgumentNotNull((Object)prefix, (String)"The prefix can't be null", (Object[])new Object[0]);
        try {
            ByteSequence prefixSeq = ByteSequence.from((String)prefix, (Charset)ENCODING);
            GetOption getOption = GetOption.newBuilder().withPrefix(prefixSeq).withSortOrder(GetOption.SortOrder.ASCEND).build();
            GetResponse response = (GetResponse)this.kv.get(prefixSeq, getOption).get();
            if (response.getCount() > 0L) {
                return EtcdClient.getResponseValues(response);
            }
            return Collections.emptyList();
        }
        catch (InterruptedException e) {
            throw new ComputerException("Interrupted while getting with prefix='%s'", (Throwable)e, new Object[]{prefix});
        }
        catch (ExecutionException e) {
            throw new ComputerException("Error while getting with prefix='%s'", (Throwable)e, new Object[]{prefix});
        }
    }

    public List<byte[]> getWithPrefix(String prefix, int count) {
        E.checkArgumentNotNull((Object)prefix, (String)"The prefix can't be null", (Object[])new Object[0]);
        E.checkArgument((count >= 0 ? 1 : 0) != 0, (String)"The count must be >= 0, but got: %s", (Object[])new Object[]{count});
        try {
            ByteSequence prefixSeq = ByteSequence.from((String)prefix, (Charset)ENCODING);
            GetOption getOption = GetOption.newBuilder().withPrefix(prefixSeq).withLimit((long)count).withSortOrder(GetOption.SortOrder.ASCEND).build();
            GetResponse response = (GetResponse)this.kv.get(prefixSeq, getOption).get();
            if (response.getCount() == (long)count) {
                return EtcdClient.getResponseValues(response);
            }
            throw new ComputerException("Expect %s elements, only find %s elements with prefix='%s'", new Object[]{count, response.getCount(), prefix});
        }
        catch (InterruptedException e) {
            throw new ComputerException("Interrupted while getting with prefix='%s', count=%s", (Throwable)e, new Object[]{prefix, count});
        }
        catch (ExecutionException e) {
            throw new ComputerException("Error while getting with prefix='%s', count=%s", (Throwable)e, new Object[]{prefix, count});
        }
    }

    public List<byte[]> getWithPrefix(String prefix, int count, long timeout, long logInterval) {
        E.checkArgumentNotNull((Object)prefix, (String)"The prefix can't be null", (Object[])new Object[0]);
        E.checkArgument((count >= 0 ? 1 : 0) != 0, (String)"The count must be >= 0, but got: %s", (Object[])new Object[]{count});
        E.checkArgument((logInterval >= 0L ? 1 : 0) != 0, (String)"The logInterval must be >= 0, but got: %s", (Object[])new Object[]{logInterval});
        ByteSequence prefixSeq = ByteSequence.from((String)prefix, (Charset)ENCODING);
        GetOption getOption = GetOption.newBuilder().withPrefix(prefixSeq).withSortOrder(GetOption.SortOrder.ASCEND).withLimit((long)count).build();
        try {
            GetResponse response = (GetResponse)this.kv.get(prefixSeq, getOption).get();
            if (response.getCount() == (long)count) {
                return EtcdClient.getResponseValues(response);
            }
            long revision = response.getHeader().getRevision();
            return this.waitAndPrefixGetFromPutEvent(prefixSeq, count, response.getKvs(), revision, timeout, logInterval);
        }
        catch (InterruptedException e) {
            throw new ComputerException("Interrupted while getting with prefix='%s', count=%s, timeout=%s", (Throwable)e, new Object[]{prefix, count, timeout});
        }
        catch (ExecutionException e) {
            throw new ComputerException("Error while getting with prefix='%s', count=%s, timeout=%s", (Throwable)e, new Object[]{prefix, count, timeout});
        }
    }

    private List<byte[]> waitAndPrefixGetFromPutEvent(ByteSequence prefixSeq, int count, List<KeyValue> existedKeyValues, long revision, long timeout, long logInterval) throws InterruptedException {
        ConcurrentHashMap<ByteSequence, ByteSequence> keyValues = new ConcurrentHashMap<ByteSequence, ByteSequence>();
        for (KeyValue kv : existedKeyValues) {
            keyValues.put(kv.getKey(), kv.getValue());
        }
        WaitEvent barrierEvent = new WaitEvent();
        Consumer<WatchResponse> consumer = watchResponse -> {
            List events = watchResponse.getEvents();
            for (WatchEvent event : events) {
                if (WatchEvent.EventType.PUT.equals((Object)event.getEventType())) {
                    KeyValue keyValue = event.getKeyValue();
                    keyValues.put(keyValue.getKey(), keyValue.getValue());
                    if (keyValues.size() != count) continue;
                    ArrayList<byte[]> result = new ArrayList<byte[]>(count);
                    for (ByteSequence byteSequence : keyValues.values()) {
                        result.add(byteSequence.getBytes());
                    }
                    barrierEvent.signalAll(result);
                    continue;
                }
                if (WatchEvent.EventType.DELETE.equals((Object)event.getEventType())) {
                    keyValues.remove(event.getKeyValue().getKey());
                    continue;
                }
                throw new ComputerException("Unexpected event type '%s'", new Object[]{event.getEventType()});
            }
        };
        WatchOption watchOption = WatchOption.newBuilder().withPrefix(prefixSeq).withRevision(revision).build();
        try (Watch.Watcher watcher = this.watch.watch(prefixSeq, watchOption, consumer);){
            List list = (List)barrierEvent.await(timeout, logInterval, () -> LOG.info("Wait for keys with prefix '{}' and timeout {}ms, expect {} keys but actual got {} keys", new Object[]{prefixSeq.toString(ENCODING), timeout, count, keyValues.size()}));
            return list;
        }
    }

    public long delete(String key) {
        E.checkArgumentNotNull((Object)key, (String)"The key can't be null", (Object[])new Object[0]);
        ByteSequence keySeq = ByteSequence.from((String)key, (Charset)ENCODING);
        try {
            DeleteResponse response = (DeleteResponse)this.client.getKVClient().delete(keySeq).get();
            return response.getDeleted();
        }
        catch (InterruptedException e) {
            throw new ComputerException("Interrupted while deleting '%s'", (Throwable)e, new Object[]{key});
        }
        catch (ExecutionException e) {
            throw new ComputerException("Error while deleting '%s'", (Throwable)e, new Object[]{key});
        }
    }

    public long deleteWithPrefix(String prefix) {
        E.checkArgumentNotNull((Object)prefix, (String)"The prefix can't be null", (Object[])new Object[0]);
        ByteSequence prefixSeq = ByteSequence.from((String)prefix, (Charset)ENCODING);
        DeleteOption deleteOption = DeleteOption.newBuilder().withPrefix(prefixSeq).build();
        try {
            DeleteResponse response = (DeleteResponse)this.client.getKVClient().delete(prefixSeq, deleteOption).get();
            return response.getDeleted();
        }
        catch (InterruptedException e) {
            throw new ComputerException("Interrupted while deleting with prefix '%s'", (Throwable)e, new Object[]{prefix});
        }
        catch (ExecutionException e) {
            throw new ComputerException("ExecutionException is thrown while deleting with prefix '%s'", (Throwable)e, new Object[]{prefix});
        }
    }

    public long deleteAllKvsInNamespace() {
        return this.deleteWithPrefix("");
    }

    public void close() {
        this.client.close();
    }

    @VisibleForTesting
    protected KV getKv() {
        return this.kv;
    }

    private static List<byte[]> getResponseValues(GetResponse response) {
        ArrayList<byte[]> values = new ArrayList<byte[]>((int)response.getCount());
        for (KeyValue kv : response.getKvs()) {
            values.add(kv.getValue().getBytes());
        }
        return values;
    }

    private static class WaitEvent<V> {
        private final BarrierEvent barrierEvent = new BarrierEvent();
        private V result = null;

        public void signalAll(V result) {
            this.result = result;
            this.barrierEvent.signalAll();
        }

        public V await(long timeout, long logInterval, Runnable logFunc) throws InterruptedException {
            long remaining = timeout;
            long deadline = timeout + System.currentTimeMillis();
            while (remaining > 0L) {
                if (this.barrierEvent.await(logInterval = Math.min(remaining, logInterval))) {
                    assert (this.result != null);
                    return this.result;
                }
                logFunc.run();
                remaining = deadline - System.currentTimeMillis();
            }
            throw new ComputerException("Timeout(%sms) to wait event", new Object[]{timeout});
        }
    }
}

