/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.inlong.sort.protocol.deserialization;

import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.annotation.JsonCreator;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.annotation.JsonInclude;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.annotation.JsonProperty;

import java.util.Objects;

/**
 * Debezium deserialization info
 */
public class DebeziumDeserializationInfo implements DeserializationInfo {

    private static final long serialVersionUID = 1L;

    @JsonProperty("ignore_parse_errors")
    private final boolean ignoreParseErrors;

    @JsonProperty("timestamp_format_standard")
    private final String timestampFormatStandard;

    @JsonInclude(JsonInclude.Include.NON_NULL)
    @JsonProperty("include_update_before")
    private final boolean includeUpdateBefore;

    public DebeziumDeserializationInfo(
            @JsonProperty("ignore_parse_errors") boolean ignoreParseErrors,
            @JsonProperty("timestamp_format_standard") String timestampFormatStandard) {
        this(ignoreParseErrors, timestampFormatStandard, true);
    }

    @JsonCreator
    public DebeziumDeserializationInfo(
            @JsonProperty("ignore_parse_errors") boolean ignoreParseErrors,
            @JsonProperty("timestamp_format_standard") String timestampFormatStandard,
            @JsonProperty("include_update_before") boolean includeUpdateBefore) {
        this.ignoreParseErrors = ignoreParseErrors;
        this.timestampFormatStandard = timestampFormatStandard;
        this.includeUpdateBefore = includeUpdateBefore;
    }

    @JsonProperty("ignore_parse_errors")
    public boolean isIgnoreParseErrors() {
        return ignoreParseErrors;
    }

    @JsonProperty("timestamp_format_standard")
    public String getTimestampFormatStandard() {
        return timestampFormatStandard;
    }

    @JsonProperty("include_update_before")
    public boolean isIncludeUpdateBefore() {
        return includeUpdateBefore;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DebeziumDeserializationInfo that = (DebeziumDeserializationInfo) o;
        return ignoreParseErrors == that.ignoreParseErrors
                && Objects.equals(timestampFormatStandard, that.timestampFormatStandard)
                && includeUpdateBefore == that.includeUpdateBefore;
    }

    @Override
    public int hashCode() {
        return Objects.hash(ignoreParseErrors, timestampFormatStandard, includeUpdateBefore);
    }

    @Override
    public String toString() {
        return "DebeziumDeserializationInfo{"
                + "ignoreParseErrors=" + ignoreParseErrors
                + ", timestampFormatStandard='" + timestampFormatStandard + '\''
                + ", includeUpdateBefore=" + includeUpdateBefore
                + '}';
    }
}
