/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */

#include "redis_stream_base.h"

#include "encoding.h"
#include "parse_util.h"
#include "time_util.h"

namespace redis {

const char *kErrLastEntryIdReached = "last possible entry id reached";
const char *kErrInvalidEntryIdSpecified = "Invalid stream ID specified as stream command argument";
const char *kErrDecodingStreamEntryValueFailure = "failed to decode stream entry value";
const char *errAddEntryIdSmallerThanLastGenerated =
    "The ID specified in XADD is equal or smaller than the target stream top item";
const char *errSequenceNumberOverflow =
    "Elements are too large to be stored";  // Redis responds with exactly this message
const char *errEntryIdOutOfRange = "The ID specified in XADD must be greater than 0-0";
const char *errStreamExhaustedEntryID = "The stream has exhausted the last possible ID, unable to add more items";

Status IncrementStreamEntryID(StreamEntryID *id) {
  if (id->seq == UINT64_MAX) {
    if (id->ms == UINT64_MAX) {
      // special case where 'id' is the last possible entry ID
      id->ms = 0;
      id->seq = 0;
      return {Status::RedisExecErr, kErrLastEntryIdReached};
    } else {
      id->ms++;
      id->seq = 0;
    }
  } else {
    id->seq++;
  }

  return Status::OK();
}

Status ParseStreamEntryID(const std::string &input, StreamEntryID *id) {
  auto pos = input.find('-');
  if (pos != std::string::npos) {
    auto ms_str = input.substr(0, pos);
    auto seq_str = input.substr(pos + 1);
    auto parse_ms = ParseInt<uint64_t>(ms_str, 10);
    auto parse_seq = ParseInt<uint64_t>(seq_str, 10);
    if (!parse_ms || !parse_seq) {
      return {Status::RedisParseErr, kErrInvalidEntryIdSpecified};
    }

    id->ms = *parse_ms;
    id->seq = *parse_seq;
  } else {
    auto parse_input = ParseInt<uint64_t>(input, 10);
    if (!parse_input) {
      return {Status::RedisParseErr, kErrInvalidEntryIdSpecified};
    }

    id->ms = *parse_input;
    id->seq = 0;
  }
  return Status::OK();
}

StatusOr<std::unique_ptr<NextStreamEntryIDGenerationStrategy>> ParseNextStreamEntryIDStrategy(
    const std::string &input) {
  if (input == "*") {
    return std::make_unique<AutoGeneratedEntryID>();
  }

  auto pos = input.find('-');
  if (pos != std::string::npos) {
    auto ms_str = input.substr(0, pos);
    auto seq_str = input.substr(pos + 1);

    if (ms_str == "*") {
      auto parse_seq = ParseInt<uint64_t>(seq_str, 10);
      if (!parse_seq) {
        return {Status::NotOK, kErrInvalidEntryIdSpecified};
      }

      return std::make_unique<CurrentTimestampWithSpecificSequenceNumber>(*parse_seq);
    }

    auto parse_ms = ParseInt<uint64_t>(ms_str, 10);
    if (!parse_ms) {
      return {Status::NotOK, kErrInvalidEntryIdSpecified};
    }

    if (seq_str == "*") {
      return std::make_unique<SpecificTimestampWithAnySequenceNumber>(*parse_ms);
    }

    auto parse_seq = ParseInt<uint64_t>(seq_str, 10);
    if (!parse_seq) {
      return {Status::NotOK, kErrInvalidEntryIdSpecified};
    }

    return std::make_unique<FullySpecifiedEntryID>(StreamEntryID{*parse_ms, *parse_seq});
  }

  auto parse_ms = ParseInt<uint64_t>(input, 10);
  if (!parse_ms) {
    return {Status::NotOK, kErrInvalidEntryIdSpecified};
  }

  return std::make_unique<FullySpecifiedEntryID>(StreamEntryID{*parse_ms, 0});
}

Status ParseRangeStart(const std::string &input, StreamEntryID *id) { return ParseStreamEntryID(input, id); }

Status ParseRangeEnd(const std::string &input, StreamEntryID *id) {
  auto pos = input.find('-');
  if (pos != std::string::npos) {
    auto ms_str = input.substr(0, pos);
    auto seq_str = input.substr(pos + 1);
    auto parse_ms = ParseInt<uint64_t>(ms_str, 10);
    auto parse_seq = ParseInt<uint64_t>(seq_str, 10);
    if (!parse_ms || !parse_seq) {
      return {Status::RedisParseErr, kErrInvalidEntryIdSpecified};
    }

    id->ms = *parse_ms;
    id->seq = *parse_seq;
  } else {
    auto parse_input = ParseInt<uint64_t>(input, 10);
    if (!parse_input) {
      return {Status::RedisParseErr, kErrInvalidEntryIdSpecified};
    }

    id->ms = *parse_input;
    id->seq = UINT64_MAX;
  }

  return Status::OK();
}

std::string EncodeStreamEntryValue(const std::vector<std::string> &args) {
  std::string dst;
  for (auto const &v : args) {
    PutVarint32(&dst, v.size());
    dst.append(v);
  }
  return dst;
}

Status DecodeRawStreamEntryValue(const std::string &value, std::vector<std::string> *result) {
  result->clear();
  rocksdb::Slice s(value);

  while (!s.empty()) {
    uint32_t len = 0;
    if (!GetVarint32(&s, &len)) {
      return {Status::RedisParseErr, kErrDecodingStreamEntryValueFailure};
    }

    result->emplace_back(s.data(), len);
    s.remove_prefix(len);
  }

  return Status::OK();
}

Status FullySpecifiedEntryID::GenerateID(const StreamEntryID &last_id, StreamEntryID *next_id) {
  if (last_id.ms == UINT64_MAX && last_id.seq == UINT64_MAX) {
    return {Status::RedisExecErr, errStreamExhaustedEntryID};
  }

  if (id_.ms == 0 && id_.seq == 0) {
    return {Status::RedisExecErr, errEntryIdOutOfRange};
  }

  if (id_ <= last_id) {
    return {Status::RedisExecErr, errAddEntryIdSmallerThanLastGenerated};
  }

  next_id->ms = id_.ms;
  next_id->seq = id_.seq;

  return Status::OK();
}

Status AutoGeneratedEntryID::GenerateID(const StreamEntryID &last_id, StreamEntryID *next_id) {
  uint64_t ms = util::GetTimeStampMS();
  if (ms > last_id.ms) {
    next_id->ms = ms;
    next_id->seq = 0;
    return Status::OK();
  }

  *next_id = last_id;

  return IncrementStreamEntryID(next_id);
}

Status SpecificTimestampWithAnySequenceNumber::GenerateID(const StreamEntryID &last_id, StreamEntryID *next_id) {
  if (ms_ < last_id.ms) {
    return {Status::RedisExecErr, errAddEntryIdSmallerThanLastGenerated};
  }

  if (ms_ == last_id.ms) {
    if (last_id.seq == UINT64_MAX) {
      return {Status::RedisExecErr, errSequenceNumberOverflow};
    }

    next_id->ms = last_id.ms;
    next_id->seq = last_id.seq + 1;
  } else {
    next_id->ms = ms_;
    next_id->seq = 0;
  }

  return Status::OK();
}

Status CurrentTimestampWithSpecificSequenceNumber::GenerateID(const StreamEntryID &last_id, StreamEntryID *next_id) {
  next_id->ms = util::GetTimeStampMS();
  next_id->seq = seq_;

  if (*next_id <= last_id) {
    return {Status::RedisExecErr, errAddEntryIdSmallerThanLastGenerated};
  }

  return Status::OK();
}

}  // namespace redis
