/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils.db;

import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import org.apache.hadoop.hdds.StringUtils;
import org.apache.hadoop.hdds.utils.db.BatchOperation;
import org.apache.hadoop.hdds.utils.db.CodecBuffer;
import org.apache.hadoop.hdds.utils.db.RocksDatabase;
import org.apache.hadoop.hdds.utils.db.managed.ManagedWriteBatch;
import org.apache.hadoop.hdds.utils.db.managed.ManagedWriteOptions;
import org.apache.ratis.util.TraditionalBinaryPrefix;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RDBBatchOperation
implements BatchOperation {
    static final Logger LOG = LoggerFactory.getLogger(RDBBatchOperation.class);
    private static final AtomicInteger BATCH_COUNT = new AtomicInteger();
    private final String name = "Batch-" + BATCH_COUNT.getAndIncrement();
    private final ManagedWriteBatch writeBatch;
    private final OpCache opCache = new OpCache();

    private static void debug(Supplier<String> message) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("\n{}", (Object)message.get());
        }
    }

    private static String byteSize2String(long length) {
        return TraditionalBinaryPrefix.long2String((long)length, (String)"B", (int)2);
    }

    private static String countSize2String(int count, long size) {
        return count + " (" + RDBBatchOperation.byteSize2String(size) + ")";
    }

    public RDBBatchOperation() {
        this.writeBatch = new ManagedWriteBatch();
    }

    public RDBBatchOperation(ManagedWriteBatch writeBatch) {
        this.writeBatch = writeBatch;
    }

    public String toString() {
        return this.name;
    }

    public void commit(RocksDatabase db) throws IOException {
        RDBBatchOperation.debug(() -> String.format("%s: commit %s", this.name, this.opCache.getCommitString()));
        try (Closeable ignored = this.opCache.prepareBatchWrite();){
            db.batchWrite(this.writeBatch);
        }
    }

    public void commit(RocksDatabase db, ManagedWriteOptions writeOptions) throws IOException {
        RDBBatchOperation.debug(() -> String.format("%s: commit-with-writeOptions %s", this.name, this.opCache.getCommitString()));
        try (Closeable ignored = this.opCache.prepareBatchWrite();){
            db.batchWrite(this.writeBatch, writeOptions);
        }
    }

    @Override
    public void close() {
        RDBBatchOperation.debug(() -> String.format("%s: close", this.name));
        this.writeBatch.close();
        this.opCache.clear();
    }

    public void delete(RocksDatabase.ColumnFamily family, byte[] key) throws IOException {
        this.opCache.delete(family, key);
    }

    public void put(RocksDatabase.ColumnFamily family, CodecBuffer key, CodecBuffer value) throws IOException {
        this.opCache.put(family, key, value);
    }

    public void put(RocksDatabase.ColumnFamily family, byte[] key, byte[] value) throws IOException {
        this.opCache.put(family, key, value);
    }

    private class OpCache {
        private final Map<String, FamilyCache> name2cache = new HashMap<String, FamilyCache>();

        private OpCache() {
        }

        void put(RocksDatabase.ColumnFamily f, CodecBuffer key, CodecBuffer value) {
            this.name2cache.computeIfAbsent(f.getName(), k -> new FamilyCache(f)).put(key, value);
        }

        void put(RocksDatabase.ColumnFamily f, byte[] key, byte[] value) {
            this.name2cache.computeIfAbsent(f.getName(), k -> new FamilyCache(f)).put(key, value);
        }

        void delete(RocksDatabase.ColumnFamily family, byte[] key) {
            this.name2cache.computeIfAbsent(family.getName(), k -> new FamilyCache(family)).delete(key);
        }

        Closeable prepareBatchWrite() throws IOException {
            for (Map.Entry<String, FamilyCache> e : this.name2cache.entrySet()) {
                e.getValue().prepareBatchWrite();
            }
            return this::clear;
        }

        void clear() {
            for (Map.Entry<String, FamilyCache> e : this.name2cache.entrySet()) {
                e.getValue().clear();
            }
            this.name2cache.clear();
        }

        String getCommitString() {
            int putCount = 0;
            int delCount = 0;
            int opSize = 0;
            int discardedCount = 0;
            int discardedSize = 0;
            for (FamilyCache f : this.name2cache.values()) {
                putCount += f.putCount;
                delCount += f.delCount;
                opSize = (int)((long)opSize + f.batchSize);
                discardedCount += f.discardedCount;
                discardedSize = (int)((long)discardedSize + f.discardedSize);
            }
            int opCount = putCount + delCount;
            return String.format("#put=%s, #del=%s, batchSize: %s, discarded: %s, committed: %s", putCount, delCount, RDBBatchOperation.countSize2String(opCount, opSize), RDBBatchOperation.countSize2String(discardedCount, discardedSize), RDBBatchOperation.countSize2String(opCount - discardedCount, opSize - discardedSize));
        }

        private class FamilyCache {
            private final RocksDatabase.ColumnFamily family;
            private final Map<Bytes, Object> ops = new HashMap<Bytes, Object>();
            private boolean isCommit;
            private long batchSize;
            private long discardedSize;
            private int discardedCount;
            private int putCount;
            private int delCount;

            FamilyCache(RocksDatabase.ColumnFamily family) {
                this.family = family;
            }

            void prepareBatchWrite() throws IOException {
                Preconditions.checkState((!this.isCommit ? 1 : 0) != 0, (String)"%s is already committed.", (Object)this);
                this.isCommit = true;
                for (Map.Entry<Bytes, Object> op : this.ops.entrySet()) {
                    Bytes key = op.getKey();
                    Object value = op.getValue();
                    if (value instanceof byte[]) {
                        this.family.batchPut(RDBBatchOperation.this.writeBatch, key.array(), (byte[])value);
                        continue;
                    }
                    if (value instanceof CodecBuffer) {
                        this.family.batchPut(RDBBatchOperation.this.writeBatch, key.asReadOnlyByteBuffer(), ((CodecBuffer)value).asReadOnlyByteBuffer());
                        continue;
                    }
                    if (value == Op.DELETE) {
                        this.family.batchDelete(RDBBatchOperation.this.writeBatch, key.array());
                        continue;
                    }
                    throw new IllegalStateException("Unexpected value: " + value + ", class=" + value.getClass().getSimpleName());
                }
                RDBBatchOperation.debug(this::summary);
            }

            private String summary() {
                return String.format("  %s %s, #put=%s, #del=%s", this, this.batchSizeDiscardedString(), this.putCount, this.delCount);
            }

            void clear() {
                boolean warn = !this.isCommit && this.batchSize > 0L;
                String details = warn ? this.summary() : null;
                for (Object value : this.ops.values()) {
                    if (!(value instanceof CodecBuffer)) continue;
                    ((CodecBuffer)value).release();
                }
                this.ops.clear();
                if (warn) {
                    LOG.warn("discarding changes {}", (Object)details);
                }
            }

            void putOrDelete(Bytes key, int keyLen, Object val, int valLen) {
                Object overwritten;
                Preconditions.checkState((!this.isCommit ? 1 : 0) != 0, (String)"%s is already committed.", (Object)this);
                this.batchSize += (long)(keyLen + valLen);
                Object previous = this.ops.remove(key);
                if (previous != null) {
                    int preLen;
                    boolean isPut;
                    boolean bl = isPut = previous != Op.DELETE;
                    if (!isPut) {
                        preLen = 0;
                    } else if (previous instanceof CodecBuffer) {
                        CodecBuffer previousValue = (CodecBuffer)previous;
                        preLen = previousValue.readableBytes();
                        previousValue.release();
                    } else if (previous instanceof byte[]) {
                        preLen = ((byte[])previous).length;
                    } else {
                        throw new IllegalStateException("Unexpected previous: " + previous + ", class=" + previous.getClass().getSimpleName());
                    }
                    this.discardedSize += (long)(keyLen + preLen);
                    ++this.discardedCount;
                    RDBBatchOperation.debug(() -> String.format("%s overwriting a previous %s", this, isPut ? "put (value: " + RDBBatchOperation.byteSize2String(preLen) + ")" : "del"));
                }
                Preconditions.checkState(((overwritten = this.ops.put(key, val)) == null ? 1 : 0) != 0);
                RDBBatchOperation.debug(() -> String.format("%s %s, %s; key=%s", this, valLen == 0 ? this.delString(keyLen) : this.putString(keyLen, valLen), this.batchSizeDiscardedString(), key));
            }

            void put(CodecBuffer key, CodecBuffer value) {
                ++this.putCount;
                value.getReleaseFuture().thenAccept(v -> key.release());
                this.putOrDelete(new Bytes(key), key.readableBytes(), value, value.readableBytes());
            }

            void put(byte[] key, byte[] value) {
                ++this.putCount;
                this.putOrDelete(new Bytes(key), key.length, value, value.length);
            }

            void delete(byte[] key) {
                ++this.delCount;
                this.putOrDelete(new Bytes(key), key.length, (Object)Op.DELETE, 0);
            }

            String putString(int keySize, int valueSize) {
                return String.format("put(key: %s, value: %s), #put=%s", RDBBatchOperation.byteSize2String(keySize), RDBBatchOperation.byteSize2String(valueSize), this.putCount);
            }

            String delString(int keySize) {
                return String.format("del(key: %s), #del=%s", RDBBatchOperation.byteSize2String(keySize), this.delCount);
            }

            String batchSizeDiscardedString() {
                return String.format("batchSize=%s, discarded: %s", RDBBatchOperation.byteSize2String(this.batchSize), RDBBatchOperation.countSize2String(this.discardedCount, this.discardedSize));
            }

            public String toString() {
                return RDBBatchOperation.this.name + ": " + this.family.getName();
            }
        }
    }

    static final class Bytes {
        private final byte[] array;
        private final CodecBuffer buffer;
        private final int hash;

        Bytes(CodecBuffer buffer) {
            this.array = null;
            this.buffer = Objects.requireNonNull(buffer, "buffer == null");
            this.hash = buffer.asReadOnlyByteBuffer().hashCode();
        }

        Bytes(byte[] array) {
            this.array = array;
            this.buffer = null;
            this.hash = ByteBuffer.wrap(array).hashCode();
        }

        byte[] array() {
            return this.array;
        }

        ByteBuffer asReadOnlyByteBuffer() {
            return this.buffer.asReadOnlyByteBuffer();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof Bytes)) {
                return false;
            }
            Bytes that = (Bytes)obj;
            if (this.hash != that.hash) {
                return false;
            }
            ByteBuffer thisBuf = this.array != null ? ByteBuffer.wrap(this.array) : this.asReadOnlyByteBuffer();
            ByteBuffer thatBuf = that.array != null ? ByteBuffer.wrap(that.array) : that.asReadOnlyByteBuffer();
            return thisBuf.equals(thatBuf);
        }

        public int hashCode() {
            return this.hash;
        }

        public String toString() {
            return this.array != null ? StringUtils.bytes2String((byte[])this.array) : StringUtils.bytes2String((ByteBuffer)this.asReadOnlyByteBuffer());
        }
    }

    private static enum Op {
        DELETE;

    }
}

