/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.cli.container.upgrade;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import org.apache.hadoop.hdds.StringUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.cli.container.upgrade.UpgradeManager;
import org.apache.hadoop.hdds.utils.db.DBStore;
import org.apache.hadoop.hdds.utils.db.FixedLengthStringCodec;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableIterator;
import org.apache.hadoop.io.nativeio.NativeIO;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.impl.ContainerDataYaml;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainer;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.BlockUtils;
import org.apache.hadoop.ozone.container.keyvalue.helpers.KeyValueContainerUtil;
import org.apache.hadoop.ozone.container.metadata.DatanodeSchemaThreeDBDefinition;
import org.apache.hadoop.ozone.container.metadata.DatanodeSchemaTwoDBDefinition;
import org.apache.hadoop.ozone.container.metadata.DatanodeStore;
import org.apache.hadoop.ozone.container.metadata.DatanodeStoreSchemaThreeImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UpgradeTask {
    public static final Logger LOG = LoggerFactory.getLogger(UpgradeTask.class);
    private final ConfigurationSource config;
    private final HddsVolume hddsVolume;
    private final DatanodeStoreSchemaThreeImpl datanodeStoreSchemaThree;
    private static final String BACKUP_CONTAINER_DATA_FILE_SUFFIX = ".backup";
    private static final Set<String> COLUMN_FAMILIES_NAME = new DatanodeSchemaTwoDBDefinition("", (ConfigurationSource)new OzoneConfiguration()).getMap().keySet();

    public UpgradeTask(ConfigurationSource config, HddsVolume hddsVolume, DatanodeStoreSchemaThreeImpl datanodeStoreSchemaThree) {
        this.config = config;
        this.hddsVolume = hddsVolume;
        this.datanodeStoreSchemaThree = datanodeStoreSchemaThree;
    }

    public CompletableFuture<UpgradeManager.Result> getUpgradeFutureByVolume() {
        return CompletableFuture.supplyAsync(() -> {
            UpgradeManager.Result result = new UpgradeManager.Result(this.hddsVolume);
            ArrayList<UpgradeContainerResult> resultList = new ArrayList<UpgradeContainerResult>();
            File hddsVolumeRootDir = this.hddsVolume.getHddsRootDir();
            Preconditions.checkNotNull((Object)hddsVolumeRootDir, (Object)"hddsVolumeRootDircannot be null");
            File[] storageDirs = hddsVolumeRootDir.listFiles(File::isDirectory);
            if (storageDirs == null) {
                result.fail(new Exception("Storage dir not found for the volume " + hddsVolumeRootDir + ", skipped upgrade."));
                return result;
            }
            if (storageDirs.length > 0) {
                File clusterIDDir;
                File idDir = clusterIDDir = new File(this.hddsVolume.getStorageDir(), this.hddsVolume.getClusterID());
                if (storageDirs.length == 1 && !clusterIDDir.exists()) {
                    idDir = storageDirs[0];
                } else if (!clusterIDDir.exists()) {
                    result.fail(new Exception("Volume " + hddsVolumeRootDir + " is in an inconsistent state. Expected clusterID directory " + clusterIDDir + " not found."));
                    return result;
                }
                LOG.info("Start to upgrade containers on volume {}", (Object)hddsVolumeRootDir);
                File currentDir = new File(idDir, "current");
                if (!currentDir.exists()) {
                    result.fail(new Exception("Storage current dir not found for the volume " + currentDir + ", skipped upgrade."));
                    return result;
                }
                File[] containerTopDirs = currentDir.listFiles();
                if (containerTopDirs != null) {
                    for (File containerTopDir : containerTopDirs) {
                        try {
                            List<UpgradeContainerResult> results = this.upgradeSubContainerDir(containerTopDir);
                            resultList.addAll(results);
                        }
                        catch (IOException e) {
                            result.fail(e);
                            return result;
                        }
                    }
                }
            } else {
                result.fail(new Exception("Storage dir not found for the volume " + hddsVolumeRootDir + ", skipped upgrade."));
                return result;
            }
            result.setResultList(resultList);
            result.success();
            return result;
        });
    }

    private List<UpgradeContainerResult> upgradeSubContainerDir(File containerTopDir) throws IOException {
        File[] containerDirs;
        ArrayList<UpgradeContainerResult> resultList = new ArrayList<UpgradeContainerResult>();
        if (containerTopDir.isDirectory() && (containerDirs = containerTopDir.listFiles()) != null) {
            for (File containerDir : containerDirs) {
                ContainerData containerData = this.parseContainerData(containerDir);
                if (containerData == null || !((KeyValueContainerData)containerData).hasSchema("2")) continue;
                UpgradeContainerResult result = new UpgradeContainerResult(containerData);
                this.upgradeContainer(containerData, result);
                resultList.add(result);
            }
        }
        return resultList;
    }

    private ContainerData parseContainerData(File containerDir) {
        try {
            File containerFile = ContainerUtils.getContainerFile((File)containerDir);
            long containerID = ContainerUtils.getContainerID((File)containerDir);
            if (containerFile.exists()) {
                try {
                    ContainerData containerData = ContainerDataYaml.readContainerFile((File)containerFile);
                    if (containerID != containerData.getContainerID()) {
                        LOG.error("Invalid ContainerID in file {}. Skipping loading of this container.", (Object)containerFile);
                        return null;
                    }
                    if (containerData.getContainerType().equals((Object)ContainerProtos.ContainerType.KeyValueContainer) && containerData instanceof KeyValueContainerData) {
                        KeyValueContainerData kvContainerData = (KeyValueContainerData)containerData;
                        containerData.setVolume(this.hddsVolume);
                        KeyValueContainerUtil.parseKVContainerData((KeyValueContainerData)kvContainerData, (ConfigurationSource)this.config);
                        return kvContainerData;
                    }
                    LOG.error("Only KeyValueContainer support. ");
                    return null;
                }
                catch (IOException ex) {
                    LOG.error("Failed to parse ContainerFile for ContainerID: {}", (Object)containerID, (Object)ex);
                    return null;
                }
            }
            LOG.error("Missing .container file for ContainerID: {}", (Object)containerDir.getName());
            return null;
        }
        catch (Throwable e) {
            LOG.error("Failed to load container from {}", (Object)containerDir.getAbsolutePath(), (Object)e);
            return null;
        }
    }

    private void upgradeContainer(ContainerData containerData, UpgradeContainerResult result) throws IOException {
        DBStore targetDBStore = this.datanodeStoreSchemaThree.getStore();
        DatanodeStore dbStore = BlockUtils.getUncachedDatanodeStore((KeyValueContainerData)((KeyValueContainerData)containerData), (ConfigurationSource)this.config, (boolean)true);
        DBStore sourceDBStore = dbStore.getStore();
        long total = 0L;
        for (String tableName : COLUMN_FAMILIES_NAME) {
            total += this.transferTableData(targetDBStore, sourceDBStore, tableName, containerData);
        }
        this.rewriteAndBackupContainerDataFile(containerData, result);
        result.success(total);
    }

    private long transferTableData(DBStore targetDBStore, DBStore sourceDBStore, String tableName, ContainerData containerData) throws IOException {
        Table deleteTransactionTable = sourceDBStore.getTable(tableName);
        Table targetDeleteTransactionTable = targetDBStore.getTable(tableName);
        return this.transferTableData((Table<byte[], byte[]>)targetDeleteTransactionTable, (Table<byte[], byte[]>)deleteTransactionTable, containerData);
    }

    private long transferTableData(Table<byte[], byte[]> targetTable, Table<byte[], byte[]> sourceTable, ContainerData containerData) throws IOException {
        long count = 0L;
        try (TableIterator iter = sourceTable.iterator();){
            while (iter.hasNext()) {
                ++count;
                Table.KeyValue next = (Table.KeyValue)iter.next();
                String key = DatanodeSchemaThreeDBDefinition.getContainerKeyPrefix((long)containerData.getContainerID()) + StringUtils.bytes2String((byte[])((byte[])next.getKey()));
                targetTable.put((Object)FixedLengthStringCodec.string2Bytes((String)key), next.getValue());
            }
        }
        return count;
    }

    private void rewriteAndBackupContainerDataFile(ContainerData containerData, UpgradeContainerResult result) throws IOException {
        if (containerData instanceof KeyValueContainerData) {
            KeyValueContainerData keyValueContainerData = (KeyValueContainerData)containerData;
            KeyValueContainerData copyContainerData = new KeyValueContainerData(keyValueContainerData);
            copyContainerData.setSchemaVersion("3");
            copyContainerData.setState(keyValueContainerData.getState());
            copyContainerData.setVolume(keyValueContainerData.getVolume());
            File originContainerFile = KeyValueContainer.getContainerFile((String)keyValueContainerData.getMetadataPath(), (long)keyValueContainerData.getContainerID());
            File bakFile = new File(keyValueContainerData.getMetadataPath(), keyValueContainerData.getContainerID() + BACKUP_CONTAINER_DATA_FILE_SUFFIX);
            NativeIO.renameTo((File)originContainerFile, (File)bakFile);
            result.setBackupContainerFilePath(bakFile.getAbsolutePath());
            ContainerDataYaml.createContainerFile((ContainerProtos.ContainerType)ContainerProtos.ContainerType.KeyValueContainer, (ContainerData)copyContainerData, (File)originContainerFile);
            result.setNewContainerData((ContainerData)copyContainerData);
            result.setNewContainerFilePath(originContainerFile.getAbsolutePath());
        }
    }

    public static class UpgradeContainerResult {
        private final ContainerData originContainerData;
        private ContainerData newContainerData;
        private long totalRow = 0L;
        private final long startTimeMs = System.currentTimeMillis();
        private long endTimeMs = 0L;
        private Status status;
        private String backupContainerFilePath;
        private String newContainerFilePath;

        public UpgradeContainerResult(ContainerData originContainerData) {
            this.originContainerData = originContainerData;
            this.status = Status.FAIL;
        }

        public long getTotalRow() {
            return this.totalRow;
        }

        public Status getStatus() {
            return this.status;
        }

        public void setNewContainerData(ContainerData newContainerData) {
            this.newContainerData = newContainerData;
        }

        public long getCostMs() {
            return this.endTimeMs - this.startTimeMs;
        }

        public ContainerData getOriginContainerData() {
            return this.originContainerData;
        }

        public ContainerData getNewContainerData() {
            return this.newContainerData;
        }

        public void setBackupContainerFilePath(String backupContainerFilePath) {
            this.backupContainerFilePath = backupContainerFilePath;
        }

        public void setNewContainerFilePath(String newContainerFilePath) {
            this.newContainerFilePath = newContainerFilePath;
        }

        public void success(long rowCount) {
            this.totalRow = rowCount;
            this.endTimeMs = System.currentTimeMillis();
            this.status = Status.SUCCESS;
        }

        public String toString() {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.append("Result{");
            stringBuilder.append("containerID=");
            stringBuilder.append(this.originContainerData.getContainerID());
            stringBuilder.append(", originContainerSchemaVersion=");
            stringBuilder.append(((KeyValueContainerData)this.originContainerData).getSchemaVersion());
            if (this.newContainerData != null) {
                stringBuilder.append(", schemaV2ContainerFileBackupPath=");
                stringBuilder.append(this.backupContainerFilePath);
                stringBuilder.append(", newContainerSchemaVersion=");
                stringBuilder.append(((KeyValueContainerData)this.newContainerData).getSchemaVersion());
                stringBuilder.append(", schemaV3ContainerFilePath=");
                stringBuilder.append(this.newContainerFilePath);
            }
            stringBuilder.append(", totalRow=");
            stringBuilder.append(this.totalRow);
            stringBuilder.append(", costMs=");
            stringBuilder.append(this.getCostMs());
            stringBuilder.append(", status=");
            stringBuilder.append((Object)this.status);
            stringBuilder.append("}");
            return stringBuilder.toString();
        }

        static enum Status {
            SUCCESS,
            FAIL;

        }
    }
}

