/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.coordination;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.LongSupplier;
import java.util.stream.Collectors;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.coordination.ApplyCommitRequest;
import org.elasticsearch.cluster.coordination.ClusterStatePublisher;
import org.elasticsearch.cluster.coordination.CoordinationState;
import org.elasticsearch.cluster.coordination.FailedToCommitClusterStateException;
import org.elasticsearch.cluster.coordination.Join;
import org.elasticsearch.cluster.coordination.PublishRequest;
import org.elasticsearch.cluster.coordination.PublishResponse;
import org.elasticsearch.cluster.coordination.PublishWithJoinResponse;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportResponse;

public abstract class Publication {
    protected final Logger logger = LogManager.getLogger(this.getClass());
    private final List<PublicationTarget> publicationTargets;
    private final PublishRequest publishRequest;
    private final ClusterStatePublisher.AckListener ackListener;
    private final LongSupplier currentTimeSupplier;
    private final long startTime;
    private Optional<ApplyCommitRequest> applyCommitRequest;
    private boolean isCompleted;
    private boolean cancelled;

    public Publication(PublishRequest publishRequest, ClusterStatePublisher.AckListener ackListener, LongSupplier currentTimeSupplier) {
        this.publishRequest = publishRequest;
        this.ackListener = ackListener;
        this.currentTimeSupplier = currentTimeSupplier;
        this.startTime = currentTimeSupplier.getAsLong();
        this.applyCommitRequest = Optional.empty();
        this.publicationTargets = new ArrayList<PublicationTarget>(publishRequest.getAcceptedState().getNodes().getNodes().size());
        publishRequest.getAcceptedState().getNodes().mastersFirstStream().forEach(n -> this.publicationTargets.add(new PublicationTarget((DiscoveryNode)n)));
    }

    public void start(Set<DiscoveryNode> faultyNodes) {
        this.logger.trace("publishing {} to {}", (Object)this.publishRequest, (Object)this.publicationTargets);
        for (DiscoveryNode faultyNode : faultyNodes) {
            this.onFaultyNode(faultyNode);
        }
        this.onPossibleCommitFailure();
        this.publicationTargets.forEach(PublicationTarget::sendPublishRequest);
    }

    public void cancel(String reason) {
        if (this.isCompleted) {
            return;
        }
        assert (!this.cancelled);
        this.cancelled = true;
        if (!this.applyCommitRequest.isPresent()) {
            this.logger.debug("cancel: [{}] cancelled before committing (reason: {})", (Object)this, (Object)reason);
            ElasticsearchException e = new ElasticsearchException("publication cancelled before committing: " + reason, new Object[0]);
            this.publicationTargets.stream().filter(PublicationTarget::isActive).forEach(pt -> pt.setFailed(e));
        }
        this.onPossibleCompletion();
    }

    public void onFaultyNode(DiscoveryNode faultyNode) {
        this.publicationTargets.forEach(t -> t.onFaultyNode(faultyNode));
        this.onPossibleCompletion();
    }

    public List<DiscoveryNode> completedNodes() {
        return this.publicationTargets.stream().filter(PublicationTarget::isSuccessfullyCompleted).map(PublicationTarget::getDiscoveryNode).collect(Collectors.toList());
    }

    public boolean isCommitted() {
        return this.applyCommitRequest.isPresent();
    }

    private void onPossibleCompletion() {
        if (this.isCompleted) {
            return;
        }
        if (!this.cancelled) {
            for (PublicationTarget target : this.publicationTargets) {
                if (!target.isActive()) continue;
                return;
            }
        }
        if (!this.applyCommitRequest.isPresent()) {
            this.logger.debug("onPossibleCompletion: [{}] commit failed", (Object)this);
            assert (!this.isCompleted);
            this.isCompleted = true;
            this.onCompletion(false);
            return;
        }
        assert (!this.isCompleted);
        this.isCompleted = true;
        this.onCompletion(true);
        assert (this.applyCommitRequest.isPresent());
        this.logger.trace("onPossibleCompletion: [{}] was successful", (Object)this);
    }

    private boolean publicationCompletedIffAllTargetsInactiveOrCancelled() {
        if (!this.cancelled) {
            for (PublicationTarget target : this.publicationTargets) {
                if (!target.isActive()) continue;
                return !this.isCompleted;
            }
        }
        return this.isCompleted;
    }

    ClusterState publishedState() {
        return this.publishRequest.getAcceptedState();
    }

    private void onPossibleCommitFailure() {
        if (this.applyCommitRequest.isPresent()) {
            this.onPossibleCompletion();
            return;
        }
        CoordinationState.VoteCollection possiblySuccessfulNodes = new CoordinationState.VoteCollection();
        for (PublicationTarget publicationTarget : this.publicationTargets) {
            if (publicationTarget.mayCommitInFuture()) {
                possiblySuccessfulNodes.addVote(publicationTarget.discoveryNode);
                continue;
            }
            assert (publicationTarget.isFailed()) : publicationTarget;
        }
        if (!this.isPublishQuorum(possiblySuccessfulNodes)) {
            this.logger.debug("onPossibleCommitFailure: non-failed nodes {} do not form a quorum, so {} cannot succeed", (Object)possiblySuccessfulNodes, (Object)this);
            FailedToCommitClusterStateException e = new FailedToCommitClusterStateException("non-failed nodes do not form a quorum", new Object[0]);
            this.publicationTargets.stream().filter(PublicationTarget::isActive).forEach(pt -> pt.setFailed(e));
            this.onPossibleCompletion();
        }
    }

    protected abstract void onCompletion(boolean var1);

    protected abstract boolean isPublishQuorum(CoordinationState.VoteCollection var1);

    protected abstract Optional<ApplyCommitRequest> handlePublishResponse(DiscoveryNode var1, PublishResponse var2);

    protected abstract void onJoin(Join var1);

    protected abstract void onMissingJoin(DiscoveryNode var1);

    protected abstract void sendPublishRequest(DiscoveryNode var1, PublishRequest var2, ActionListener<PublishWithJoinResponse> var3);

    protected abstract void sendApplyCommit(DiscoveryNode var1, ApplyCommitRequest var2, ActionListener<TransportResponse.Empty> var3);

    public String toString() {
        return "Publication{term=" + this.publishRequest.getAcceptedState().term() + ", version=" + this.publishRequest.getAcceptedState().version() + '}';
    }

    void logIncompleteNodes(Level level) {
        String message = this.publicationTargets.stream().filter(PublicationTarget::isActive).map(publicationTarget -> publicationTarget.getDiscoveryNode() + " [" + (Object)((Object)publicationTarget.getState()) + "]").collect(Collectors.joining(", "));
        if (!message.isEmpty()) {
            TimeValue elapsedTime = TimeValue.timeValueMillis(this.currentTimeSupplier.getAsLong() - this.startTime);
            this.logger.log(level, "after [{}] publication of cluster state version [{}] is still waiting for {}", (Object)elapsedTime, (Object)this.publishRequest.getAcceptedState().version(), (Object)message);
        }
    }

    class PublicationTarget {
        private final DiscoveryNode discoveryNode;
        private boolean ackIsPending = true;
        private PublicationTargetState state = PublicationTargetState.NOT_STARTED;

        PublicationTarget(DiscoveryNode discoveryNode) {
            this.discoveryNode = discoveryNode;
        }

        PublicationTargetState getState() {
            return this.state;
        }

        public String toString() {
            return "PublicationTarget{discoveryNode=" + this.discoveryNode + ", state=" + (Object)((Object)this.state) + ", ackIsPending=" + this.ackIsPending + '}';
        }

        void sendPublishRequest() {
            if (this.isFailed()) {
                return;
            }
            assert (this.state == PublicationTargetState.NOT_STARTED) : (Object)((Object)this.state) + " -> " + (Object)((Object)PublicationTargetState.SENT_PUBLISH_REQUEST);
            this.state = PublicationTargetState.SENT_PUBLISH_REQUEST;
            Publication.this.sendPublishRequest(this.discoveryNode, Publication.this.publishRequest, new PublishResponseHandler());
            assert (Publication.this.publicationCompletedIffAllTargetsInactiveOrCancelled());
        }

        void handlePublishResponse(PublishResponse publishResponse) {
            assert (this.isWaitingForQuorum()) : this;
            Publication.this.logger.trace("handlePublishResponse: handling [{}] from [{}])", (Object)publishResponse, (Object)this.discoveryNode);
            if (Publication.this.applyCommitRequest.isPresent()) {
                this.sendApplyCommit();
            } else {
                try {
                    Publication.this.handlePublishResponse(this.discoveryNode, publishResponse).ifPresent(applyCommit -> {
                        assert (!Publication.this.applyCommitRequest.isPresent());
                        Publication.this.applyCommitRequest = Optional.of(applyCommit);
                        Publication.this.ackListener.onCommit(TimeValue.timeValueMillis(Publication.this.currentTimeSupplier.getAsLong() - Publication.this.startTime));
                        Publication.this.publicationTargets.stream().filter(PublicationTarget::isWaitingForQuorum).forEach(PublicationTarget::sendApplyCommit);
                    });
                }
                catch (Exception e) {
                    this.setFailed(e);
                    Publication.this.onPossibleCommitFailure();
                }
            }
        }

        void sendApplyCommit() {
            assert (this.state == PublicationTargetState.WAITING_FOR_QUORUM) : (Object)((Object)this.state) + " -> " + (Object)((Object)PublicationTargetState.SENT_APPLY_COMMIT);
            this.state = PublicationTargetState.SENT_APPLY_COMMIT;
            assert (Publication.this.applyCommitRequest.isPresent());
            Publication.this.sendApplyCommit(this.discoveryNode, (ApplyCommitRequest)Publication.this.applyCommitRequest.get(), new ApplyCommitResponseHandler());
            assert (Publication.this.publicationCompletedIffAllTargetsInactiveOrCancelled());
        }

        void setAppliedCommit() {
            assert (this.state == PublicationTargetState.SENT_APPLY_COMMIT) : (Object)((Object)this.state) + " -> " + (Object)((Object)PublicationTargetState.APPLIED_COMMIT);
            this.state = PublicationTargetState.APPLIED_COMMIT;
            this.ackOnce(null);
        }

        void setFailed(Exception e) {
            assert (this.state != PublicationTargetState.APPLIED_COMMIT) : (Object)((Object)this.state) + " -> " + (Object)((Object)PublicationTargetState.FAILED);
            this.state = PublicationTargetState.FAILED;
            this.ackOnce(e);
        }

        void onFaultyNode(DiscoveryNode faultyNode) {
            if (this.isActive() && this.discoveryNode.equals(faultyNode)) {
                Publication.this.logger.debug("onFaultyNode: [{}] is faulty, failing target in publication {}", (Object)faultyNode, (Object)Publication.this);
                this.setFailed(new ElasticsearchException("faulty node", new Object[0]));
                Publication.this.onPossibleCommitFailure();
            }
        }

        DiscoveryNode getDiscoveryNode() {
            return this.discoveryNode;
        }

        private void ackOnce(Exception e) {
            if (this.ackIsPending) {
                this.ackIsPending = false;
                Publication.this.ackListener.onNodeAck(this.discoveryNode, e);
            }
        }

        boolean isActive() {
            return this.state != PublicationTargetState.FAILED && this.state != PublicationTargetState.APPLIED_COMMIT;
        }

        boolean isSuccessfullyCompleted() {
            return this.state == PublicationTargetState.APPLIED_COMMIT;
        }

        boolean isWaitingForQuorum() {
            return this.state == PublicationTargetState.WAITING_FOR_QUORUM;
        }

        boolean mayCommitInFuture() {
            return this.state == PublicationTargetState.NOT_STARTED || this.state == PublicationTargetState.SENT_PUBLISH_REQUEST || this.state == PublicationTargetState.WAITING_FOR_QUORUM;
        }

        boolean isFailed() {
            return this.state == PublicationTargetState.FAILED;
        }

        private class ApplyCommitResponseHandler
        implements ActionListener<TransportResponse.Empty> {
            private ApplyCommitResponseHandler() {
            }

            @Override
            public void onResponse(TransportResponse.Empty ignored) {
                if (PublicationTarget.this.isFailed()) {
                    Publication.this.logger.debug("ApplyCommitResponseHandler.handleResponse: already failed, ignoring response from [{}]", (Object)PublicationTarget.this.discoveryNode);
                    return;
                }
                PublicationTarget.this.setAppliedCommit();
                Publication.this.onPossibleCompletion();
                assert (Publication.this.publicationCompletedIffAllTargetsInactiveOrCancelled());
            }

            @Override
            public void onFailure(Exception e) {
                assert (e instanceof TransportException);
                TransportException exp = (TransportException)e;
                Publication.this.logger.debug(() -> new ParameterizedMessage("ApplyCommitResponseHandler: [{}] failed", (Object)PublicationTarget.this.discoveryNode), (Throwable)exp);
                assert (((TransportException)e).getRootCause() instanceof Exception);
                PublicationTarget.this.setFailed((Exception)exp.getRootCause());
                Publication.this.onPossibleCompletion();
                assert (Publication.this.publicationCompletedIffAllTargetsInactiveOrCancelled());
            }
        }

        private class PublishResponseHandler
        implements ActionListener<PublishWithJoinResponse> {
            private PublishResponseHandler() {
            }

            @Override
            public void onResponse(PublishWithJoinResponse response) {
                if (PublicationTarget.this.isFailed()) {
                    Publication.this.logger.debug("PublishResponseHandler.handleResponse: already failed, ignoring response from [{}]", (Object)PublicationTarget.this.discoveryNode);
                    assert (Publication.this.publicationCompletedIffAllTargetsInactiveOrCancelled());
                    return;
                }
                if (response.getJoin().isPresent()) {
                    Join join = response.getJoin().get();
                    assert (PublicationTarget.this.discoveryNode.equals(join.getSourceNode()));
                    assert (join.getTerm() == response.getPublishResponse().getTerm()) : response;
                    Publication.this.logger.trace("handling join within publish response: {}", (Object)join);
                    Publication.this.onJoin(join);
                } else {
                    Publication.this.logger.trace("publish response from {} contained no join", (Object)PublicationTarget.this.discoveryNode);
                    Publication.this.onMissingJoin(PublicationTarget.this.discoveryNode);
                }
                assert (PublicationTarget.this.state == PublicationTargetState.SENT_PUBLISH_REQUEST) : (Object)((Object)PublicationTarget.access$800(PublicationTarget.this)) + " -> " + (Object)((Object)PublicationTargetState.WAITING_FOR_QUORUM);
                PublicationTarget.this.state = PublicationTargetState.WAITING_FOR_QUORUM;
                PublicationTarget.this.handlePublishResponse(response.getPublishResponse());
                assert (Publication.this.publicationCompletedIffAllTargetsInactiveOrCancelled());
            }

            @Override
            public void onFailure(Exception e) {
                assert (e instanceof TransportException);
                TransportException exp = (TransportException)e;
                Publication.this.logger.debug(() -> new ParameterizedMessage("PublishResponseHandler: [{}] failed", (Object)PublicationTarget.this.discoveryNode), (Throwable)exp);
                assert (((TransportException)e).getRootCause() instanceof Exception);
                PublicationTarget.this.setFailed((Exception)exp.getRootCause());
                Publication.this.onPossibleCommitFailure();
                assert (Publication.this.publicationCompletedIffAllTargetsInactiveOrCancelled());
            }
        }
    }

    static enum PublicationTargetState {
        NOT_STARTED,
        FAILED,
        SENT_PUBLISH_REQUEST,
        WAITING_FOR_QUORUM,
        SENT_APPLY_COMMIT,
        APPLIED_COMMIT;

    }
}

