/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.bootstrap;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.AllPermission;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.util.Constants;
import org.elasticsearch.bootstrap.BootstrapCheck;
import org.elasticsearch.bootstrap.BootstrapContext;
import org.elasticsearch.bootstrap.BootstrapInfo;
import org.elasticsearch.bootstrap.BootstrapSettings;
import org.elasticsearch.bootstrap.JNACLibrary;
import org.elasticsearch.bootstrap.JNANatives;
import org.elasticsearch.bootstrap.JavaVersion;
import org.elasticsearch.bootstrap.Natives;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.common.io.PathUtils;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.transport.BoundTransportAddress;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.discovery.DiscoveryModule;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.monitor.process.ProcessProbe;
import org.elasticsearch.node.Node;
import org.elasticsearch.node.NodeValidationException;

final class BootstrapChecks {
    static final String ES_ENFORCE_BOOTSTRAP_CHECKS = "es.enforce.bootstrap.checks";

    private BootstrapChecks() {
    }

    static void check(BootstrapContext context, BoundTransportAddress boundTransportAddress, List<BootstrapCheck> additionalChecks) throws NodeValidationException {
        List<BootstrapCheck> builtInChecks = BootstrapChecks.checks();
        ArrayList<BootstrapCheck> combinedChecks = new ArrayList<BootstrapCheck>(builtInChecks);
        combinedChecks.addAll(additionalChecks);
        BootstrapChecks.check(context, BootstrapChecks.enforceLimits(boundTransportAddress, DiscoveryModule.DISCOVERY_TYPE_SETTING.get(context.settings)), Collections.unmodifiableList(combinedChecks), Node.NODE_NAME_SETTING.get(context.settings));
    }

    static void check(BootstrapContext context, boolean enforceLimits, List<BootstrapCheck> checks, String nodeName) throws NodeValidationException {
        BootstrapChecks.check(context, enforceLimits, checks, Loggers.getLogger(BootstrapChecks.class, nodeName));
    }

    static void check(BootstrapContext context, boolean enforceLimits, List<BootstrapCheck> checks, Logger logger) throws NodeValidationException {
        boolean enforceBootstrapChecks;
        ArrayList<String> errors = new ArrayList<String>();
        ArrayList<String> ignoredErrors = new ArrayList<String>();
        String esEnforceBootstrapChecks = System.getProperty(ES_ENFORCE_BOOTSTRAP_CHECKS);
        if (esEnforceBootstrapChecks == null) {
            enforceBootstrapChecks = false;
        } else if (Boolean.TRUE.toString().equals(esEnforceBootstrapChecks)) {
            enforceBootstrapChecks = true;
        } else {
            String message = String.format(Locale.ROOT, "[%s] must be [true] but was [%s]", ES_ENFORCE_BOOTSTRAP_CHECKS, esEnforceBootstrapChecks);
            throw new IllegalArgumentException(message);
        }
        if (enforceLimits) {
            logger.info("bound or publishing to a non-loopback address, enforcing bootstrap checks");
        } else if (enforceBootstrapChecks) {
            logger.info("explicitly enforcing bootstrap checks");
        }
        for (BootstrapCheck check : checks) {
            BootstrapCheck.BootstrapCheckResult result = check.check(context);
            if (!result.isFailure()) continue;
            if (!(enforceLimits || enforceBootstrapChecks || check.alwaysEnforce())) {
                ignoredErrors.add(result.getMessage());
                continue;
            }
            errors.add(result.getMessage());
        }
        if (!ignoredErrors.isEmpty()) {
            ignoredErrors.forEach(error -> BootstrapChecks.log(logger, error));
        }
        if (!errors.isEmpty()) {
            ArrayList<String> messages = new ArrayList<String>(1 + errors.size());
            messages.add("[" + errors.size() + "] bootstrap checks failed");
            for (int i = 0; i < errors.size(); ++i) {
                messages.add("[" + (i + 1) + "]: " + (String)errors.get(i));
            }
            NodeValidationException ne = new NodeValidationException(String.join((CharSequence)"\n", messages));
            errors.stream().map(IllegalStateException::new).forEach(ne::addSuppressed);
            throw ne;
        }
    }

    static void log(Logger logger, String error) {
        logger.warn(error);
    }

    static boolean enforceLimits(BoundTransportAddress boundTransportAddress, String discoveryType) {
        Predicate<TransportAddress> isLoopbackAddress = t -> t.address().getAddress().isLoopbackAddress();
        boolean bound = !Arrays.stream(boundTransportAddress.boundAddresses()).allMatch(isLoopbackAddress) || !isLoopbackAddress.test(boundTransportAddress.publishAddress());
        return bound && !"single-node".equals(discoveryType);
    }

    static List<BootstrapCheck> checks() {
        ArrayList<BootstrapCheck> checks = new ArrayList<BootstrapCheck>();
        checks.add(new HeapSizeCheck());
        FileDescriptorCheck fileDescriptorCheck = Constants.MAC_OS_X ? new OsXFileDescriptorCheck() : new FileDescriptorCheck();
        checks.add(fileDescriptorCheck);
        checks.add(new MlockallCheck());
        if (Constants.LINUX) {
            checks.add(new MaxNumberOfThreadsCheck());
        }
        if (Constants.LINUX || Constants.MAC_OS_X) {
            checks.add(new MaxSizeVirtualMemoryCheck());
        }
        if (Constants.LINUX || Constants.MAC_OS_X) {
            checks.add(new MaxFileSizeCheck());
        }
        if (Constants.LINUX) {
            checks.add(new MaxMapCountCheck());
        }
        checks.add(new ClientJvmCheck());
        checks.add(new UseSerialGCCheck());
        checks.add(new SystemCallFilterCheck());
        checks.add(new OnErrorCheck());
        checks.add(new OnOutOfMemoryErrorCheck());
        checks.add(new EarlyAccessCheck());
        checks.add(new G1GCCheck());
        checks.add(new AllPermissionCheck());
        return Collections.unmodifiableList(checks);
    }

    static class AllPermissionCheck
    implements BootstrapCheck {
        AllPermissionCheck() {
        }

        @Override
        public final BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.isAllPermissionGranted()) {
                return BootstrapCheck.BootstrapCheckResult.failure("granting the all permission effectively disables security");
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        boolean isAllPermissionGranted() {
            SecurityManager sm = System.getSecurityManager();
            assert (sm != null);
            try {
                sm.checkPermission(new AllPermission());
            }
            catch (SecurityException e) {
                return false;
            }
            return true;
        }
    }

    static class G1GCCheck
    implements BootstrapCheck {
        G1GCCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if ("Oracle Corporation".equals(this.jvmVendor()) && this.isJava8() && this.isG1GCEnabled()) {
                String jvmVersion = this.jvmVersion();
                Pattern pattern = Pattern.compile("(\\d+)\\.(\\d+)-b\\d+");
                Matcher matcher = pattern.matcher(jvmVersion);
                boolean matches = matcher.matches();
                assert (matches) : jvmVersion;
                int major = Integer.parseInt(matcher.group(1));
                int update = Integer.parseInt(matcher.group(2));
                if (major == 25 && update < 40) {
                    String message = String.format(Locale.ROOT, "JVM version [%s] can cause data corruption when used with G1GC; upgrade to at least Java 8u40", jvmVersion);
                    return BootstrapCheck.BootstrapCheckResult.failure(message);
                }
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        String jvmVendor() {
            return Constants.JVM_VENDOR;
        }

        boolean isG1GCEnabled() {
            assert ("Oracle Corporation".equals(this.jvmVendor()));
            return JvmInfo.jvmInfo().useG1GC().equals("true");
        }

        String jvmVersion() {
            assert ("Oracle Corporation".equals(this.jvmVendor()));
            return Constants.JVM_VERSION;
        }

        boolean isJava8() {
            assert ("Oracle Corporation".equals(this.jvmVendor()));
            return JavaVersion.current().equals(JavaVersion.parse("1.8"));
        }
    }

    static class EarlyAccessCheck
    implements BootstrapCheck {
        EarlyAccessCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            String javaVersion = this.javaVersion();
            if ("Oracle Corporation".equals(this.jvmVendor()) && javaVersion.endsWith("-ea")) {
                String message = String.format(Locale.ROOT, "Java version [%s] is an early-access build, only use release builds", javaVersion);
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        String jvmVendor() {
            return Constants.JVM_VENDOR;
        }

        String javaVersion() {
            return Constants.JAVA_VERSION;
        }
    }

    static class OnOutOfMemoryErrorCheck
    extends MightForkCheck {
        OnOutOfMemoryErrorCheck() {
        }

        @Override
        boolean mightFork() {
            String onOutOfMemoryError = this.onOutOfMemoryError();
            return onOutOfMemoryError != null && !onOutOfMemoryError.equals("");
        }

        String onOutOfMemoryError() {
            return JvmInfo.jvmInfo().onOutOfMemoryError();
        }

        @Override
        String message(BootstrapContext context) {
            return String.format(Locale.ROOT, "OnOutOfMemoryError [%s] requires forking but is prevented by system call filters ([%s=true]); upgrade to at least Java 8u92 and use ExitOnOutOfMemoryError", this.onOutOfMemoryError(), BootstrapSettings.SYSTEM_CALL_FILTER_SETTING.getKey());
        }
    }

    static class OnErrorCheck
    extends MightForkCheck {
        OnErrorCheck() {
        }

        @Override
        boolean mightFork() {
            String onError = this.onError();
            return onError != null && !onError.equals("");
        }

        String onError() {
            return JvmInfo.jvmInfo().onError();
        }

        @Override
        String message(BootstrapContext context) {
            return String.format(Locale.ROOT, "OnError [%s] requires forking but is prevented by system call filters ([%s=true]); upgrade to at least Java 8u92 and use ExitOnOutOfMemoryError", this.onError(), BootstrapSettings.SYSTEM_CALL_FILTER_SETTING.getKey());
        }
    }

    static abstract class MightForkCheck
    implements BootstrapCheck {
        MightForkCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.isSystemCallFilterInstalled() && this.mightFork()) {
                return BootstrapCheck.BootstrapCheckResult.failure(this.message(context));
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        abstract String message(BootstrapContext var1);

        boolean isSystemCallFilterInstalled() {
            return Natives.isSystemCallFilterInstalled();
        }

        abstract boolean mightFork();

        @Override
        public final boolean alwaysEnforce() {
            return true;
        }
    }

    static class SystemCallFilterCheck
    implements BootstrapCheck {
        SystemCallFilterCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (BootstrapSettings.SYSTEM_CALL_FILTER_SETTING.get(context.settings).booleanValue() && !this.isSystemCallFilterInstalled()) {
                String message = "system call filters failed to install; check the logs and fix your configuration or disable system call filters at your own risk";
                return BootstrapCheck.BootstrapCheckResult.failure("system call filters failed to install; check the logs and fix your configuration or disable system call filters at your own risk");
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        boolean isSystemCallFilterInstalled() {
            return Natives.isSystemCallFilterInstalled();
        }
    }

    static class UseSerialGCCheck
    implements BootstrapCheck {
        UseSerialGCCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.getUseSerialGC().equals("true")) {
                String message = String.format(Locale.ROOT, "JVM is using the serial collector but should not be for the best performance; either it's the default for the VM [%s] or -XX:+UseSerialGC was explicitly specified", JvmInfo.jvmInfo().getVmName());
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        String getUseSerialGC() {
            return JvmInfo.jvmInfo().useSerialGC();
        }
    }

    static class ClientJvmCheck
    implements BootstrapCheck {
        ClientJvmCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.getVmName().toLowerCase(Locale.ROOT).contains("client")) {
                String message = String.format(Locale.ROOT, "JVM is using the client VM [%s] but should be using a server VM for the best performance", this.getVmName());
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        String getVmName() {
            return JvmInfo.jvmInfo().getVmName();
        }
    }

    static class MaxMapCountCheck
    implements BootstrapCheck {
        private static final long LIMIT = 262144L;

        MaxMapCountCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.getMaxMapCount() != -1L && this.getMaxMapCount() < 262144L) {
                String message = String.format(Locale.ROOT, "max virtual memory areas vm.max_map_count [%d] is too low, increase to at least [%d]", this.getMaxMapCount(), 262144L);
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        long getMaxMapCount() {
            return this.getMaxMapCount(Loggers.getLogger(BootstrapChecks.class));
        }

        /*
         * Enabled aggressive exception aggregation
         */
        long getMaxMapCount(Logger logger) {
            block19: {
                Path path = this.getProcSysVmMaxMapCountPath();
                try {
                    Throwable throwable = null;
                    try (BufferedReader bufferedReader = this.getBufferedReader(path);){
                        String rawProcSysVmMaxMapCount = this.readProcSysVmMaxMapCount(bufferedReader);
                        if (rawProcSysVmMaxMapCount == null) break block19;
                        try {
                            long l = this.parseProcSysVmMaxMapCount(rawProcSysVmMaxMapCount);
                            return l;
                        }
                        catch (NumberFormatException e) {
                            try {
                                logger.warn(() -> new ParameterizedMessage("unable to parse vm.max_map_count [{}]", (Object)rawProcSysVmMaxMapCount), (Throwable)e);
                            }
                            catch (Throwable throwable2) {
                                throwable = throwable2;
                                throw throwable2;
                            }
                            catch (Throwable throwable3) {
                                throw throwable3;
                            }
                        }
                    }
                }
                catch (IOException e) {
                    logger.warn(() -> new ParameterizedMessage("I/O exception while trying to read [{}]", (Object)path), (Throwable)e);
                }
            }
            return -1L;
        }

        @SuppressForbidden(reason="access /proc/sys/vm/max_map_count")
        private Path getProcSysVmMaxMapCountPath() {
            return PathUtils.get("/proc/sys/vm/max_map_count", new String[0]);
        }

        BufferedReader getBufferedReader(Path path) throws IOException {
            return Files.newBufferedReader(path);
        }

        String readProcSysVmMaxMapCount(BufferedReader bufferedReader) throws IOException {
            return bufferedReader.readLine();
        }

        long parseProcSysVmMaxMapCount(String procSysVmMaxMapCount) throws NumberFormatException {
            return Long.parseLong(procSysVmMaxMapCount);
        }
    }

    static class MaxFileSizeCheck
    implements BootstrapCheck {
        MaxFileSizeCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            long maxFileSize = this.getMaxFileSize();
            if (maxFileSize != Long.MIN_VALUE && maxFileSize != this.getRlimInfinity()) {
                String message = String.format(Locale.ROOT, "max file size [%d] for user [%s] is too low, increase to [unlimited]", this.getMaxFileSize(), BootstrapInfo.getSystemProperties().get("user.name"));
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        long getRlimInfinity() {
            return JNACLibrary.RLIM_INFINITY;
        }

        long getMaxFileSize() {
            return JNANatives.MAX_FILE_SIZE;
        }
    }

    static class MaxSizeVirtualMemoryCheck
    implements BootstrapCheck {
        MaxSizeVirtualMemoryCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.getMaxSizeVirtualMemory() != Long.MIN_VALUE && this.getMaxSizeVirtualMemory() != this.getRlimInfinity()) {
                String message = String.format(Locale.ROOT, "max size virtual memory [%d] for user [%s] is too low, increase to [unlimited]", this.getMaxSizeVirtualMemory(), BootstrapInfo.getSystemProperties().get("user.name"));
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        long getRlimInfinity() {
            return JNACLibrary.RLIM_INFINITY;
        }

        long getMaxSizeVirtualMemory() {
            return JNANatives.MAX_SIZE_VIRTUAL_MEMORY;
        }
    }

    static class MaxNumberOfThreadsCheck
    implements BootstrapCheck {
        private static final long MAX_NUMBER_OF_THREADS_THRESHOLD = 4096L;

        MaxNumberOfThreadsCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.getMaxNumberOfThreads() != -1L && this.getMaxNumberOfThreads() < 4096L) {
                String message = String.format(Locale.ROOT, "max number of threads [%d] for user [%s] is too low, increase to at least [%d]", this.getMaxNumberOfThreads(), BootstrapInfo.getSystemProperties().get("user.name"), 4096L);
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        long getMaxNumberOfThreads() {
            return JNANatives.MAX_NUMBER_OF_THREADS;
        }
    }

    static class MlockallCheck
    implements BootstrapCheck {
        MlockallCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (BootstrapSettings.MEMORY_LOCK_SETTING.get(context.settings).booleanValue() && !this.isMemoryLocked()) {
                return BootstrapCheck.BootstrapCheckResult.failure("memory locking requested for elasticsearch process but memory is not locked");
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        boolean isMemoryLocked() {
            return Natives.isMemoryLocked();
        }
    }

    static class FileDescriptorCheck
    implements BootstrapCheck {
        private final int limit;

        FileDescriptorCheck() {
            this(65536);
        }

        protected FileDescriptorCheck(int limit) {
            if (limit <= 0) {
                throw new IllegalArgumentException("limit must be positive but was [" + limit + "]");
            }
            this.limit = limit;
        }

        @Override
        public final BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            long maxFileDescriptorCount = this.getMaxFileDescriptorCount();
            if (maxFileDescriptorCount != -1L && maxFileDescriptorCount < (long)this.limit) {
                String message = String.format(Locale.ROOT, "max file descriptors [%d] for elasticsearch process is too low, increase to at least [%d]", this.getMaxFileDescriptorCount(), this.limit);
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        long getMaxFileDescriptorCount() {
            return ProcessProbe.getInstance().getMaxFileDescriptorCount();
        }
    }

    static class OsXFileDescriptorCheck
    extends FileDescriptorCheck {
        OsXFileDescriptorCheck() {
            super(10240);
        }
    }

    static class HeapSizeCheck
    implements BootstrapCheck {
        HeapSizeCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            long initialHeapSize = this.getInitialHeapSize();
            long maxHeapSize = this.getMaxHeapSize();
            if (initialHeapSize != 0L && maxHeapSize != 0L && initialHeapSize != maxHeapSize) {
                String message = String.format(Locale.ROOT, "initial heap size [%d] not equal to maximum heap size [%d]; this can cause resize pauses and prevents mlockall from locking the entire heap", this.getInitialHeapSize(), this.getMaxHeapSize());
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        long getInitialHeapSize() {
            return JvmInfo.jvmInfo().getConfiguredInitialHeapSize();
        }

        long getMaxHeapSize() {
            return JvmInfo.jvmInfo().getConfiguredMaxHeapSize();
        }
    }
}

