% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iff.8svx.r
\name{WaveToIFF}
\alias{WaveToIFF}
\title{Convert WaveMC objects into an Interchange File Format object}
\usage{
WaveToIFF(x, loop.start = NA, octaves = 1, compress = c("sCmpNone",
  "sCmpFibDelta"), ...)
}
\arguments{
\item{x}{A \code{\link[tuneR]{WaveMC}} object that needs to be converted into an \code{\link{IFFChunk}} object. \code{x}
can also be any other class object that can be coerced into a \code{\link[tuneR]{WaveMC}} object. \code{\link[tuneR]{Wave}}
and \code{\link[ProTrackR]{PTSample}} objects are therefore also allowed.}

\item{loop.start}{If the sample should be looped from a specific position to the
end of the sample, this argument specifies the starting position in samples (with
a base of 0) for looping. \code{loop.start} therefore should be a whole non-negative
number. When set to \code{NA} or negative values, the sample will not be looped.}

\item{octaves}{A whole positive \code{numeric} value indicating the number of octaves
that should be stored in the resulting IFF chunk. The original wave will be resampled
for each value larger than 1. Each subsequent octave will contain precisely twice
as many samples as the previous octave.}

\item{compress}{A \code{character} string indicating whether compression should be applied to the waveform. "\code{sCmpNone}"
(default) applies no compression, "\code{sCmpFibDelta}" applies the lossy \code{\link{deltaFibonacciCompress}}ion.}

\item{...}{Currently ignored.}
}
\value{
Returns an \code{\link{IFFChunk-class}} object with a FORM container that
contains an 8SVX waveform based on \code{x}.
}
\description{
Convert \code{\link[tuneR]{WaveMC}} objects (or objects that can be coerced to
\code{WaveMC} objects) into an \code{\link{IFFChunk-class}} object which
can be stored as a valid Iterchange File Format (\code{\link{write.iff}}).
}
\details{
\code{\link[tuneR]{WaveMC}} objects can be read from contempory file containers
with \code{\link[tuneR]{readWave}} or \code{\link[tuneR]{readMP3}}. With this
function such objects can be converted into an \code{\link{IFFChunk-class}} object
which can be stored conform the Interchange File Format (\code{\link{write.iff}}).

When \code{x} is not a pcm formatted 8-bit sample, \code{x} will first be
normalized and scaled to a pcm-formatted 8-bit sample using
\code{\link[tuneR]{normalize}}. If you don't like the result you need to convert
the sample to 8-bit pcm yourself before calling this function.
}
\examples{
\dontrun{
## First get an audio sample from the ProTrackR package
snare.samp <- ProTrackR::PTSample(ProTrackR::mod.intro, 2)

## The sample can easily be converted into an IFFChunk:
snare.iff <- WaveToIFF(snare.samp)

## You could also first convert the sample into a Wave object:
snare.wav <- as(snare.samp, "Wave")

## And then convert into an IFFChunk. The result is the same:
snare.iff <- WaveToIFF(snare.wav)

## You could also use a sine wave as input (although you will get some warnings).
## This will work because the vector of numeric data can be coerced to
## a WaveMC object
sine.iff <- WaveToIFF(sin((0:2000)/20))
}
}
\references{
\url{https://en.wikipedia.org/wiki/8SVX}
}
\seealso{
Other iff.operations: \code{\link{IFFChunk-class}},
  \code{\link{as.raster}}, \code{\link{as.raw}},
  \code{\link{getIFFChunk}},
  \code{\link{interpretIFFChunk}},
  \code{\link{rasterToIFF}}, \code{\link{read.iff}},
  \code{\link{write.iff}}
}
\author{
Pepijn de Vries
}
