% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cmps.R
\name{extract_feature_cmps}
\alias{extract_feature_cmps}
\title{Computes the CMPS score of a comparison between two bullet profiles/signatures}
\usage{
extract_feature_cmps(
  x,
  y,
  seg_length = 50,
  Tx = 25,
  npeaks.set = c(5, 3, 1),
  include = NULL,
  outlength = NULL
)
}
\arguments{
\item{x}{a numeric vector, vector of the reference bullet signature/profile that will be divided into basis segments}

\item{y}{a numeric vector, vector of the comparison bullet signature/profile}

\item{seg_length}{a positive integer, the length of a basis segment}

\item{Tx}{a positive integer, the tolerance zone is \verb{+/- Tx}}

\item{npeaks.set}{a numeric vector, specify the number of peaks to be found at each segment scale level
\itemize{
\item If \code{length(npeaks.set) == 1}, the algorithm uses multi-peak inspection only at the basis scale level;
\item If \code{length(npeaks.set) > 1}, the algorithm uses multi-peak inspection at
different segment scale levels.
\item By default, \code{npeaks.set = c(5,3,1)}. Including more segment scale levels will reduce the number of false positive results
}}

\item{include}{\code{NULL} or a vector of character strings indicating what additional information should be included in
the output of \code{extract_feature_cmps}. All possible options are: "nseg", "congruent.pos", "congruent.seg",
"congruent.seg.idx", "pos.df", "ccp.list","segments", and "parameters". If one wants to include them all, one can use
\code{include = "full_result"}. By default, \code{include = NULL} and only the CMPS score is returned}

\item{outlength}{\code{NULL} or a numeric vector, specify the segment length of each level of the basis segment when the
multi-segment lengths strategy is being used. If \code{outlength = NULL}, then the length of a basis segment will be doubled
at each segment level}
}
\value{
a numeric value or a list
\itemize{
\item if \code{include = NULL}, returns the CMPS score (a numeric value) only
\item if \verb{include = } one or a vector of strings listed above:
\itemize{
\item \code{nseg}: number of basis segments
\item \code{congruent.seg}: a vector of boolean values. \code{TRUE} means this basis segment is a congruent matching profile segment (CMPS)
\item \code{congruent.seg.idx}: the indices of all CMPS
\item \code{pos.df}: a dataframe that includes positions of correlation peaks and the CMPS score of these positions
\item \code{ccp.list}: a list of consistent correlation peaks of each basis segment.
\item \code{segments}: a list of all basis segments
\item \code{parameters}: a list that stores all parameters used in the function call
}
}
}
\description{
Compute the Congruent Matching Profile Segments (CMPS) score based on two bullet profiles/signatures.
The reference profile will be divided into consecutive, non-overlapping, basis segments of the same length.
Then the number of segments that are congruent matching will be found as the CMPS score.
By default, \code{extract_feature_cmps} implements the algorithm with multi-peak inspection at three
different segment scale levels. By setting \code{npeaks.set} as a single-length vector, users can switch to the algorithm
with multi-peak inspection at the basis scale level only.
}
\examples{
library(tidyverse)
library(cmpsR)

data("bullets")
land2_3 <- bullets$sigs[bullets$bulletland == "2-3"][[1]]
land1_2 <- bullets$sigs[bullets$bulletland == "1-2"][[1]]

# compute cmps

# algorithm with multi-peak insepction at three different segment scale levels
cmps_with_multi_scale <- extract_feature_cmps(land2_3$sig, land1_2$sig, include = "full_result" )

# algorithm with multi-peak inspection at the basis scale level only
cmps_without_multi_scale <- extract_feature_cmps(land2_3$sig, land1_2$sig, 
                                                 npeaks.set = 5, include = "full_result" )

# Another example
library(tidyverse)

data("bullets")

lands <- unique(bullets$bulletland)

comparisons <- data.frame(expand.grid(land1 = lands[1:6], land2 = lands[7:12]),
                          stringsAsFactors = FALSE)

comparisons <- comparisons \%>\%
  left_join(bullets \%>\% select(bulletland, sig1=sigs),
            by = c("land1" = "bulletland")) \%>\%
  left_join(bullets \%>\% select(bulletland, sig2=sigs),
            by = c("land2" = "bulletland"))

comparisons <- comparisons \%>\% mutate(
  cmps = purrr::map2(sig1, sig2, .f = function(x, y) {
    extract_feature_cmps(x$sig, y$sig, include = "full_result")
  })
)

comparisons <- comparisons \%>\%
  mutate(
    cmps_score = sapply(comparisons$cmps, function(x) x$CMPS.score),
    cmps_nseg = sapply(comparisons$cmps, function(x) x$nseg)
  )
  
cp1 <- comparisons \%>\% select(land1, land2, cmps_score, cmps_nseg)
cp1  

}
\references{
Chen, Zhe, Wei Chu, Johannes A Soons, Robert M Thompson, John Song,
and Xuezeng Zhao. 2019. “Fired Bullet Signature Correlation Using the
Congruent Matching Profile Segments (CMPS) Method.” Forensic Science
International, December, #109964. https://doi.org/10.1016/j.forsciint.2019.109964.
}
