% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CDVineCopulaConditional.r
\name{CDVineCondFit}
\alias{CDVineCondFit}
\title{Selection of a C- or D- vine copula model for conditional sampling}
\usage{
CDVineCondFit(data, Nx, treecrit = "AIC", type = "CVine-DVine",
  selectioncrit = "AIC", familyset = NA, indeptest = FALSE,
  level = 0.05, se = FALSE, rotations = TRUE, method = "mle",
  Matrix = FALSE)
}
\arguments{
\item{data}{An \code{N x d} data matrix (with uniform margins).
The data of the conditioning variable(s) have to occupy the last column(s) of this matrix.}

\item{Nx}{Number of conditioning variables.}

\item{treecrit}{Character indicating the criteria used to select the vine. All possible vines are fitted trough 
the function \code{\link[VineCopula]{RVineCopSelect}} of the package \code{VineCopula}. Then the vines are ranked with respect 
the Akaike information criterion (\code{treecrit = "AIC"}, default) or Bayesian information criterion (\code{treecrit = "BIC"}). 
This need the estimation and model selection for all the pairs of all the possible vines, therefore could 
require long time in case of large datasets, i.e. large \code{N x d}.}

\item{type}{Type of vine to be fitted: \cr
C-Vine: "CVine" or 1; \cr
D-Vine: "DVine" or 2; \cr
Both C and D-Vine: "CVine-DVine" or "1-2" (default).}

\item{selectioncrit}{Character indicating the criterion for pair-copula selection.
Possible choices are \code{"AIC"} (default) and \code{"BIC"}.}

\item{familyset}{"Integer vector of pair-copula families to select from. The vector has to include at least one
pair-copula family that allows for positive and one that allows for negative
dependence. Not listed copula families might be included to better handle
limit cases.  If \code{familyset = NA} (default), selection among all
possible families is performed. If a vector of negative numbers is provided,
selection among all but \code{abs(familyset)} is performed. Coding of bivariate copula families: \cr
\code{0} = independence copula \cr
\code{1} = Gaussian copula \cr
\code{2} = Student t copula (t-copula) \cr
\code{3} = Clayton copula \cr
\code{4} = Gumbel copula \cr
\code{5} = Frank copula \cr
\code{6} = Joe copula \cr
\code{7} = BB1 copula \cr
\code{8} = BB6 copula \cr
\code{9} = BB7 copula \cr
\code{10} = BB8 copula \cr
\code{13} = rotated Clayton copula (180 degrees; ``survival Clayton'') \cr
\code{14} = rotated Gumbel copula (180 degrees; ``survival Gumbel'') \cr
\code{16} = rotated Joe copula (180 degrees; ``survival Joe'') \cr
\code{17} = rotated BB1 copula (180 degrees; ``survival BB1'')\cr
\code{18} = rotated BB6 copula (180 degrees; ``survival BB6'')\cr
\code{19} = rotated BB7 copula (180 degrees; ``survival BB7'')\cr
\code{20} = rotated BB8 copula (180 degrees; ``survival BB8'')\cr
\code{23} = rotated Clayton copula (90 degrees) \cr
\code{24} = rotated Gumbel copula (90 degrees) \cr
\code{26} = rotated Joe copula (90 degrees) \cr
\code{27} = rotated BB1 copula (90 degrees) \cr
\code{28} = rotated BB6 copula (90 degrees) \cr
\code{29} = rotated BB7 copula (90 degrees) \cr
\code{30} = rotated BB8 copula (90 degrees) \cr
\code{33} = rotated Clayton copula (270 degrees) \cr
\code{34} = rotated Gumbel copula (270 degrees) \cr
\code{36} = rotated Joe copula (270 degrees) \cr
\code{37} = rotated BB1 copula (270 degrees) \cr
\code{38} = rotated BB6 copula (270 degrees) \cr
\code{39} = rotated BB7 copula (270 degrees) \cr
\code{40} = rotated BB8 copula (270 degrees) \cr
\code{104} = Tawn type 1 copula \cr
\code{114} = rotated Tawn type 1 copula (180 degrees) \cr
\code{124} = rotated Tawn type 1 copula (90 degrees) \cr
\code{134} = rotated Tawn type 1 copula (270 degrees) \cr
\code{204} = Tawn type 2 copula \cr
\code{214} = rotated Tawn type 2 copula (180 degrees) \cr
\code{224} = rotated Tawn type 2 copula (90 degrees) \cr
\code{234} = rotated Tawn type 2 copula (270 degrees)}

\item{indeptest}{Logical; whether a hypothesis test for the independence of
\code{u1} and \code{u2} is performed before bivariate copula selection
(default: \code{indeptest = FALSE}; see \code{BiCopIndTest}).  The
independence copula is chosen for a (conditional) pair if the null
hypothesis of independence cannot be rejected.}

\item{level}{numeric; significance level of the independence test (default:
\code{level = 0.05}).}

\item{se}{Logical; whether standard errors are estimated (default: \code{se
= FALSE}).}

\item{rotations}{logical; if \code{TRUE}, all rotations of the families in
\code{familyset} are included.}

\item{method}{indicates the estimation method: either maximum
likelihood estimation (\code{method = "mle"}; default) or inversion of
Kendall's tau (\code{method = "itau"}). For \code{method = "itau"} only
one parameter families and the Student t copula can be used (\code{family =
1,2,3,4,5,6,13,14,16,23,24,26,33,34} or \code{36}). For the t-copula,
\code{par2} is found by a crude profile likelihood optimization over the
interval (2, 10]." (VineCopula Documentation, version 2.1.1, pp. 73-75)}

\item{Matrix}{\code{d x d} matrix that defines the vine structure. 
If \code{Matrix} is not given, the routine finds the best vine structure according to \code{selectioncrit}. 
If \code{Matrix} is given, the fit is performed only if the structure respects the necessary conditions 
for the conditional sampling (if the conditions are not respected, an error message is returned).}
}
\value{
An \code{\link[VineCopula]{RVineMatrix}} object describing the selected copula model 
(for further details about \code{\link[VineCopula]{RVineMatrix}} objects see the documentation file of the \code{VineCopula} package). 
The selected families are stored 
in \code{$family}, and the sequentially estimated parameters in \code{$par} and \code{$par2}. 
The fit of the model is performed via the function \code{\link[VineCopula]{RVineCopSelect}} of the package \code{VineCopula}.
 
"The object \code{\link[VineCopula]{RVineMatrix}} includes the following information about the fit:
\describe{
\item{\code{se, se2}}{standard errors for the parameter estimates  (if \code{se = TRUE}; note that these are only 
approximate since they do not account for the sequential nature of the estimation,}  

\item{\code{nobs}}{number of observations,} 

\item{\code{logLik, pair.logLik}}{log likelihood (overall and pairwise)} 

\item{\code{AIC, pair.AIC}}{Aikaike's Informaton Criterion (overall and pairwise),} 

\item{\code{BIC, pair.BIC}}{Bayesian's Informaton Criterion (overall and pairwise),} 

\item{\code{emptau}}{matrix of empirical values of Kendall's tau,} 

\item{\code{p.value.indeptest}}{matrix of p-values of the independence test.}
}
}
\description{
This function fits either a C- or a D- vine model to a d-dimensional dataset of uniform variables. 
The fit of the pair-copula families is performed sequentially through the function \code{\link[VineCopula]{RVineCopSelect}} of 
the package \code{VineCopula}. The vine structure is selected among a group of C- and a D- vines which satisfy the requirement 
discussed in \emph{Bevacqua et al. (2017)}. This group is composed by all C- and D- vines from which the conditioning variables 
would be sampled as first when following the algorithms from \emph{Aas et al. (2009)}. Alternatively, if the 
vine matrix describing the vine structure is given to the function, the fit of the pair-copulas is directly performed skipping the vine structure 
selection procedure.
}
\note{
For a comprehensive summary of the vine copula model, use
\code{summary(object)}; to see all its contents, use \code{str(object)}". (VineCopula Documentation, version 2.1.1, pp. 103)
}
\examples{

# Example 1

# Read data 
data(dataset) 
data <- dataset$data[1:100,1:5]

# Define the variables Y and X. X are the conditioning variables, 
# which have to be positioned in the last columns of the data.frame
colnames(data) <- c("Y1","Y2","X3","X4","X5")

\dontrun{
# Select and fit a C- vine copula model, requiring that the 
RVM <- CDVineCondFit(data,Nx=3,treecrit="BIC",type="CVine",selectioncrit="AIC")
summary(RVM)
RVM$Matrix
}



# Example 2

# Read data 
data(dataset) 
data <- dataset$data[1:80,1:5]

# Define the variables Y and X. X are the conditioning variables, 
# which have to be positioned in the last columns of the data.frame
colnames(data) <- c("Y1","Y2","X3","X4","X5")

# Define a VineMatrix which can be used for conditional sampling
ListVines <- CDVineCondListMatrices(data,Nx=3)
Matrix=ListVines$DVine[[1]]
Matrix

\dontrun{
# Fit copula families for the defined vine:
RVM <- CDVineCondFit(data,Nx=3,Matrix=Matrix)
summary(RVM)
RVM$Matrix
RVM$family

# check
identical(RVM$Matrix,Matrix)

# Fit copula families for the defined vine, given a group of families to select from:
RVM <- CDVineCondFit(data,Nx=3,Matrix=Matrix,familyset=c(1,2,3,14))
summary(RVM)
RVM$Matrix
RVM$family

# Try to fit copula families for a vine which is not among those 
# that allow for conditional sampling:
Matrix
Matrix[which(Matrix==4)]=40
Matrix[which(Matrix==2)]=20
Matrix[which(Matrix==40)]=2
Matrix[which(Matrix==20)]=4
Matrix
RVM <- CDVineCondFit(data,Nx=3,Matrix=Matrix)
RVM
}

}
\author{
Emanuele Bevacqua
}
\references{
Bevacqua, E., Maraun, D., Hobaek Haff, I., Widmann, M., and Vrac, M.: Multivariate statistical modelling of compound events via pair-copula constructions: analysis of floods in Ravenna (Italy), 
Hydrol. Earth Syst. Sci., 21, 2701-2723, https://doi.org/10.5194/hess-21-2701-2017, 2017.
\href{https://www.researchgate.net/publication/317414374_Multivariate_statistical_modelling_of_compound_events_via_pair-copula_constructions_Analysis_of_floods_in_Ravenna_Italy}{[link]} 
\href{https://www.hydrol-earth-syst-sci.net/21/2701/2017/hess-21-2701-2017.html}{[link]} 

Aas, K., Czado, C., Frigessi, A. and Bakken, H.: Pair-copula constructions of multiple dependence, Insurance: 
Mathematics and Economics, 44(2), 182-198, <doi:10.1016/j.insmatheco.2007.02.001>, 2009.
\href{http://www.sciencedirect.com/science/article/pii/S0167668707000194}{[link]} 

Ulf Schepsmeier, Jakob Stoeber, Eike Christian Brechmann, Benedikt Graeler, Thomas 
Nagler and Tobias Erhardt (2017). VineCopula: Statistical Inference of Vine Copulas. R 
package version 2.1.1. \href{https://CRAN.R-project.org/package=VineCopula}{[link]}
}
\seealso{
\code{\link{CDVineCondSim}}, \code{\link{CDVineCondRank}}
}

