# Copyright 2025 Observational Health Data Sciences and Informatics
#
# This file is part of CohortGenerator
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

#' Create Rx Norm Cohort Template Definition
#' @description
#' Template cohort definition for all RxNorm ingredients. This cohort will use the
#' vocabulary tables to automatically generate a set of cohorts that have the
#' cohortId = conceptId * 1000. The "identifierExpression" can be customized for uniqueness.
#' @param connection Database connection object
#' @param identifierExpression An expression for setting the cohort id for the resulting cohort. Must produce unique ids
#' @param cdmDatabaseSchema CDM database schema
#' @param tempEmulationSchema Temporary emulation schema
#' @param cohortDatabaseSchema Cohort database schema
#' @param priorObservationPeriod (optional) Required prior observation period for individuals
#' @param vocabularyDatabaseSchema Vocabulary database schema
#' @param nameSuffix    A name suffix to use to add to the cohort names - this is useful if you're using multiple
#'                      parameterized versions of this definition
#' @returns A CohortTemplateDefinition instance
#' @export
createRxNormCohortTemplateDefinition <- function(connection,
                                                 identifierExpression = "CAST(concept_id as bigint) * 1000",
                                                 cdmDatabaseSchema,
                                                 tempEmulationSchema = getOption("sqlRenderTempEmulationSchema"),
                                                 cohortDatabaseSchema,
                                                 priorObservationPeriod = 365,
                                                 nameSuffix = "",
                                                 vocabularyDatabaseSchema = cdmDatabaseSchema) {
  sql <- SqlRender::readSql(
    sourceFile = system.file(
      "sql/sql_server/templates/rx_norm/references.sql",
      package = "CohortGenerator",
      mustWork = TRUE
    )
  )
  sql <- SqlRender::render(
    sql = sql,
    identifier_expression = identifierExpression,
    name_suffix = nameSuffix,
    vocabulary_database_schema = vocabularyDatabaseSchema
  )
  sql <- SqlRender::translate(
    sql = sql,
    targetDialect = connection@dbms,
    tempEmulationSchema = tempEmulationSchema
  )

  references <- DatabaseConnector::querySql(connection, sql, snakeCaseToCamelCase = TRUE)

  templateSql <- SqlRender::readSql(
    sourceFile = system.file(
      "sql/sql_server/templates/rx_norm/definition.sql",
      package = "CohortGenerator",
      mustWork = TRUE
    )
  )
  templateSql <- SqlRender::render(
    sql = templateSql,
    prior_observation_period = priorObservationPeriod,
    identifier_expression = identifierExpression,
    warnOnMissingParameters = FALSE
  )
  templateSql <- SqlRender::translate(
    sql = templateSql,
    targetDialect = connection@dbms,
    tempEmulationSchema = tempEmulationSchema
  )

  def <- createCohortTemplateDefintion(
    name = "All RxNorm ingredient exposures",
    templateSql = templateSql,
    references = references,
    sqlArgs = list(warnOnMissingParameters = FALSE)
  )

  return(invisible(def))
}

#' Create ATC Cohort Template Definition
#' @description
#' Template cohort definition for all ATC level 4 class exposures. The cohortId = conceptId * 1000 + 4.
#' The "identifierExpression" can be customized for uniqueness.
#' @param connection Database connection object
#' @param identifierExpression An expression for setting the cohort id for the resulting cohort. Must produce unique ids
#' @param cdmDatabaseSchema CDM database schema
#' @param tempEmulationSchema Temporary emulation schema
#' @param cohortDatabaseSchema Cohort database schema
#' @param mergeIngredientEras (optional) Boolean indicating if different ingredients under the same ATC code should be merged
#' @param priorObservationPeriod (optional) Required prior observation period for individuals
#' @param vocabularyDatabaseSchema Vocabulary database schema
#' @param nameSuffix    A name suffix to use to add to the cohort names - this is useful if you're using multiple
#'                      parameterized versions of this definition
#' @returns A CohortTemplateDefinition instance
#' @export
createAtcCohortTemplateDefinition <- function(connection,
                                              identifierExpression = "CAST(concept_id as bigint) * 1000",
                                              cdmDatabaseSchema,
                                              tempEmulationSchema = getOption("sqlRenderTempEmulationSchema"),
                                              cohortDatabaseSchema,
                                              nameSuffix = "",
                                              mergeIngredientEras = TRUE,
                                              priorObservationPeriod = 365,
                                              vocabularyDatabaseSchema = cdmDatabaseSchema) {
  sql <- SqlRender::readSql(
    sourceFile = system.file(
      "sql/sql_server/templates/atc/references.sql",
      package = "CohortGenerator",
      mustWork = TRUE
    )
  )
  sql <- SqlRender::render(
    sql = sql,
    identifier_expression = identifierExpression,
    name_suffix = nameSuffix,
    vocabulary_database_schema = vocabularyDatabaseSchema
  )
  sql <- SqlRender::translate(
    sql = sql,
    targetDialect = connection@dbms,
    tempEmulationSchema = tempEmulationSchema
  )
  #
  # sql <- SqlRender::loadRenderTranslateSql(
  #   sqlFilename = file.path("templates", "atc", "references.sql"),
  #   packageName = utils::packageName(),
  #   identifier_expression = identifierExpression,
  #   tempEmulationSchema = tempEmulationSchema,
  #   vocabulary_database_schema = vocabularyDatabaseSchema,
  #   name_suffix = nameSuffix
  # )

  references <- DatabaseConnector::querySql(
    connection = connection,
    sql = sql,
    snakeCaseToCamelCase = TRUE
  )
  templateSql <- SqlRender::readSql(
    sourceFile = system.file(
      "sql/sql_server/templates/atc/definition.sql",
      package = "CohortGenerator",
      mustWork = TRUE
    )
  )
  templateSql <- SqlRender::render(
    sql = templateSql,
    prior_observation_period = priorObservationPeriod,
    identifier_expression = identifierExpression,
    merge_ingredient_eras = mergeIngredientEras,
    warnOnMissingParameters = FALSE
  )
  templateSql <- SqlRender::translate(
    sql = templateSql,
    targetDialect = connection@dbms,
    tempEmulationSchema = tempEmulationSchema
  )

  def <- createCohortTemplateDefintion(
    name = paste0("All ATC class exposures", nameSuffix),
    templateSql = templateSql,
    references = references,
    sqlArgs = list(warnOnMissingParameters = FALSE)
  )

  return(invisible(def))
}

#' Create SNOMED Cohort Template Definition
#' @description
#' Template cohort definition for all OHDSI standard conditions. The cohortId = conceptId * 1000.
#' The "identifierExpression" can be customized for uniqueness.
#' This definition uses any valid SNOMED condition code and all its descendants.
#'
#' Excluded terms include word patterns:
#'
#'   '%finding', 'disorder of%', 'finding of%', 'disease of%' 'injury of%' '%by site' '%by body site' '%by mechanism'
#'    '%of body region' '%of anatomical site' '%of specific body structure%'
#'
#' Cohorts are first event.
#'
#' @param connection Database connection object
#' @param identifierExpression An expression for setting the cohort id for the resulting cohort. Must produce unique ids
#' @param cdmDatabaseSchema CDM database schema
#' @param tempEmulationSchema Temporary emulation schema
#' @param priorObservationPeriod (optional) Required prior observation period for individuals
#' @param requireSecondDiagnosis (optional) Require more than one diagnosis code
#' @param vocabularyDatabaseSchema Vocabulary database schema
#' @param nameSuffix    A name suffix to use to add to the cohort names - this is useful if you're using multiple
#'                      parameterized versions of this definition
#' @returns A CohortTemplateDefinition instance
#' @export
createSnomedCohortTemplateDefinition <- function(connection,
                                                 identifierExpression = "CAST(concept_id as bigint) * 1000",
                                                 cdmDatabaseSchema,
                                                 tempEmulationSchema = getOption("sqlRenderTempEmulationSchema"),
                                                 priorObservationPeriod = 365,
                                                 requireSecondDiagnosis = FALSE,
                                                 nameSuffix = "",
                                                 vocabularyDatabaseSchema = cdmDatabaseSchema) {
  sql <- SqlRender::readSql(
    sourceFile = system.file(
      "sql/sql_server/templates/snomed/references.sql",
      package = "CohortGenerator",
      mustWork = TRUE
    )
  )
  sql <- SqlRender::render(
    sql = sql,
    identifier_expression = identifierExpression,
    require_second_diagnosis = requireSecondDiagnosis,
    name_suffix = nameSuffix,
    vocabulary_database_schema = vocabularyDatabaseSchema
  )
  sql <- SqlRender::translate(
    sql = sql,
    targetDialect = connection@dbms,
    tempEmulationSchema = tempEmulationSchema
  )

  references <- DatabaseConnector::querySql(
    connection = connection,
    sql = sql,
    snakeCaseToCamelCase = TRUE
  )

  templateSql <- SqlRender::readSql(
    sourceFile = system.file(
      "sql/sql_server/templates/snomed/definition.sql",
      package = "CohortGenerator",
      mustWork = TRUE
    )
  )
  templateSql <- SqlRender::render(
    sql = templateSql,
    identifier_expression = identifierExpression,
    require_second_diagnosis = requireSecondDiagnosis
  )
  templateSql <- SqlRender::translate(
    sql = templateSql,
    targetDialect = connection@dbms,
    tempEmulationSchema = tempEmulationSchema
  )

  def <- createCohortTemplateDefintion(
    name = paste("All SNOMED Conditions", nameSuffix),
    templateSql = templateSql,
    references = references,
    sqlArgs = list(warnOnMissingParameters = FALSE)
  )

  return(invisible(def))
}
