#' Summarize Model Performance Metrics
#'
#' Computes mean performance metrics across multiple calibration and validation sets.
#' Typically used to summarize the results of soil property prediction models 
#' generated by preprocessing functions such as \code{pcr_preprocess()}, 
#' \code{plsr_preprocess()}, \code{lasso_preprocess()}, \code{rf_preprocess()}, 
#' or \code{cubist_preprocess()}.
#'
#' @param metric.list A list of MSD metric objects returned by one of the
#'   preprocessing/model functions. Each element corresponds to a model fit on a
#'   calibration/validation split.
#' @param soil_type Optional, an integer or string indicating which soil property
#'   was modeled (currently not used internally but kept for consistency).
#'
#' @return A named numeric vector of mean performance metrics across all splits:
#'   \describe{
#'     \item{LV}{Latent variable / model index}
#'     \item{cv-r2}{Cross-validated R-squared for calibration set}
#'     \item{cv-bias}{Bias in cross-validation for calibration set}
#'     \item{cv-rmse}{Root mean squared error in cross-validation for calibration set}
#'     \item{cal-mse}{Mean squared error for calibration set}
#'     \item{cal-rpiq}{Ratio of performance to interquartile distance for calibration set}
#'     \item{val-r2}{R-squared for validation set}
#'     \item{val-bias}{Bias for validation set}
#'     \item{val-rmse}{Root mean squared error for validation set}
#'     \item{val-mse}{Mean squared error for validation set}
#'     \item{val-rpiq}{Ratio of performance to interquartile distance for validation set}
#'   }
#'
#' @examples
#' \donttest{
#' msd_list <- pcr_preprocess(soil, vnir.matrix, j, preprocess = scale, type_of_soil = 2)
#' results_summary <- results(msd_list)
#' }
#'
#' @rdname soil_preprocess
#' @export
results <- function(metric.list, soil_type) {
  
  table.metrics <- matrix(data = NA, ncol = 11, nrow = 0)
  
  n <- length(metric.list)
  for(s in 1:n){
    met.tab <- matrix(data = NA, ncol = 11, nrow = 0)
    
    stats <- metric.list[[s]]
    temp <- c(
      names(metric.list)[[s]], 
      stats[4,1], stats[5,1], stats[6,1], stats[7,1], stats[14,1],
      stats[4,2], stats[5,2], stats[6,2], stats[7,2], stats[14,2]
    )
    
    met.tab <- rbind(met.tab, temp)
    table.metrics <- rbind(table.metrics, met.tab)
    
    rm(met.tab)
  }
  
  colnames(table.metrics) <- c(
    "LV", "cv-r2", "cv-bias", "cv-rmse", "cal-mse", "cal-rpiq", 
    "val-r2", "val-bias", "val-rmse", "val-mse", "val-rpiq"
  )
  
  return(apply(apply(table.metrics, 2, as.numeric), 2, mean))
}
