% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calcAOV.R
\name{calcAOV}
\alias{calcAOV}
\title{Calculate reliability using one-way ANOVA method}
\usage{
calcAOV(
  df,
  entity = "entity",
  y = "y",
  df.aggregate = FALSE,
  n = "n",
  mean = "mean",
  std.dev = "sd",
  ctrPerf = controlPerf()
)
}
\arguments{
\item{df}{dataframe (assumed to be observation-level unless \code{df.aggregate} is changed below)}

\item{entity}{data column containing the accountable entity identifier}

\item{y}{data column containing the outcome variable}

\item{df.aggregate}{set this to \code{TRUE} if the data have already been aggregated to include only summary data (sample size, means, and standard deviations) for each entity; default is \code{FALSE}.}

\item{n}{if using aggregated data, data column containing the sample sizes for each entity; default is \code{n}.}

\item{mean}{if using aggregated data, data column containing the sample means for each entity; default is \code{mean}.}

\item{std.dev}{if using aggregated data, data column containing the sample standard deviations for each entity entity; default is \code{sd}.}

\item{ctrPerf}{parameters to control performance measure calculation}
}
\value{
A list containing:
\itemize{
\item \code{MSB}: mean squares between entities
\item \code{MSW}: mean squares within entities
\item \code{F.stat}: F-statistic from one-way ANOVA
\item \code{entity}: list of entities
\item \code{n}: sample sizes for each entity
\item \code{n0}: aggregate sample size used in reliability calculation
\item \code{var.b.aov}: between-entity variance
\item \code{var.w.aov}: within-entity variance
\item \code{est.aov}: entity-level reliability estimates
}
}
\description{
This function estimates reliability using the one-way ANOVA method.
}
\details{
This function uses the \code{aov()} function from the \code{stats} package to calculate mean squares between
and mean squares within entities.
}
\examples{
# Simulate data
df <- simulateData(n.entity = 50, n.obs = 100, mu = 25, r = .7, data.type = 'normal')

# Calculate reliability
out <- calcAOV(df = df, entity = 'entity', y = 'y')
summary(out$est.aov)

# Plot reliability by entity sample size
plot(out$n, out$est.aov)


## Reliability can also be calculated with data aggregated by entity
df.agg <- data.frame(
          entity = aggregate(y ~ entity, data = df, length)$entity,
          n = aggregate(y ~ entity, data = df, length)$y,
          mean = aggregate(y ~ entity, data = df, mean)$y,
          sd = aggregate(y ~ entity, data = df, sd)$y
          )

out2 <- calcAOV(df = df.agg, df.aggregate = TRUE, n = 'n', mean = 'mean', std.dev = 'sd')
summary(out2$est.aov)

}
\references{
Nieser KJ, Harris AH. Comparing methods for assessing the reliability of health care quality measures. Statistics in Medicine. 2024 Oct 15;43(23):4575-94.
}
\author{
Kenneth Nieser (nieser@stanford.edu)
}
