#' Fit QuanDA for imbalanced binary classification
#'
#' QuanDA fits a quantile-regression-based discriminant with label jittering.
#' For each candidate quantile level \eqn{\tau}, the binary labels are jittered
#' (adding \eqn{U(0,1)}), a penalized quantile regression is fit multiple times,
#' and the coefficient vectors are averaged. The best \eqn{\tau} is selected by AUC.
#'
#' @param x A numeric matrix of predictors with \eqn{n} rows (observations) and \eqn{p} columns (features).
#' @param y A binary response vector of length \eqn{n} with values \code{0} or \code{1}.
#' @param lambda Optional numeric vector of penalty values (largest \code{lambda[1]}).
#'   If \code{NULL}, a default sequence will be generated from the data.
#' @param lam2 Numeric, secondary penalty (ridge/elastic term) passed to \code{hdqr}. Default \code{0.01}.
#' @param n_rep Integer, number of jittering repetitions (averaged). Default \code{10}.
#' @param tau_window Width around the class rate to explore quantiles.
#'   Candidate \eqn{\tau} are \eqn{b + \{-w,\ldots,w\}} in steps of 0.01,
#'   clipped to \eqn{[0,1]}, where \eqn{b} is the class rate and \eqn{w} is \code{tau_window}.
#'   Default \code{0.1}.
#' @param nfolds Integer, number of CV folds used by \code{cv_z()}. Default \code{5}.
#' @param maxit,maxit_cv,eps,eps_cv Controls for inner optimizers and CV helper.
#'
#' @details
#' We jitter labels via \eqn{z_i = y_i + U_i}, where \eqn{U_i \sim \mathrm{Unif}(0,1)},
#' fit penalized quantile regression at multiple \eqn{\tau}, average coefficients over \code{n_rep} jitters,
#' compute AUCs on the original \eqn{(x,y)}, and pick the \eqn{\tau} that maximizes AUC.
#'
#' @return An object of class \code{"quanda"} with elements:
#' \describe{
#'   \item{beta}{Numeric vector of length \eqn{p+1} (intercept first).}
#'   \item{tau_grid}{Numeric vector of candidate \eqn{\tau} values.}
#'   \item{tau_best}{Chosen \eqn{\tau}.}
#'   \item{auc}{Vector of AUCs across \eqn{\tau}.}
#'   \item{call}{The matched call.}
#' }
#'
#' @keywords quantile regression binary-classification imbalanced-learning
#' @importFrom pROC roc
#' @importFrom hdqr hdqr
#' @importFrom stats coef runif
#' @examples
#' data(breast)
#' X <- as.matrix(X)
#' y <- as.numeric(as.character(y))
#' y[y==-1]=0
#' \donttest{fit <- quanda(X, y)}
#' \donttest{pred <- predict(fit, tail(X))}
#' @export
quanda <- function(x, y,
                   lambda = 10^(seq(1, -4, length.out = 30)),
                   lam2 = 0.01,
                   n_rep = 10,
                   tau_window = 0.05,
                   nfolds = 5,
                   maxit = 1e4, eps = 1e-7,
                   maxit_cv = 1e4, eps_cv = 1e-5) {
  call <- match.call()

  x <- as.matrix(x)
  if (!is.numeric(x)) stop("x must be numeric.")
  n <- nrow(x); p <- ncol(x)
  if (length(y) != n) stop("length(y) must equal nrow(x).")
  if (!all(y %in% c(0, 1))) stop("y must be binary {0,1}.")
  lambda <- as.numeric(lambda)
  lambda <- sort(lambda[lambda > 0], decreasing = TRUE)
  if (!length(lambda)) stop("lambda must contain positive values.")

  base_rate <- mean(y == 0)
  tau_grid <- seq(max(0, base_rate - tau_window),
                  min(1, base_rate + tau_window), by = 0.01)
  n_tau <- length(tau_grid)

  coef_arr <- array(NA, dim = c(p + 1, n_rep, n_tau))
  auc_mat <- matrix(NA, rep, n_tau)
  tau.win <- rep(0, n_tau)
  for (r in seq_len(n_rep)) {
    z <- y + runif(n)  # jitter labels
    for (i in seq_len(n_tau)) {
      tau_i <- tau_grid[i]
      cv.fit <- cv_z(x, y, z,
                     lambda = lambda, lam2 = lam2, tau = tau_i,
                     nfolds = nfolds, eps = eps_cv, maxit = maxit_cv)
      lam <- cv.fit$lambda.min
      lam_list <- seq(lam+2, lam, length.out=5)
      fit <- hdqr(x, z, lambda = lam_list, lam2=lam2, tau = tau_i, eps = eps, maxit = maxit)
      cf <- coef(fit, s = lam)
      coef_arr[, r, i] <- as.numeric(cf[, 1])  
      auc_mat[r, i] <- cv.fit$cvmax
    }
    best_idx <- which(auc_mat[r, ] == max(auc_mat[r, ], na.rm = TRUE))
    if (length(best_idx) > 1) {
      pi0 <- base_rate
      best_idx <- best_idx[ which.min(abs(tau_grid[best_idx] - pi0)) ]
    }
    tau.win[best_idx] <- tau.win[best_idx]+1
  }
  i_best <- which.max(tau.win)
  coef_avg <- apply(coef_arr, c(1, 3), mean, na.rm = TRUE)  
  beta_best <- coef_avg[, i_best]

  out <- list(
    beta     = as.numeric(beta_best),
    tau_grid = tau_grid,
    tau_best = tau_grid[i_best],
    call     = call
  )
  class(out) <- "quanda"
  out
}
