% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/halfnorm.like.R
\name{halfnorm.like}
\alias{halfnorm.like}
\title{Half-normal distance function}
\usage{
halfnorm.like(a, dist, covars, w.hi = NULL)
}
\arguments{
\item{a}{A vector or matrix of covariate 
and expansion term 
coefficients. If matrix, dimension is 
\emph{k} X \emph{p}, where 
\emph{k} = \code{nrow(a)}) is the number of coefficient
vectors to evaluate (cases) and \emph{p} 
= \code{ncol(a)})
is the number of covariate and expansion 
coefficients in the likelihood (i.e., rows are 
cases and columns are covariates). If \code{a} is a 
dimensionless vector, it is interpreted as a 
single row with \emph{k} = 1. 
Covariate coefficients in \code{a} are the first 
\emph{q} values (\emph{q} <= \emph{p}), and must 
be on a log scale.}

\item{dist}{A numeric vector of length \emph{n} or 
a single-column matrix (dimension \emph{n}X1) containing 
detection distances at which to evaluate the likelihood.}

\item{covars}{A numeric vector of length \emph{q} or a
matrix of dimension \emph{n}X\emph{q} containing 
covariate values 
associated with distances in argument \code{dist}.}

\item{w.hi}{A numeric scalar containing maximum 
distance. The right-hand cutoff or upper limit. 
Ignored by some likelihoods (such as halfnorm, 
negexp, and hazrate), but is a fixed parameter
in other likelihoods (such as oneStep, heber, 
and uniform).}
}
\value{
A list containing the following two components:
\itemize{
  \item \bold{L.unscaled}: A matrix of size 
   \emph{n}X\emph{k}X\emph{b} 
   containing likelihood values evaluated at 
   distances in \code{dist}.
   Each row is associated with 
   a single distance, and each column is associated with 
   a single case (row of \code{a}).  This matrix is  
   "unscaled" because the underlying likelihood does 
   not integrate to one. Values in \code{L.unscaled} 
   are always greater than or equal to zero.
   
 \item \bold{params}: A \emph{n}X\emph{k}X\emph{b} array 
 of the 
 likelihood's (canonical) parameters. First page contains 
 parameter values related to covariates (i.e., 
 \emph{s = exp(x'a)}),
 while subsequent pages contain other parameters. 
 \emph{b} = 1 for halfnorm, negexp; \emph{b} = 2 for 
 hazrate and others.
 Rows correspond to distances in \code{dist}. Columns 
 correspond to rows from argument \code{a}. 
}
}
\description{
Evaluate the half-normal distance function, 
for sighting distances, potentially including covariates
and expansion terms
}
\details{
The half-normal distance function is 
\deqn{f(d|s) = \exp(-d^2 / (2*s^2))}{f(d|s) = exp(-d^2 / (2*s^2))}
where \eqn{s = exp(x'a)}, \eqn{x} is a vector of 
covariate values associated with distance \eqn{d} 
(i.e., a row of \code{covars}), and 
\eqn{a} is a vector of the first $q$ (=\code{ncol(covars)}) 
values in argument \code{a}.

Some authors parameterize the halfnorm without 
the "2" in the denominator of the exponent. 
\code{Rdistance} includes 
"2" in this denominator to make 
quantiles of the half normal agree with 
the standard normal. This means that half-normal 
coefficients in 
Rdistance (i.e., \emph{s = exp(x'a)}) can be 
interpreted as normal standard errors.  
Approximately 95\% of distances should 
occur between 0 and 2\emph{s}.
}
\examples{
 
d <- seq(0, 100, length=100)
covs <- matrix(1,length(d),1)
halfnorm.like(log(20), d, covs)

plot(d, halfnorm.like(log(20), d, covs)$L.unscaled, type="l", col="red")
lines(d, halfnorm.like(log(40), d, covs)$L.unscaled, col="blue")

# Evaluate 3 functions at once using matrix of coefficients:
# sigma ~ 20, 30, 40
coefs <- matrix(log(c(7.39,7.33, 4.48,44.80, 2.72,216.54))
               , byrow = TRUE
               , ncol=2) # (3 coef vectors)X(2 covars)
covs <- matrix(c(rep(1,length(d))
               , rep(0.5,length(d)))
               , nrow = length(d)) # 100 X 2
L <- halfnorm.like( coefs, d, covs ) 
L$L.unscaled # 100 X (3 coef vectors)
L$params     # 100 X (3 coef vectors); ~ log(c(20,30,40))
matplot(d, L$L.unscaled, type="l")

}
\seealso{
\code{\link{dfuncEstim}}, 
         \code{\link{abundEstim}},
         other \code{<likelihood>.like} functions
}
