% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computePersistentEntropy}
\alias{computePersistentEntropy}
\title{A Vector Summary of the Persistent Entropy Summary Function}

\description{

For a given persistence diagram \eqn{D={(b_i,d_i)}_{i=1}^N} (corresponding to a specified homological dimension), \code{computePersistentEntropy()} vectorizes the persistent entropy summary function \deqn{S(t)=-\sum_{i=1}^N \frac{l_i}{L}\log_2{(\frac{l_i}{L}})\bold 1_{[b_i,d_i]}(t),} where \eqn{l_i=d_i-b_i} and \eqn{L=\sum_{i=1}^Nl_i}, based on a scale sequence \code{scaleSeq}. If \eqn{N=1}, \eqn{L} is set to 1. The evaluation method depends on the argument \code{evaluate}. Points in \eqn{D} with infinite death values are ignored.
}

\usage{
computePersistentEntropy(D, homDim, scaleSeq, evaluate = "intervals")
}

\arguments{
\item{D}{a persistence diagram: a matrix with three columns containing the homological dimension, birth and death values respectively.}

\item{homDim}{the homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.). Rows in \code{D} are filtered based on this value.}

\item{scaleSeq}{a numeric vector of increasing scale values used for vectorization.}

\item{evaluate}{a character string indicating the evaluation method. Must be either \code{"intervals"} (default) or \code{"points"}.}

}

\details{
The function extracts rows from \code{D} where the first column equals \code{homDim}, and computes values based on the filtered data and \code{scaleSeq}. If \code{D} does not contain any points corresponding to \code{homDim}, a vector of zeros is returned.

}
\value{
A numeric vector containing elements computed using \code{scaleSeq}=\eqn{\{t_1,t_2,\ldots,t_n\}} according to the method specified by \code{evaluate}.

\itemize{
  \item \code{"intervals"}: Computes average values of the persistent entropy summary function over intervals defined by consecutive elements in \code{scaleSeq}:

  \deqn{\Big(\frac{1}{\Delta t_1}\int_{t_1}^{t_2}S(t)dt,\frac{1}{\Delta t_2}\int_{t_2}^{t_3}S(t)dt,\ldots,\frac{1}{\Delta t_{n-1}}\int_{t_{n-1}}^{t_n}S(t)dt\Big)\in\mathbb{R}^{n-1},}
where \eqn{\Delta t_k=t_{k+1}-t_k}.

  \item \code{"points"}: Computes values of the persistent entropy summary function at each point in \code{scaleSeq}:

  \deqn{(S(t_1),S(t_2),\ldots,S(t_n))\in\mathbb{R}^n.}
}
}

\author{Umar Islambekov}

\references{
1. Atienza, N., Gonzalez-Díaz, R., & Soriano-Trigueros, M. (2020). On the stability of persistent entropy and new summary functions for topological data analysis. Pattern Recognition, 107, 107509.
}

\examples{
N <- 100 # The number of points to sample

set.seed(123) # Set a random seed for reproducibility

# Sample N points uniformly from the unit circle and add Gaussian noise
theta <- runif(N, min = 0, max = 2 * pi)
X <- cbind(cos(theta), sin(theta)) + rnorm(2 * N, mean = 0, sd = 0.2)

# Compute the persistence diagram using the Rips filtration built on top of X
# The 'threshold' parameter specifies the maximum distance for building simplices
D <- TDAstats::calculate_homology(X, threshold = 2)

scaleSeq = seq(0, 2, length.out = 11) # A sequence of scale values

# Compute a vector summary of the persistent entropy summary function for homological dimension H_0
computePersistentEntropy(D, homDim = 0, scaleSeq)

# Compute a vector summary of the persistent entropy summary function for homological dimension H_1
computePersistentEntropy(D, homDim = 1, scaleSeq)
}
