# printing methods: print.callback, print.stat_paired, print.callback_stat,
# print.callback_comp

#'
#' Prints the structure of the experiment
#'
#' @description
#' Computes the number of tests available for each pair of candidates
#'
#' @param x a \code{stat_paired} object.
#' @param ... further arguments passed to or from other methods.
#'
#' @return Printed output.
#' @examples
#' data(labour1)
#' x <- callback(data=labour1,cluster="offer",candid="hist",callback="callback",
#' comp = "all")
#' print(x)
#' @rdname print.callback
#' @export

print.callback <- function(x, ...) {
  cat(
    "\n",
    "Structure of the experiment \n",
    "---------------------------\n",
    "\n",
    "Candidates defined by:",
    x[["candid"]],
    "\n",
    "Callback variable:",
    x[["callback"]],
    "\n",
    "\n Number of tests for each candidate:\n"
  )
  single <- table(x[["fds"]][, "candid"])
  print(single)
  
  paired <- as.data.frame(lapply(x[["pfds"]], nrow), row.names = "")
  cat("\n", "\n Number of tests for each pair of candidates:\n\n")
  print(paired, row.names = FALSE)
  z <- list(single = single, paired = paired)
  
  if (is.null(x$ofds)) {
    cat("\n", "\n Duplication detected")
  } else {
    cat("\n",
        "\n Number of tests with all the candidates:",
        nrow(x$ofds))
  }
}

#'
#' Print the callback counts analysis
#'
#' @description
#' Computes the callback numbers for each candidate.
#'
#' @param x a \code{stat_paired} object.
#' @param ... further arguments passed to or from other methods.
#'
#' @return Printed output.
#'
#' @examples
#' data(labour1)
#' print(stat_paired(callback(labour1, "offer", "hist", "callback",
#' comp = "all")))
#' @rdname print.stat_paired
#' @export

print.stat_paired <- function(x, ...) {
  m <- x$counts
  cat("\n", "Callback counts:\n", "----------------\n")
  colnames(m)[c(5:9)] <-
    c("Neither", "Only 1", "Only 2", "Both", "Difference")
  print(m)
}

#'
#' Prints the callback proportions analysis
#'
#' @description
#' Prints the statistics about the callback rates or shares.
#'
#' @param x a \code{callback_stat} object.
#' @param digits minimal number of significant digits.
#' @param ... further arguments passed to or from other methods.
#'
#' @return Printed output.
#'
#' @examples
#' data(labour1)
#' x <- callback(labour1, "offer", "hist", "callback", comp = "all")
#' print(stat_mcr(x))
#' @export

print.callback_stat <- function(x, digits = 3, ...) {
  l_num <- as.integer(round(x$specif["level"] * 100, 1))
  conv <- as.character(x$specif["convention"])
  conf <- as.character(x$specif["confid"])
  sname <- as.character(x$specif["name"])
    cat(
      "\n",
      "Proportions:", conv,
      "\n",
      "Confidence intervals:",conf,"at",l_num,"%\n",
      "\n"
    )
    print(x$props)
    
    if (!is.null(x$stats)){
    cat(
      "\n",
      sname,
      "\n"
    )
    print(x$stats)
    cat("\n Signif. codes: 0 '***' 0.001 '**' 0.01 '*' 0.05 '.' 0.10 ' ' 1")}
}

#'
#' Prints the structure of a component model
#'
#' @param x a list with class \code{"callback_comp"}
#' @param ... further arguments passed to or from other methods.
#'
#' @return Printed output.
#'
#' @author Emmanuel Duguet
#'
#' @examples
#' model <- list(c("license"),c("woman"),c("woman","license","inter"))
#' print(callback_comp(mobility1,"offer",c("gender","licenses"),"callback",model))
#' @export

print.callback_comp <- function(x, ...) {
  aux_cand <- x$aux_cand
  aux_boole <- x$aux_boole
  aux_det <- x$aux_det
  aux_model <- x$aux_model
  eq_list <- list()
  
  for (cand in aux_cand) {
    eq_list[[cand]] <- paste(cand,
                             paste(aux_model[[cand]], collapse = " + "),
                             sep = " = ")
  }
  
  m <- t(data.frame(eq_list))
  
  cat("\n",
      "Your components model:",
      "----------------------",
      " ",
      m,
      sep = "\n")
  
  cat(" \n", "Boolean matrix (A):", "-------------------", " ", sep = "\n")
  print(aux_boole)
  
  cat(" \n", "Determinant of A'A = ", aux_det, ", ", sep = "")
  
  rb <- nrow(aux_boole)
  cb <- ncol(aux_boole)
  
  val <- ifelse((rb < cb) |
                  (aux_det == 0),
                "the model is not identifiable.",
                ifelse(
                  rb == cb,
                  "the model is just identified.",
                  paste(
                    "the model is overidentified with",
                    rb - cb,
                    "degree(s) of freedom."
                  )
                )
  )
  cat(val)
}

#'
#' Prints the components of a component model
#'
#' @param x a list with class \code{"callback_reg"}
#' @param ... further arguments passed to or from other methods.
#'
#' @return Printed output.
#'
#' @author Emmanuel Duguet
#'
#' @examples
#' model <- list(c("license"),c("woman"),c("woman","license","inter"))
#' z <- reg(callback_comp(mobility1,"offer",c("gender","licenses"),"callback",model))
#' print(z)
#' @export

print.callback_reg <- function(x, ...) {
  cat("\n",
      "Components:\n",
      "\n")
  b <- x$reg$estim$coef
  names(b) <- x$reg$estim$c_names
  print(b)
  cat("\n",
      "Use summary() for the regression table.")
}
