#' Print a 'degrossData' object.
#'
#' @description Print a summary of the information contained in a \code{\link{degrossData.object}} used by \code{\link{degross}} for density estimation from tabulated frequency and moment data.
#'
#' @usage
#' \method{print}{degrossData}(x, ...)
#'
#' @param x  A \link{degrossData.object} generated by \link{degrossData}.
#' @param ... Possible additional printing options for a matrix object.
#'
#' @return Print the tabulated summary statistics contained in the \link{degrossData.object} \code{x}, with additional information on the total sample size, numbers of small and big bins with their limits, the number of B-splines planned for density estimation using \code{\link{degross}}.
#'
#' @author Philippe Lambert \email{p.lambert@uliege.be}
#' @references
#' Lambert, P. (2021) Moment-based density and risk estimation from grouped summary statistics. arXiv:2107.03883.
#'
#' @seealso \code{\link{degross}}, \code{\link{degrossData}}
#'
#' @examples
#' sim = simDegrossData(n=3500, plotting=TRUE)
#' obj.data = degrossData(Big.bins=sim$Big.bins, freq.j=sim$freq.j, m.j=sim$m.j)
#' print(obj.data)
#'
#' @export
print.degrossData = function(x,...){
  cat("\nTotal sample size:",sum(x$freq.j),"\n\n")
  mat = cbind(freq.j=x$freq.j, m.j=x$m.j)
  rownames(mat) = paste("Bin",1:x$J,sep="")
  cat("Observed class frequencies and central moments:\n")
  print.default(mat,...)
  cat("\nBig bin limits:",x$Big.bins,"\n")
  cat("No of big bins:",x$J,
      " ; No of small bins:",x$I,
      " ; No of B-splines:",ncol(x$B.i),"\n")
}
