% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weights.R
\name{getWeights}
\alias{getWeights}
\title{Calculate phecode-specific weights for phenotype risk scores}
\usage{
getWeights(
  demos,
  phecodeOccurrences,
  method = c("prevalence", "logistic", "cox", "loglinear", "prevalence_precalc"),
  methodFormula = NULL,
  negativeWeights = FALSE,
  dopar = FALSE
)
}
\arguments{
\item{demos}{A data.table having one row per person in the cohort. Must have
a column \code{person_id}. When the \code{cox} method is used, \code{demos}
must have columns \code{first_age} and \code{last_age} corresponding to first and
last age of visit (in years).}

\item{phecodeOccurrences}{A data.table of phecode occurrences for each person
in the cohort. Must have columns \code{person_id} and \code{phecode} under the
"prevalence" or "logistic" methods, columns \code{person_id}, \code{phecode}, and
\code{num_occurrences} under the "loglinear" method, and columns \code{person_id},
\code{phecode}, and \code{occurrence_age} under the "cox" method. \code{num_occurrences}
refers to the number of unique dates a phecode was recorded for a person.
\code{occurrence_age} refers to the first age (in years) a person acquired a
phecode.}

\item{method}{A string indicating the statistical model for calculating
weights.}

\item{methodFormula}{A formula representing the right-hand side of the model
corresponding to \code{method}. All terms in the formula must correspond to
columns in \code{demos}. A method formula is not required for the "prevalence"
and "prevalence_precalc" methods. Do not use age-related covariates with
the "cox" method.}

\item{negativeWeights}{Logical indicating whether to allow negative weights
for individuals with no occurrences of a phecode. This option is not
required for the "loglinear" method since under this method, individuals
with a nonzero phecode occurrence can also have negative weights.}

\item{dopar}{Logical indicating whether to run calculations in parallel if
a parallel backend is already set up, e.g., using
\code{\link[doParallel:registerDoParallel]{doParallel::registerDoParallel()}}. Recommended to minimize runtime.}
}
\value{
A data.table with columns \code{person_id}, \code{phecode}, \code{pred}, and \code{w}.
The column \code{pred} represents a different quantity depending on \code{method}.
Under the "prevalence" \code{method}, it is fraction of the cohort that has
at least one occurrence of the given phecode. The "prevalence_precalc"
\code{method} is similar to the "prevalence" \code{method} but \code{pred} is calculated
based on EHR data from the Vanderbilt University Medical Center.
Under "logistic" or "cox" \code{method}, it is the predicted probability of
given individual having a given phecode based on \code{methodFormula}.
Under the "loglinear" \code{method}, it is the predicted
\code{log2(num_occurrences + 1)} of a given phecode for a given individual
based on \code{methodFormula}. For the "prevalence", "prevalence_precalc",
"cox", and "logistic" \code{method}s, weight is calculated as \code{-log10(pred)}
when an individual has non-zero phecode occurrence and \code{log10(1 - pred)}
when an individual has zero phecode occurrence. For the "loglinear" \code{method}
weight is calculated as the difference between the observed
\code{log2(num_occurrences + 1)} and \code{pred}.
}
\description{
This is typically the second step of an analysis using phenotype risk scores,
the next is \code{\link[=getScores]{getScores()}}.
}
\examples{
library('data.table')
library('survival')

# map ICD codes to phecodes
phecodeOccurrences = getPhecodeOccurrences(icdSample)

# calculate weights using the prevalence method
weightsPrev = getWeights(demoSample, phecodeOccurrences)

# calculate weights using the prevalence method
# (assign negative weights to those with zero phecode occurrence)
weightsPrevNeg = getWeights(
  demoSample, phecodeOccurrences, negativeWeights = TRUE)

# calculate weights using the logistic method
weightsLogistic = getWeights(
  demoSample, phecodeOccurrences, method = 'logistic', methodFormula = ~ sex)

# calculate weights using the loglinear method
phecodeOccurrences2 = phecodeOccurrences[, .(
  num_occurrences = uniqueN(entry_date)), by = .(person_id, phecode)]
weightsLoglinear = getWeights(
  demoSample, phecodeOccurrences2, method = 'loglinear', methodFormula = ~ sex)

# calculate weights using the cox method
phecodeOccurrences3 = phecodeOccurrences[, .(
  first_occurrence_date = min(entry_date)) , by = .(person_id, phecode)]
phecodeOccurrences3 = merge(
  phecodeOccurrences3, demoSample[, .(person_id, dob)], by = 'person_id')
phecodeOccurrences3[,
  occurrence_age := as.numeric((first_occurrence_date - dob)/365.25)]
phecodeOccurrences3[, `:=`(first_occurrence_date = NULL, dob = NULL)]
demoSample3 = demoSample[, .(
  person_id, sex,
  first_age = as.numeric((first_visit_date - dob)/365.25),
  last_age = as.numeric((last_visit_date - dob)/365.25))]
weightsCox = getWeights(
  demoSample3, phecodeOccurrences3, method = 'cox', methodFormula = ~ sex)

# calculate weights using pre-calculated weights based on data from
# Vanderbilt University Medical Center
weightsPreCalc = getWeights(
  demoSample, phecodeOccurrences, method = 'prevalence_precalc')
}
\seealso{
\code{\link[=getPhecodeOccurrences]{getPhecodeOccurrences()}}, \code{\link[=getScores]{getScores()}}
}
