% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotluck.R
\name{plotluck}
\alias{plotluck}
\title{"I'm feeling lucky" for ggplot}
\usage{
plotluck(data, formula, weights, opts = plotluck.options(), ...)
}
\arguments{
\item{data}{a data frame.}

\item{formula}{an object of class \code{\link[stats]{formula}}: a symbolic description
 of the relationship of up to three variables.
\tabular{lll}{
\strong{Formula}\tab\strong{Meaning}\tab\strong{Plot types}\cr
\code{y~1}\tab Distribution of single variable\tab Density, histogram, scatter, dot, bar\cr
\code{y~x}\tab One explanatory variable\tab Scatter, hex, violin, box, spine, heat\cr
\code{y~x+z}\tab Two explanatory variables\tab heat, spine\cr
\code{y~1|z} or \code{y~x|z}\tab One conditional variable\tab Represented through coloring or facetting\cr
\code{y~1|x+z}\tab Two conditional variables\tab Represented through facetting\cr}
In addition to these base plot types, the dot symbol \code{"."} can also be used,
and denotes all variables in the data frame. This gives rise to a lattice or
series of plots (use with caution, can be slow).
\tabular{lll}{
\strong{Formula}\tab\strong{Meaning}\cr
\code{.~1}\tab Distribution of each variable in the data frame, separately\cr
\code{y~.}\tab Plot \code{y} against each variable in the data frame\cr
\code{.~x}\tab Plot each variable in the data frame against \code{x}\cr
\code{.~.}\tab Plot each variable in the data frame against each other.\cr}
 See also section "Generating multiple plots at once" below.}

\item{weights}{observation weights or frequencies (optional).}

\item{opts}{a named list of options (optional); See also \code{\link{plotluck.options}}.}

\item{...}{additional parameters to be passed to the respective ggplot2 geom objects.}
}
\value{
a ggplot object, or a plotluck.multi object if the dot symbol was used.
}
\description{
The purpose of \code{plotluck} is to let the user focus on \emph{what} to plot,
and automate the \emph{how}. Given a dependency formula with up to three
variables, it tries to choose the most suitable type of plot. It also automates
sampling large datasets, correct handling of observation weights, logarithmic
axis scaling, ordering and pruning of factor levels, and overlaying smoothing
curves or median lines.
}
\section{Determining the type of plot}{
 Besides the shape of the formula, the
  algorithm takes into account the type of variables as either numeric, ordered,
  or unordered factors. Often, it makes sense to treat ordered factors similarly
  as numeric types.

  One-variable numeric (resp. factor) distributions are usually represented by
  density (resp. Cleveland dot) charts, but can be overridden to histograms or
  bar plots using the \code{geom} option. Density plots come with an overlaid
  vertical median line.

  For two numerical variables, by default a scatter plot is produced, but for
  high numbers of points a hexbin is preferred (option \code{min.points.hex}).
  These plots come with a smoothing line and standard deviation.

  The relation between two factor variables can be depicted best by spine
  (a.k.a., mosaic) plots, unless they have too many levels (options
  \code{max.factor.levels.spine.x}, \code{max.factor.levels.spine.y},
  \code{max.factor.levels.spine.z}). Otherwise, a heat map is produced.

  For a mixed-type (factor/numeric) pair of variables, violin (overridable
  to box) plots are generated. However, if the resulting graph would contain
  too many (more than \code{max.factor.levels.violin}) violin plots in a row,
  the algorithm switches automatically. The number of bins of a histogram can
  be customized with \code{n.breaks.histogram}. The default setting, \code{NA},
  applies a heuristic estimate.

  The case of a response two dependent variables (`y~x+z`) is covered by
  either a spine plot (if all are factors) or a heat map.

  In many cases with few points for one of the aggregate plots, a scatter
  looks better (options \code{min.points.density}, \code{min.points.violin},
  \code{min.points.hex}).

  If each factor combination occurs only once in the data set, we resort to
  bar plots.
}

\section{Conditional variables}{
 Conditional variables are represented by either
  trying to fit into the same graph using coloring (\code{max.factor.levels.color}),
  or by facetting (preferred dimensions \code{facet.num.wrap} (resp.
  \code{facet.num.grid}) for one resp. two variables). Numeric vectors are
  discretized accordingly. Facets are laid out horizontally or vertically
  according to the plot type, up to maximum dimensions of \code{facet.max.rows}
  and \code{facet.max.cols}.
}

\section{Reordering of factor levels}{
 To better illustrate the relation
 between an independent factor variable and a dependent numerical variable
 (or an ordered factor), levels are reordered according to the value of the
 dependent variable. If no other numeric or ordered variable exists, we
 sort by frequency.
}

\section{Instance weights}{
 Argument \code{weights} allows to specify weights
 or frequency counts for each row of data. All plots and summary statistics
 take weights into account when supplied. In scatter and heat maps, weights
 are indicated either by a shaded disk with proportional area (default) or by
 jittering (option \code{dedupe.scatter}), if the number of duplicated points
 exceeds \code{min.points.jitter}. The amount of jittering can be controlled
 with \code{jitter.x} and \code{jitter.y}.
}

\section{Axis scaling}{
 \code{plotluck} supports logarithmic and log-modulus
 axis scaling. log-modulus is considered if values are both positive and
 negative; in this case, the transform function is \code{f(x) = sign(x) *
 log(1+abs(x))}.

 The heuristic to apply scaling is based on the proportion of total display
 range that is occupied by the 'core' region of the distribution between the
 lower and upper quartiles; namely, the fact whether the transform could
 magnify this region by a factor of at least \code{trans.log.thresh}.
}

\section{Missing values}{
 By default, missing (\code{NA} or \code{NaN}) values
 in factors are are shown as a special factor level code{"?"}. They can be
 removed by setting \code{na.rm=TRUE}. Conventionally, missing numeric values
 are not shown.
}

\section{Sampling}{
 For very large data sets, plots can take a very long time
 (or even crash R). \code{plotluck} has a built-in stop-gap: If the data
 comprises more than \code{sample.max.rows}, it will be sampled down to that
 size (taking into account \code{weights}, if supplied).
}

\section{Factor preprocessing}{
  Character (resp. logical) vectors are converted to
 unordered (resp. ordered) factors.

 Frequently, when numeric variables have very few values despite sufficient
 data size, it helps to treat these values as the levels of a factor; this is
 governed by option \code{few.unique.as.factor}.

 If an unordered factor has too many levels, plots can get messy. In this
 case, only the \code{max.factor.levels} most frequent ones are retained,
 while the rest are merged into a default level \code{".other."}.
}

\section{Coloring}{
 If \code{color} or \code{fill} aesthetics are used to
 distinguish different levels or ranges of a variable, the color scheme adjusts
 to the type. Preferably, a sequential (resp. qualitative) palette is chosen
 for a numeric/ordered (unordered) factor (\code{palette.brewer.seq},
 \code{palette.brewer.qual}); see also \link[RColorBrewer]{RColorBrewer}.
}

\section{Generating multiple plots at once}{
 If \code{formula} contains a dot
 (\code{"."}) symbol, the function creates a number of 1D or 2D plots by calling
 \code{plotluck} repeatedly. As described above, this allows either single
 distribution, one-vs-all and all-vs-all variable plots. To save space,
 rendering is minimal without axis labels.

 In the all-vs-all case, the diagonal contains 1D distribution plots, analogous
 to the behavior of the default plot method for data frames, see
 \code{\link[graphics]{plot.data.frame}}.

 With setting \code{in.grid=FALSE}, plots are produced in a sequence, otherwise
 together on one or multiple pages, if necessary (default). Page size is
 controlled by \code{multi.max.rows} and  \code{multi.max.cols}.

 With \code{entropy.order=TRUE}, plots are sorted by an estimate of
 empirical conditional entropy, with the goal of prioritizing the more
 predictive variables. Set \code{verbose=TRUE} if you want to see the actual
 values. For large data sets the calculation can be time consuming; entropy
 calculation can be suppressed by setting \code{multi.entropy.order=FALSE}.

 @note The return value is an object of class \code{plotluck_multi}. This
  class does not have any functionality; its sole purpose is to make this
  function work in the same way as \code{ggplot} and \code{plotluck}, namely,
  do the actual drawing if and only if the return value is not assigned.
}

\section{Debugging}{
 With the option \code{verbose=TRUE} turned on, the function
 will print out information about the chosen and applicable plot types, ordering,
 log scaling, etc.
}

\section{Column name matching}{
 Variable names can be abbreviated if they match
 a column name uniquely by prefix.
}

\section{Remarks on supported plot types}{
 By default, \code{plotluck}
 uses violin and density plots in place of the more traditional box-and-whisker
 plots and histograms; these modern graph types convey the shape of a
 distribution better. In the former case, summary statistics like mean and
 quantiles are less useful if the distribution is not unimodal; a wrong
 choice of the number of bins of a histogram can create misleading artifacts.

 Following Cleveland's advice, factors are plotted on the y-axis to make labels
 most readable and compact at the same time. This direction can be controlled
 using option \code{prefer.factors.vert}.

 Due to their well-documented problematic aspects, pie charts and stacked bar
 graphs are not supported.

 With real-world data (as opposed to smooth mathematical functions),
 three-dimensional scatter, surface, or contour plots can often be hard to
 read if the shape of the distribution is not suitable, data coverage is
 uneven, or if the perspective is not carefully chosen depending on the data.
 Since they usually require manual tweaking, we have refrained from
 incorporating them.
}

\section{Remarks on the use of options}{
 For completeness, we have included the
 description of option parameters in the current help page. However, the
 tenet of this function is to be usable "out-of-the-box", with no or very
 little manual tweaking required. If you find yourself needing to change
 option values repeatedly or find the presets to be suboptimal, please
 contact the author.
}

\section{Limitations}{
 \code{plotluck} is designed for generic out-of-the-box
 plotting, and not suitable to produce more specialized types of plots that
 arise in specific application domains (e.g., association, stem-and-leaf,
 star plots, geographic maps, etc). It is restricted to at most three variables.
 Parallel plots with variables on different scales (such as time
 series of multiple related signals) are not supported.
}

\examples{
# Single-variable density
data(diamonds, package='ggplot2')
plotluck(diamonds, price~1)
invisible(readline(prompt="Press [enter] to continue"))

# Violin plot
data(iris)
plotluck(iris, Species~Petal.Length)
invisible(readline(prompt="Press [enter] to continue"))

# Scatter plot
data(mpg, package='ggplot2')
plotluck(mpg, cty~model)
invisible(readline(prompt="Press [enter] to continue"))

# Spine plot
data(Titanic)
plotluck(as.data.frame(Titanic), Survived~Class+Sex, weights=Freq)
invisible(readline(prompt="Press [enter] to continue"))

# Facetting
data(msleep, package='ggplot2')
plotluck(msleep, sleep_total~bodywt|vore)
invisible(readline(prompt="Press [enter] to continue"))

# Heat map
plotluck(diamonds, price~cut+color)

\donttest{
# Multi plots
# All 1D distributions
plotluck(iris, .~1)

# 2D dependencies with one fixed variable on vertical axis
plotluck(iris, Species~.)
}
# See also tests/testthat/test_plotluck.R for more examples!

}
\seealso{
\code{\link{plotluck.options}}, \code{\link{sample.plotluck}}, \code{\link[ggplot2]{ggplot}}
}
\concept{automation}
\concept{bar plot}
\concept{box plot}
\concept{density plot}
\concept{exploratory data analysis}
\concept{ggplot}
\concept{ggplot2}
\concept{heat map}
\concept{hexbin}
\concept{histogram}
\concept{mosaic plot}
\concept{plotting}
\concept{scatter plot}
\concept{spine plot}
\concept{violin plot}
\concept{visualization}
\keyword{aplot}
\keyword{dplot}
\keyword{hplot}
