% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/polywog.r
\name{polywog}
\alias{polywog}
\title{Polynomial regression with oracle variable selection}
\usage{
polywog(
  formula,
  data,
  subset,
  weights,
  na.action,
  degree = 3,
  family = c("gaussian", "binomial"),
  method = c("alasso", "scad"),
  penwt.method = c("lm", "glm"),
  unpenalized = character(0),
  .parallel = FALSE,
  boot = 0,
  control.boot = control.bp(.parallel = .parallel),
  lambda = NULL,
  nlambda = 100,
  lambda.min.ratio = 1e-04,
  nfolds = 10,
  foldid = NULL,
  thresh = ifelse(method == "alasso", 1e-07, 0.001),
  maxit = ifelse(method == "alasso", 1e+05, 5000),
  model = TRUE,
  X = FALSE,
  y = FALSE
)
}
\arguments{
\item{formula}{model formula specifying the response and input
variables.  See "Details" for more information.}

\item{data}{a data frame, list or environment containing the variables
specified in the model formula.}

\item{subset}{an optional vector specifying a subset of observations to be
used in fitting.}

\item{weights}{an optional vector specifying weights for each observation to
be used in fitting.}

\item{na.action}{a function specifying what to do with observations
containing \code{NA}s (default \code{\link{na.omit}}).}

\item{degree}{integer specifying the degree of the polynomial expansion of
the input variables.}

\item{family}{\code{"gaussian"} (default) or \code{"binomial"} for logistic
regression (binary response only).}

\item{method}{variable selection method: \code{"alasso"} (default) for
adaptive LASSO or \code{"scad"} for SCAD.  You can also select \code{method
= "none"} to return the model matrix and other information without fitting.}

\item{penwt.method}{estimator for obtaining first-stage estimates in
logistic models when \code{method = "alasso"}: \code{"lm"} (default) for a
linear probability model, \code{"glm"} for logistic regression.}

\item{unpenalized}{names of model terms to be exempt from the adaptive
penalty (only available when \code{method = "alasso"}).}

\item{.parallel}{logical: whether to perform k-fold cross-validation in
parallel (only available when \code{method = "alasso"}).  See "Details"
below for more information on parallel computation.}

\item{boot}{number of bootstrap iterations (0 for no bootstrapping).}

\item{control.boot}{list of arguments to be passed to
\code{\link{bootPolywog}} when bootstrapping; see \code{\link{control.bp}}.}

\item{lambda}{a vector of values from which the penalty factor is to be
selected via k-fold cross-validation.  \code{lambda} is left unspecified
by default, in which case a sequence of values is generated automatically,
controlled by the \code{nlambda} and \code{lambda.min.ratio} arguments.
Naturally, k-fold cross-validation is skipped if \code{lambda} contains
exactly one value.}

\item{nlambda}{number of values of the penalty factor to examine via
cross-validation if \code{lambda} is not specified in advance; see
"Details".}

\item{lambda.min.ratio}{ratio of the lowest value to the highest in the
generated sequence of values of the penalty factor if \code{lambda} is
not specified; see "Details".}

\item{nfolds}{number of folds to use in cross-validation to select the
penalization factor.}

\item{foldid}{optional vector manually assigning fold numbers to each
observation used for fitting (only available when \code{method =
"alasso"}).}

\item{thresh}{convergence threshold, passed as the \code{thresh} argument
to \code{\link[glmnet]{glmnet}} when \code{method = "alasso"} and as the
\code{eps} argument to \code{\link[ncvreg]{ncvreg}} when \code{method = "scad"}.}

\item{maxit}{maximum number of iterations to allow in adaptive LASSO or
SCAD fitting.}

\item{model}{logical: whether to include the model frame in the returned
object.}

\item{X}{logical: whether to include the raw design matrix (i.e., the
matrix of input variables prior to taking their polynomial expansion) in
the returned object.}

\item{y}{logical: whether to include the response variable in the returned
object.}
}
\value{
An object of class \code{"polywog"}, a list containing: \describe{
  \item{\code{coefficients}}{the estimated coefficients.}
  \item{\code{lambda}}{value of the penalty factor \eqn{\lambda} used to
fit the final model.}
  \item{\code{lambda.cv}}{a list containing the results of the
cross-validation procedure used to select the penalty factor: \describe{
    \item{\code{lambda}}{values of the penalty factor tested in
cross-validation.}
    \item{\code{cvError}}{out-of-fold prediction error corresponding to
each value of \code{lambda}.}
    \item{\code{lambdaMin}}{value of \code{lambda} with the minimal
cross-validation error.}
    \item{\code{errorMin}}{minimized value of the cross-validation error.}
  }}
  \item{\code{fitted.values}}{the fitted mean values for each observation
used in fitting.}
  \item{\code{lmcoef}}{coefficients from an unpenalized least-squares
regression of the response variable on the polynomial expansion of the
input variables.}
  \item{\code{penwt}}{adaptive weight given to each term in the LASSO
penalty (\code{NULL} for models fit via SCAD).}
  \item{\code{formula}}{model formula, as a \code{\link[Formula:Formula]{Formula}} object.}
  \item{\code{degree}}{degree of the polynomial basis expansion.}
  \item{\code{family}}{model family, \code{"gaussian"} or
\code{"binomial"}.}
  \item{\code{weights}}{observation weights if specified.}
  \item{\code{method}}{the specified regularization method.}
  \item{\code{penwt.method}}{the specified method for calculating
the adaptive LASSO weights (\code{NULL} for models fit via SCAD).}
  \item{\code{unpenalized}}{logical vector indicating which terms were not
included in the LASSO penalty.}
  \item{\code{thresh}}{convergence threshold used in fitting.}
  \item{\code{maxit}}{iteration limit used in fitting.}
  \item{\code{terms}}{the \code{\link{terms}} object used to construct the
model frame.}
  \item{\code{polyTerms}}{a matrix indicating the power of each raw input
term (columns) in each term of the polynomial expansion used in fitting
(rows).}
  \item{\code{nobs}}{the number of observations used to fit the model.}
  \item{\code{na.action}}{information on how \code{NA} values in the input
data were handled.}
  \item{\code{xlevels}}{levels of factor variables used in fitting.}
  \item{\code{varNames}}{names of the raw input variables included in the
model formula.}
  \item{\code{call}}{the original function call.}
  \item{\code{model}}{if \code{model = TRUE}, the model frame used in
fitting; otherwise \code{NULL}.}
  \item{\code{X}}{if \code{X = TRUE}, the raw model matrix (i.e., prior to
taking the polynomial expansion); otherwise \code{NULL}.  For calculating
the expanded model matrix, see \code{\link{model.matrix.polywog}}.}
  \item{\code{y}}{if \code{y = TRUE}, the response variable used in
fitting; otherwise \code{NULL}.}
  \item{\code{boot.matrix}}{if \code{boot > 0}, a sparse matrix of class
\code{"\link[Matrix:dgCMatrix-class]{dgCMatrix-class}"} where each column is the estimate from a
bootstrap replicate.  See \code{\link{bootPolywog}} for more information
on bootstrapping.}
}
}
\description{
Fits a regression model using a polynomial basis expansion of the input
variables, with penalization via the adaptive LASSO or SCAD to provide
oracle variable selection.
}
\details{
The design matrix for the regression is a polynomial basis expansion of the
matrix of raw input variables.  This includes all powers and interactions of
the input variables up to the specified \code{degree}.  For example, the
following terms will be included in \code{polywog(y ~ x1 + x2, degree = 3,
...)}:
\itemize{
  \item terms of degree 0: intercept
  \item terms of degree 1: \code{x1}, \code{x2}
  \item terms of degree 2: \code{x1^2}, \code{x2^2}, \code{x1*x2}
  \item terms of degree 3: \code{x1^3}, \code{x2^3}, \code{x1*x2^2},
\code{x1^2*x2}
}
To exclude certain terms from the basis expansion, use a model formula like
\code{y ~ x1 + x2 | z1 + z2}.  Only the degree 1 terms of \code{z1} and
\code{z2} will be included.

It is possible that the "raw" basis expansion will be rank-deficient, such
as if there are binary input variables (in which case \eqn{x_i = x_i^n} for
all \eqn{n > 0}).  The procedure detects collinearity via \code{\link{qr}} and
removes extraneous columns before fitting.

For both the adaptive LASSO and SCAD, the penalization factor \eqn{\lambda}
is chosen by k-fold cross-validation.  The selected value minimizes the
average mean squared error of out-of-sample fits.  (To select both
\eqn{\lambda} and the polynomial degree simultaneously via cross-validation,
see \code{\link{cv.polywog}}.)

The cross-validation process may be run in parallel via
\code{\link[foreach]{foreach}} by registering an appropriate backend and specifying
\code{.parallel = TRUE}.  The appropriate backend is system-specific; see
\code{\link[foreach]{foreach}} for information on selecting and registering a
backend.  The bootstrap iterations may also be run in parallel by
specifying \code{control.boot = control.bp(.parallel = TRUE)}.
}
\examples{
## Using occupational prestige data
data(Prestige, package = "carData")
Prestige <- transform(Prestige, income = income / 1000)

## Fit a polywog model with bootstrap iterations
## (note: using low convergence threshold to shorten computation time of the
## example, *not* recommended in practice!)
set.seed(22)
fit1 <- polywog(prestige ~ education + income + type,
                data = Prestige,
                degree = 2,
                boot = 5,
                thresh = 1e-4)

## Basic information
print(fit1)
summary(fit1)

## See how fitted values change with education holding all else fixed
predVals(fit1, "education", n = 10)

## Plot univariate relationships
plot(fit1)

## Use SCAD instead of adaptive LASSO
fit2 <- update(fit1, method = "scad", thresh = 1e-3)
cbind(coef(fit1), coef(fit2))
}
\references{
Brenton Kenkel and Curtis S. Signorino.  2012.  "A Method for
Flexible Functional Form Estimation: Bootstrapped Basis Regression with
Variable Selection."  Typescript, University of Rochester.
}
\seealso{
To estimate variation via the bootstrap, see
\code{\link{bootPolywog}}.  To generate fitted values, see
\code{\link{predVals}} (and the underlying method
\code{\link{predict.polywog}}).  For plots, see \code{\link{plot.polywog}}.
The polynomial degree may be selected via cross-validation using
\code{\link{cv.polywog}}.

Adaptive LASSO estimates are provided via \code{\link[glmnet]{glmnet}} and
\code{\link[glmnet]{cv.glmnet}} from the \pkg{glmnet} package.  SCAD estimates are
via \code{\link[ncvreg]{ncvreg}} and \code{\link[ncvreg]{cv.ncvreg}} in the \pkg{ncvreg}
package.
}
\author{
Brenton Kenkel and Curtis S. Signorino
}
