% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/splinecopula.R
\name{spline.copula}
\alias{spline.copula}
\title{B-spline copula using the five M-spline basis functions}
\usage{
spline.copula(
  u,
  v,
  R = "PE1",
  mat = FALSE,
  density = FALSE,
  Kendall = FALSE,
  Spearman = FALSE
)
}
\arguments{
\item{u}{A numeric vector of values in \eqn{[0, 1]} (uniform marginals for the
first variable).}

\item{v}{A numeric vector of values in \eqn{[0, 1]} (uniform marginals for the
second variable).}

\item{R}{A 5×5 non-negative coefficient matrix defining the copula structure.
The matrix must satisfy the following conditions:
\itemize{
\item All entries must be non-negative (\eqn{R_{kl} \ge 0}).
\item The sum of all entries must be exactly 1.
\item The row sums and column sums must equal
\code{c(1/8, 1/4, 1/4, 1/4, 1/8)} (in order).
}
These conditions ensure that the resulting function is a valid copula density.
You may also specify one of the built-in presets:
\code{"PE1"}, \code{"PE2"}, \code{"PE3"},
\code{"PN1"}, \code{"PN2"}, \code{"PN3"},
\code{"I"},
\code{"NE1"}, \code{"NE2"}, \code{"NE3"},
\code{"NN1"}, \code{"NN2"}, \code{"NN3"}.
Default is \code{"PE1"}.}

\item{mat}{Logical; if \code{TRUE}, returns the full matrix (outer product) of
copula evaluations; otherwise returns only the diagonal values, i.e.,
\eqn{C(u_i, v_i)} or \eqn{c(u_i, v_i)} for \eqn{i = 1, \ldots, n}.
Default is \code{FALSE}.}

\item{density}{Logical; if \code{TRUE}, evaluates the copula density; if
\code{FALSE}, evaluates the copula function. Default is \code{FALSE}.}

\item{Kendall}{Logical; if \code{TRUE}, returns Kendall’s tau in addition to
copula values. Default is \code{FALSE}.}

\item{Spearman}{Logical; if \code{TRUE}, returns Spearman’s rho in addition to
copula values. Default is \code{FALSE}.}
}
\value{
If both \code{Kendall = FALSE} and \code{Spearman = FALSE} (default), returns:
\itemize{
\item A numeric vector of length \code{length(u)} if \code{mat = FALSE}.
\item A numeric matrix of size \code{length(u)} \eqn{\times} \code{length(v)}
if \code{mat = TRUE}.
}

If \code{Kendall = TRUE} or \code{Spearman = TRUE}, returns a list containing:
\itemize{
\item \code{value}: A numeric vector or matrix representing the evaluated
copula function or copula density.
\item \code{Kendall_tau}: (Optional) Kendall’s tau, included only if
\code{Kendall = TRUE}.
\item \code{Spearman_rho}: (Optional) Spearman’s rho, included only if
\code{Spearman = TRUE}.
}
}
\description{
\code{spline.copula} computes the B-spline copula (or its density) based on the
five-parameter M-spline basis functions. This copula is a specific instance of
the B-spline copula family and can be implemented using matrix-based operations
with \code{M.spline} and \code{I.spline} from the \pkg{joint.Cox} package.
}
\details{
If \code{density = TRUE}, the function computes the copula density \eqn{c(u, v)};
otherwise, it returns the copula distribution function \eqn{C(u, v)}.
If \code{density = FALSE}, it returns the copula function. The implementation
uses five M-spline or I-spline basis functions defined on \eqn{[0,1]}. The
coefficient matrix is fixed internally but can be modified in advanced use.
}
\examples{
N <- 50
u <- v <- seq(from = 0, to = 1, length.out = N)
U <- rep(u, N)
V <- rep(v, each = N)

c.data <- data.frame(
  U = U,
  V = V,
  C = spline.copula(U, V, R = "PE1", density = TRUE)
)

if (requireNamespace("ggplot2", quietly = TRUE)) {
  ggplot2::ggplot(c.data, ggplot2::aes(x = U, y = V)) +
    ggplot2::geom_contour(
      ggplot2::aes(z = C, colour = ggplot2::after_stat(level)),
      bins = 25
    ) +
    ggplot2::xlab("u") + ggplot2::ylab("v")
}
}
\seealso{
\code{\link[joint.Cox]{M.spline}}, \code{\link[joint.Cox]{I.spline}}
}
