package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Healthcheck}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableHealthcheck.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableHealthcheck implements Healthcheck {
  private final @Nullable List<String> test;
  private final @Nullable Long interval;
  private final @Nullable Long timeout;
  private final @Nullable Integer retries;
  private final @Nullable Long startPeriod;

  private ImmutableHealthcheck(
      @Nullable List<String> test,
      @Nullable Long interval,
      @Nullable Long timeout,
      @Nullable Integer retries,
      @Nullable Long startPeriod) {
    this.test = test;
    this.interval = interval;
    this.timeout = timeout;
    this.retries = retries;
    this.startPeriod = startPeriod;
  }

  /**
   * @return The value of the {@code test} attribute
   */
  @JsonProperty("Test")
  @Override
  public @Nullable List<String> test() {
    return test;
  }

  /**
   * @return The value of the {@code interval} attribute
   */
  @JsonProperty("Interval")
  @Override
  public @Nullable Long interval() {
    return interval;
  }

  /**
   * @return The value of the {@code timeout} attribute
   */
  @JsonProperty("Timeout")
  @Override
  public @Nullable Long timeout() {
    return timeout;
  }

  /**
   * @return The value of the {@code retries} attribute
   */
  @JsonProperty("Retries")
  @Override
  public @Nullable Integer retries() {
    return retries;
  }

  /**
   * @return The value of the {@code startPeriod} attribute
   */
  @JsonProperty("StartPeriod")
  @Override
  public @Nullable Long startPeriod() {
    return startPeriod;
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Healthcheck#test() test}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableHealthcheck withTest(@Nullable String... elements) {
    if (elements == null) {
      return new ImmutableHealthcheck(null, this.interval, this.timeout, this.retries, this.startPeriod);
    }
    @Nullable List<String> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableHealthcheck(newValue, this.interval, this.timeout, this.retries, this.startPeriod);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Healthcheck#test() test}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of test elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableHealthcheck withTest(@Nullable Iterable<String> elements) {
    if (this.test == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableHealthcheck(newValue, this.interval, this.timeout, this.retries, this.startPeriod);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Healthcheck#interval() interval} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for interval (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableHealthcheck withInterval(@Nullable Long value) {
    if (Objects.equals(this.interval, value)) return this;
    return new ImmutableHealthcheck(this.test, value, this.timeout, this.retries, this.startPeriod);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Healthcheck#timeout() timeout} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for timeout (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableHealthcheck withTimeout(@Nullable Long value) {
    if (Objects.equals(this.timeout, value)) return this;
    return new ImmutableHealthcheck(this.test, this.interval, value, this.retries, this.startPeriod);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Healthcheck#retries() retries} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for retries (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableHealthcheck withRetries(@Nullable Integer value) {
    if (Objects.equals(this.retries, value)) return this;
    return new ImmutableHealthcheck(this.test, this.interval, this.timeout, value, this.startPeriod);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Healthcheck#startPeriod() startPeriod} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for startPeriod (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableHealthcheck withStartPeriod(@Nullable Long value) {
    if (Objects.equals(this.startPeriod, value)) return this;
    return new ImmutableHealthcheck(this.test, this.interval, this.timeout, this.retries, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableHealthcheck} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableHealthcheck
        && equalTo(0, (ImmutableHealthcheck) another);
  }

  private boolean equalTo(int synthetic, ImmutableHealthcheck another) {
    return Objects.equals(test, another.test)
        && Objects.equals(interval, another.interval)
        && Objects.equals(timeout, another.timeout)
        && Objects.equals(retries, another.retries)
        && Objects.equals(startPeriod, another.startPeriod);
  }

  /**
   * Computes a hash code from attributes: {@code test}, {@code interval}, {@code timeout}, {@code retries}, {@code startPeriod}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(test);
    h += (h << 5) + Objects.hashCode(interval);
    h += (h << 5) + Objects.hashCode(timeout);
    h += (h << 5) + Objects.hashCode(retries);
    h += (h << 5) + Objects.hashCode(startPeriod);
    return h;
  }

  /**
   * Prints the immutable value {@code Healthcheck} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Healthcheck{"
        + "test=" + test
        + ", interval=" + interval
        + ", timeout=" + timeout
        + ", retries=" + retries
        + ", startPeriod=" + startPeriod
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Healthcheck} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Healthcheck instance
   */
  public static ImmutableHealthcheck copyOf(Healthcheck instance) {
    if (instance instanceof ImmutableHealthcheck) {
      return (ImmutableHealthcheck) instance;
    }
    return ImmutableHealthcheck.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableHealthcheck ImmutableHealthcheck}.
   * <pre>
   * ImmutableHealthcheck.builder()
   *    .test(List&amp;lt;String&amp;gt; | null) // nullable {@link Healthcheck#test() test}
   *    .interval(Long | null) // nullable {@link Healthcheck#interval() interval}
   *    .timeout(Long | null) // nullable {@link Healthcheck#timeout() timeout}
   *    .retries(Integer | null) // nullable {@link Healthcheck#retries() retries}
   *    .startPeriod(Long | null) // nullable {@link Healthcheck#startPeriod() startPeriod}
   *    .build();
   * </pre>
   * @return A new ImmutableHealthcheck builder
   */
  public static ImmutableHealthcheck.Builder builder() {
    return new ImmutableHealthcheck.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableHealthcheck ImmutableHealthcheck}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements Healthcheck.Builder {
    private List<String> test = null;
    private Long interval;
    private Long timeout;
    private Integer retries;
    private Long startPeriod;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Healthcheck} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Healthcheck instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable List<String> testValue = instance.test();
      if (testValue != null) {
        addAllTest(testValue);
      }
      @Nullable Long intervalValue = instance.interval();
      if (intervalValue != null) {
        interval(intervalValue);
      }
      @Nullable Long timeoutValue = instance.timeout();
      if (timeoutValue != null) {
        timeout(timeoutValue);
      }
      @Nullable Integer retriesValue = instance.retries();
      if (retriesValue != null) {
        retries(retriesValue);
      }
      @Nullable Long startPeriodValue = instance.startPeriod();
      if (startPeriodValue != null) {
        startPeriod(startPeriodValue);
      }
      return this;
    }

    /**
     * Adds one element to {@link Healthcheck#test() test} list.
     * @param element A test element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder test(String element) {
      if (this.test == null) {
        this.test = new ArrayList<String>();
      }
      this.test.add(Objects.requireNonNull(element, "test element"));
      return this;
    }

    /**
     * Adds elements to {@link Healthcheck#test() test} list.
     * @param elements An array of test elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder test(String... elements) {
      if (this.test == null) {
        this.test = new ArrayList<String>();
      }
      for (String element : elements) {
        this.test.add(Objects.requireNonNull(element, "test element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Healthcheck#test() test} list.
     * @param elements An iterable of test elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Test")
    public final Builder test(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.test = null;
        return this;
      }
      this.test = new ArrayList<String>();
      return addAllTest(elements);
    }

    /**
     * Adds elements to {@link Healthcheck#test() test} list.
     * @param elements An iterable of test elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllTest(Iterable<String> elements) {
      Objects.requireNonNull(elements, "test element");
      if (this.test == null) {
        this.test = new ArrayList<String>();
      }
      for (String element : elements) {
        this.test.add(Objects.requireNonNull(element, "test element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Healthcheck#interval() interval} attribute.
     * @param interval The value for interval (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Interval")
    public final Builder interval(@Nullable Long interval) {
      this.interval = interval;
      return this;
    }

    /**
     * Initializes the value for the {@link Healthcheck#timeout() timeout} attribute.
     * @param timeout The value for timeout (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Timeout")
    public final Builder timeout(@Nullable Long timeout) {
      this.timeout = timeout;
      return this;
    }

    /**
     * Initializes the value for the {@link Healthcheck#retries() retries} attribute.
     * @param retries The value for retries (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Retries")
    public final Builder retries(@Nullable Integer retries) {
      this.retries = retries;
      return this;
    }

    /**
     * Initializes the value for the {@link Healthcheck#startPeriod() startPeriod} attribute.
     * @param startPeriod The value for startPeriod (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("StartPeriod")
    public final Builder startPeriod(@Nullable Long startPeriod) {
      this.startPeriod = startPeriod;
      return this;
    }

    /**
     * Builds a new {@link ImmutableHealthcheck ImmutableHealthcheck}.
     * @return An immutable instance of Healthcheck
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableHealthcheck build() {
      return new ImmutableHealthcheck(
          test == null ? null : createUnmodifiableList(true, test),
          interval,
          timeout,
          retries,
          startPeriod);
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
