package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ManagerStatus}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableManagerStatus.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableManagerStatus implements ManagerStatus {
  private final @Nullable Boolean leader;
  private final String reachability;
  private final String addr;

  private ImmutableManagerStatus(
      @Nullable Boolean leader,
      String reachability,
      String addr) {
    this.leader = leader;
    this.reachability = reachability;
    this.addr = addr;
  }

  /**
   * @return The value of the {@code leader} attribute
   */
  @JsonProperty("Leader")
  @Override
  public @Nullable Boolean leader() {
    return leader;
  }

  /**
   * @return The value of the {@code reachability} attribute
   */
  @JsonProperty("Reachability")
  @Override
  public String reachability() {
    return reachability;
  }

  /**
   * @return The value of the {@code addr} attribute
   */
  @JsonProperty("Addr")
  @Override
  public String addr() {
    return addr;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ManagerStatus#leader() leader} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for leader (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableManagerStatus withLeader(@Nullable Boolean value) {
    if (Objects.equals(this.leader, value)) return this;
    return new ImmutableManagerStatus(value, this.reachability, this.addr);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ManagerStatus#reachability() reachability} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for reachability
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableManagerStatus withReachability(String value) {
    String newValue = Objects.requireNonNull(value, "reachability");
    if (this.reachability.equals(newValue)) return this;
    return new ImmutableManagerStatus(this.leader, newValue, this.addr);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ManagerStatus#addr() addr} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for addr
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableManagerStatus withAddr(String value) {
    String newValue = Objects.requireNonNull(value, "addr");
    if (this.addr.equals(newValue)) return this;
    return new ImmutableManagerStatus(this.leader, this.reachability, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableManagerStatus} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableManagerStatus
        && equalTo(0, (ImmutableManagerStatus) another);
  }

  private boolean equalTo(int synthetic, ImmutableManagerStatus another) {
    return Objects.equals(leader, another.leader)
        && reachability.equals(another.reachability)
        && addr.equals(another.addr);
  }

  /**
   * Computes a hash code from attributes: {@code leader}, {@code reachability}, {@code addr}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(leader);
    h += (h << 5) + reachability.hashCode();
    h += (h << 5) + addr.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code ManagerStatus} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ManagerStatus{"
        + "leader=" + leader
        + ", reachability=" + reachability
        + ", addr=" + addr
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ManagerStatus} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ManagerStatus instance
   */
  public static ImmutableManagerStatus copyOf(ManagerStatus instance) {
    if (instance instanceof ImmutableManagerStatus) {
      return (ImmutableManagerStatus) instance;
    }
    return ImmutableManagerStatus.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableManagerStatus ImmutableManagerStatus}.
   * <pre>
   * ImmutableManagerStatus.builder()
   *    .leader(Boolean | null) // nullable {@link ManagerStatus#leader() leader}
   *    .reachability(String) // required {@link ManagerStatus#reachability() reachability}
   *    .addr(String) // required {@link ManagerStatus#addr() addr}
   *    .build();
   * </pre>
   * @return A new ImmutableManagerStatus builder
   */
  public static ImmutableManagerStatus.Builder builder() {
    return new ImmutableManagerStatus.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableManagerStatus ImmutableManagerStatus}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_REACHABILITY = 0x1L;
    private static final long INIT_BIT_ADDR = 0x2L;
    private long initBits = 0x3L;

    private Boolean leader;
    private String reachability;
    private String addr;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ManagerStatus} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ManagerStatus instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable Boolean leaderValue = instance.leader();
      if (leaderValue != null) {
        leader(leaderValue);
      }
      this.reachability(instance.reachability());
      this.addr(instance.addr());
      return this;
    }

    /**
     * Initializes the value for the {@link ManagerStatus#leader() leader} attribute.
     * @param leader The value for leader (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Leader")
    public final Builder leader(@Nullable Boolean leader) {
      this.leader = leader;
      return this;
    }

    /**
     * Initializes the value for the {@link ManagerStatus#reachability() reachability} attribute.
     * @param reachability The value for reachability 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Reachability")
    public final Builder reachability(String reachability) {
      this.reachability = Objects.requireNonNull(reachability, "reachability");
      initBits &= ~INIT_BIT_REACHABILITY;
      return this;
    }

    /**
     * Initializes the value for the {@link ManagerStatus#addr() addr} attribute.
     * @param addr The value for addr 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Addr")
    public final Builder addr(String addr) {
      this.addr = Objects.requireNonNull(addr, "addr");
      initBits &= ~INIT_BIT_ADDR;
      return this;
    }

    /**
     * Builds a new {@link ImmutableManagerStatus ImmutableManagerStatus}.
     * @return An immutable instance of ManagerStatus
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableManagerStatus build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableManagerStatus(leader, reachability, addr);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_REACHABILITY) != 0) attributes.add("reachability");
      if ((initBits & INIT_BIT_ADDR) != 0) attributes.add("addr");
      return "Cannot build ManagerStatus, some of required attributes are not set " + attributes;
    }
  }
}
