/**
 * Copyright (c) 2010-2015, Balazs Grill, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Balazs Grill - initial API and implementation
 * Peter Lunk - EMFScope support added
 */
package org.eclipse.viatra.query.testing.core.api;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.inject.Injector;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.apache.log4j.Level;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.viatra.query.patternlanguage.emf.specification.SpecificationBuilder;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternModel;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQueryGroup;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.viatra.query.runtime.matchers.backend.IQueryBackendFactory;
import org.eclipse.viatra.query.runtime.matchers.backend.QueryEvaluationHint;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PQuery;
import org.eclipse.viatra.query.runtime.registry.IDefaultRegistryView;
import org.eclipse.viatra.query.runtime.registry.QuerySpecificationRegistry;
import org.eclipse.viatra.query.testing.core.IMatchSetModelProvider;
import org.eclipse.viatra.query.testing.core.InitializedSnapshotMatchSetModelProvider;
import org.eclipse.viatra.query.testing.core.ViatraQueryTestCase;
import org.eclipse.viatra.query.testing.core.XmiModelUtil;
import org.eclipse.viatra.query.testing.core.api.IPatternExecutionAnalyzer;
import org.eclipse.viatra.query.testing.core.api.JavaObjectAccess;
import org.eclipse.viatra.query.testing.core.api.MatchRecordEquivalence;
import org.eclipse.viatra.query.testing.core.internal.AnalyzedPatternBasedMatchSetModelProvider;
import org.eclipse.viatra.query.testing.core.internal.DefaultMatchRecordEquivalence;
import org.eclipse.viatra.query.testing.snapshot.QuerySnapshot;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This class defines an API to easily construct test cases. The base conception is to provide
 * a set of match set sources (from a snapshot or from the execution of a pattern) and make assertions
 * on them being equal.
 */
@SuppressWarnings("all")
public class ViatraQueryTest {
  private final ViatraQueryTestCase testCase;
  
  private final List<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> patterns = new LinkedList<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>();
  
  private Map<String, JavaObjectAccess> accessMap;
  
  private final List<IPatternExecutionAnalyzer> analyzers = new LinkedList<IPatternExecutionAnalyzer>();
  
  private ViatraQueryTest() {
    this(Maps.<String, JavaObjectAccess>newHashMap());
  }
  
  /**
   * Initializes a {@link ViatraQueryTest} with a map containing {@link JavaObjectAccess} objects for
   * serialization and deserialization of plain Java types.
   * 
   * @since 1.6
   */
  private ViatraQueryTest(final Map<String, JavaObjectAccess> accessMap) {
    this.accessMap = accessMap;
    ViatraQueryTestCase _viatraQueryTestCase = new ViatraQueryTestCase(accessMap);
    this.testCase = _viatraQueryTestCase;
  }
  
  /**
   * Test the specified query
   */
  public static <Match extends IPatternMatch> ViatraQueryTest test(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> pattern) {
    return new ViatraQueryTest().and(pattern);
  }
  
  /**
   * Tests every query in the given group.
   */
  public static ViatraQueryTest test(final IQueryGroup patterns) {
    return new ViatraQueryTest().and(patterns);
  }
  
  /**
   * Tests the given query with the given Java object access settings.
   */
  public static <Match extends IPatternMatch> ViatraQueryTest test(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> pattern, final Map<String, JavaObjectAccess> accessMap) {
    return new ViatraQueryTest(accessMap).and(pattern);
  }
  
  /**
   * Tests every query in the given group with the given Java object access settings.
   */
  public static ViatraQueryTest test(final IQueryGroup patterns, final Map<String, JavaObjectAccess> accessMap) {
    return new ViatraQueryTest(accessMap).and(patterns);
  }
  
  /**
   * Tests every query in the given group.
   */
  public static ViatraQueryTest test() {
    return new ViatraQueryTest();
  }
  
  /**
   * Test the query with the given fully qualified name.
   */
  public static <Match extends IPatternMatch> ViatraQueryTest test(final String pattern) {
    return ViatraQueryTest.test().and(pattern);
  }
  
  /**
   * Tests every query in the given group.
   */
  public ViatraQueryTest and(final IQueryGroup patterns) {
    ViatraQueryTest _xblockexpression = null;
    {
      final Consumer<IQuerySpecification<?>> _function = (IQuerySpecification<?> it) -> {
        this.patterns.add(((IQuerySpecification<ViatraQueryMatcher<IPatternMatch>>) it));
      };
      patterns.getSpecifications().forEach(_function);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Test the given query parsed from file
   */
  public boolean and(final URI patternModel, final Injector injector, final String patternName) {
    boolean _xblockexpression = false;
    {
      final ResourceSet resourceSet = XmiModelUtil.prepareXtextResource(injector);
      final Resource resource = resourceSet.getResource(patternModel, true);
      boolean _isEmpty = resource.getContents().isEmpty();
      boolean _not = (!_isEmpty);
      Preconditions.checkState(_not);
      final EObject patternmodel = resource.getContents().get(0);
      Preconditions.checkState((patternmodel instanceof PatternModel));
      final Function1<Pattern, Boolean> _function = (Pattern it) -> {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, patternName));
      };
      final Iterable<Pattern> patterns = IterableExtensions.<Pattern>filter(((PatternModel) patternmodel).getPatterns(), _function);
      int _size = IterableExtensions.size(patterns);
      boolean _equals = (_size == 1);
      Preconditions.checkState(_equals);
      final SpecificationBuilder builder = new SpecificationBuilder();
      _xblockexpression = this.patterns.add(builder.getOrCreateSpecification(((Pattern[])Conversions.unwrapArray(patterns, Pattern.class))[0]));
    }
    return _xblockexpression;
  }
  
  /**
   * Test the specified query
   */
  public ViatraQueryTest and(final IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> pattern) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.patterns.add(pattern);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Test the query with the given fully qualified name.
   */
  public ViatraQueryTest and(final String pattern) {
    ViatraQueryTest _xblockexpression = null;
    {
      final IDefaultRegistryView view = QuerySpecificationRegistry.getInstance().getDefaultView();
      IQuerySpecification<?> _get = view.getEntry(pattern).get();
      _xblockexpression = this.and(((IQuerySpecification<ViatraQueryMatcher<IPatternMatch>>) _get));
    }
    return _xblockexpression;
  }
  
  /**
   * Enables analyzing all referred patterns.
   * 
   * @since 1.6
   */
  public ViatraQueryTest andAllReferredPatterns() {
    ViatraQueryTest _xblockexpression = null;
    {
      final Function1<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>, Iterable<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>> _function = (IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> pattern) -> {
        final Function1<PQuery, Iterable<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>> _function_1 = (PQuery it) -> {
          final Function1<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>, IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> _function_2 = (IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> it_1) -> {
            return it_1;
          };
          final Function1<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>, IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> _final_function_2 = _function_2;
          return IterableExtensions.<IQuerySpecification, IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>map(Iterables.<IQuerySpecification>filter(it.publishedAs(), IQuerySpecification.class), ((Function1<? super IQuerySpecification, ? extends IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>)_final_function_2));
        };
        return Iterables.<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>concat(IterableExtensions.<PQuery, Iterable<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>>map(pattern.getInternalQueryRepresentation().getAllReferredQueries(), _function_1));
      };
      final Set<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> allReferredPatterns = IterableExtensions.<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>toSet(Iterables.<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>concat(ListExtensions.<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>, Iterable<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>>map(this.patterns, _function)));
      Iterables.<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>addAll(this.patterns, allReferredPatterns);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Add Java Object Access elements
   * 
   * @since 1.6
   */
  public ViatraQueryTest withClasses(final Map<String, JavaObjectAccess> accessmap) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.accessMap = accessmap;
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Add an analyzer to pattern executions
   * 
   * @since 1.6
   */
  public ViatraQueryTest analyzeWith(final IPatternExecutionAnalyzer analyzer) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.analyzers.add(analyzer);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Adds a pattern-based match result set evaluated using the given hints.
   * 
   * @since 1.6
   */
  public ViatraQueryTest withPatternMatches(final QueryEvaluationHint hint) {
    return this.with(hint);
  }
  
  /**
   * Adds a pattern-based match result set evaluated using the given hints.
   */
  public ViatraQueryTest with(final QueryEvaluationHint hint) {
    ViatraQueryTest _xblockexpression = null;
    {
      final AnalyzedPatternBasedMatchSetModelProvider modelProvider = new AnalyzedPatternBasedMatchSetModelProvider(hint, this.accessMap, this.analyzers);
      this.testCase.addMatchSetModelProvider(modelProvider);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Adds a pattern-based match result set evaluated using the given query backend.
   */
  public ViatraQueryTest with(final IQueryBackendFactory queryBackendFactory) {
    ViatraQueryTest _xblockexpression = null;
    {
      final QueryEvaluationHint hint = new QueryEvaluationHint(null, queryBackendFactory);
      _xblockexpression = this.with(hint);
    }
    return _xblockexpression;
  }
  
  /**
   * Add match result set loaded from the given snapshots.
   * 
   * @since 1.6
   */
  public ViatraQueryTest withSnapshotMatches(final QuerySnapshot... snapshot) {
    return this.with(snapshot);
  }
  
  /**
   * Add match result set loaded from the given snapshot.
   * 
   * @since 1.5.2
   */
  public ViatraQueryTest with(final QuerySnapshot... snapshot) {
    ViatraQueryTest _xblockexpression = null;
    {
      InitializedSnapshotMatchSetModelProvider _initializedSnapshotMatchSetModelProvider = new InitializedSnapshotMatchSetModelProvider(snapshot);
      this.testCase.addMatchSetModelProvider(_initializedSnapshotMatchSetModelProvider);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Initialize test model using EMF Scope
   * @since 1.5.2
   */
  public ViatraQueryTest on(final EMFScope scope) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.testCase.setScope(scope);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Execute the given operation on the model. This call will also remove every non-incremental result set.
   */
  public <T extends EObject> ViatraQueryTest modify(final Class<T> clazz, final Predicate<T> condition, final Consumer<T> operation) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.testCase.<T>modifyModel(clazz, condition, operation);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Execute all queries and check that the result sets are equal, then return itself to continue the testing. This
   * is separated from assertEquals because JUnit requires test methods to return void, therefore a test shall end with
   * assertEquals and this method shall be used where further actions are intended (e.g. incremental scenarios)
   */
  public ViatraQueryTest assertEqualsThen() {
    ViatraQueryTest _xblockexpression = null;
    {
      this.assertEquals();
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Assert that there were no log events with higher level than given severity during the execution
   */
  public void assertLogSeverityThreshold(final Level severity) {
    this.testCase.assertLogSeverityThreshold(severity);
  }
  
  /**
   * Assert that the highest level of log events occurred during execution equals to the given severity
   */
  public void assertLogSeverity(final Level severity) {
    this.testCase.assertLogSeverity(severity);
  }
  
  /**
   * Assert that there were no log events with higher level than given severity during the execution
   */
  public ViatraQueryTest assertLogSeverityThresholdThen(final Level severity) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.assertLogSeverityThreshold(severity);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Assert that the highest level of log events occurred during execution equals to the given severity
   */
  public ViatraQueryTest assertLogSeverityThen(final Level severity) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.assertLogSeverity(severity);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Assert that the specified {@link IMatchSetModelProvider} instances produce the same match set snapshots, using a user defined equivalence algorithm.
   * 
   * @since 1.6
   */
  public void assertEquals(final MatchRecordEquivalence equivalence) {
    this.assertEquals(Level.INFO, equivalence);
  }
  
  /**
   * Execute all queries and check that the result sets are equal and no error log message was thrown
   */
  public void assertEquals() {
    this.assertEquals(Level.INFO);
  }
  
  /**
   * Execute all queries and check that the result sets are equal with a selected log level threshold, using a user specified equivalence algorithm.
   * 
   * @since 1.6
   */
  public void assertEquals(final Level treshold, final MatchRecordEquivalence equivalence) {
    final Consumer<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> _function = (IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> it) -> {
      this.testCase.<IPatternMatch>assertMatchSetsEqual(((IQuerySpecification<ViatraQueryMatcher<IPatternMatch>>) it), equivalence);
    };
    this.patterns.forEach(_function);
    this.testCase.assertLogSeverityThreshold(treshold);
  }
  
  /**
   * Execute all queries and check that the result sets are equal with a selected log level threshold
   */
  public void assertEquals(final Level treshold) {
    final Consumer<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> _function = (IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> it) -> {
      DefaultMatchRecordEquivalence _defaultMatchRecordEquivalence = new DefaultMatchRecordEquivalence(this.accessMap);
      this.testCase.<IPatternMatch>assertMatchSetsEqual(((IQuerySpecification<ViatraQueryMatcher<IPatternMatch>>) it), _defaultMatchRecordEquivalence);
    };
    this.patterns.forEach(_function);
    this.testCase.assertLogSeverityThreshold(treshold);
  }
  
  /**
   * Make assumptions that each provided engine and snapshot can provide Match result set for each tested patterns
   */
  public ViatraQueryTest assumeInputs() {
    ViatraQueryTest _xblockexpression = null;
    {
      final Consumer<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> _function = (IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> it) -> {
        this.testCase.<IPatternMatch>assumeMatchSetsAreAvailable(((IQuerySpecification<ViatraQueryMatcher<IPatternMatch>>) it));
      };
      this.patterns.forEach(_function);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
}
