/**
 * Copyright (c) 2010-2017, Dénes Harmath, IncQueryLabs
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Dénes Harmath - initial API and implementation
 */
package org.eclipse.viatra.query.testing.core.coverage;

import com.google.common.base.Charsets;
import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.io.Files;
import java.io.File;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import org.eclipse.emf.common.util.URI;
import org.eclipse.viatra.query.runtime.matchers.psystem.PBody;
import org.eclipse.viatra.query.runtime.matchers.psystem.PConstraint;
import org.eclipse.viatra.query.runtime.matchers.psystem.PTraceable;
import org.eclipse.viatra.query.runtime.matchers.psystem.basicdeferred.ExportedParameter;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PQuery;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PVisibility;
import org.eclipse.viatra.query.testing.core.coverage.CoverageAnalyzer;
import org.eclipse.viatra.query.testing.core.coverage.CoverageContext;
import org.eclipse.viatra.query.testing.core.coverage.CoverageInfo;
import org.eclipse.viatra.query.testing.core.coverage.CoverageState;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Utility methods to report the results of a {@link CoverageAnalyzer}.
 * 
 * @since 1.6
 */
@SuppressWarnings("all")
public class CoverageReporter {
  /**
   * Prints coverage report to the standard output.
   */
  public static void reportConsole(final CoverageAnalyzer it) {
    StringConcatenation _builder = new StringConcatenation();
    {
      final Function1<PQuery, String> _function = (PQuery it_1) -> {
        return it_1.getFullyQualifiedName();
      };
      List<PQuery> _sortBy = IterableExtensions.<PQuery, String>sortBy(Iterables.<PQuery>filter(it.getCoverage().getElementCoverage().keySet(), PQuery.class), _function);
      for(final PQuery query : _sortBy) {
        _builder.append("pattern ");
        String _fullyQualifiedName = query.getFullyQualifiedName();
        _builder.append(_fullyQualifiedName);
        _builder.append(" (");
        CoverageState _get = it.getCoverage().getElementCoverage().get(query);
        _builder.append(_get);
        _builder.append(") ");
        CharSequence _formatPercent = CoverageReporter.formatPercent(it.getCoverage().getCoveragePercent(query));
        _builder.append(_formatPercent);
        _builder.newLineIfNotEmpty();
        {
          Set<PBody> _bodies = query.getDisjunctBodies().getBodies();
          boolean _hasElements = false;
          for(final PBody body : _bodies) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(" or ", "");
            }
            _builder.append("{ (");
            CoverageState _get_1 = it.getCoverage().getElementCoverage().get(body);
            _builder.append(_get_1);
            _builder.append(")");
            _builder.newLineIfNotEmpty();
            {
              Set<PConstraint> _constraints = body.getConstraints();
              for(final PConstraint constraint : _constraints) {
                _builder.append("\t");
                _builder.append(constraint, "\t");
                _builder.append(" (");
                CoverageState _get_2 = it.getCoverage().getElementCoverage().get(constraint);
                _builder.append(_get_2, "\t");
                _builder.append(")");
                _builder.newLineIfNotEmpty();
              }
            }
            _builder.append("}");
            _builder.newLine();
          }
        }
      }
    }
    InputOutput.<String>println(_builder.toString());
  }
  
  private static CharSequence formatPercent(final double percent) {
    StringConcatenation _builder = new StringConcatenation();
    String _format = String.format("%1$.2f", Double.valueOf(percent));
    _builder.append(_format);
    _builder.append("%");
    return _builder;
  }
  
  /**
   * Saves coverage report in HTML format to the given {@link File}.
   */
  public static void reportHtml(final CoverageAnalyzer it, final File file) {
    try {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("<html><body>");
      _builder.newLine();
      _builder.append("<h1>Pattern coverage report</h1>");
      _builder.newLine();
      _builder.append("<h2>Pattern & body coverage</h2>");
      _builder.newLine();
      _builder.append("<h3>Overall: ");
      CharSequence _bodyCoverage = CoverageReporter.getBodyCoverage(it.getCoverage(), Iterables.<PBody>filter(it.getCoverage().getElementCoverage().keySet(), PBody.class));
      _builder.append(_bodyCoverage);
      _builder.append("</h3>");
      _builder.newLineIfNotEmpty();
      _builder.append("<ul>");
      _builder.newLine();
      {
        final Function1<PQuery, String> _function = (PQuery it_1) -> {
          return it_1.getFullyQualifiedName();
        };
        List<PQuery> _sortBy = IterableExtensions.<PQuery, String>sortBy(Iterables.<PQuery>filter(it.getCoverage().getElementCoverage().keySet(), PQuery.class), _function);
        for(final PQuery query : _sortBy) {
          _builder.append("            ");
          final Set<PBody> bodies = query.getDisjunctBodies().getBodies();
          _builder.newLineIfNotEmpty();
          _builder.append("<li>");
          _builder.newLine();
          _builder.append("  ");
          _builder.append("<a href=\"#");
          String _fullyQualifiedName = query.getFullyQualifiedName();
          _builder.append(_fullyQualifiedName, "  ");
          _builder.append("\"><span ");
          String _attributes = CoverageReporter.getAttributes(it.getCoverage(), query, bodies);
          _builder.append(_attributes, "  ");
          _builder.append(">");
          _builder.newLineIfNotEmpty();
          _builder.append("  ");
          {
            PVisibility _visibility = query.getVisibility();
            boolean _equals = Objects.equal(_visibility, PVisibility.PRIVATE);
            if (_equals) {
              _builder.append("private");
            }
          }
          _builder.newLineIfNotEmpty();
          _builder.append("  ");
          _builder.append("pattern ");
          String _fullyQualifiedName_1 = query.getFullyQualifiedName();
          _builder.append(_fullyQualifiedName_1, "  ");
          _builder.append("</span></a> ");
          CharSequence _bodyCoverage_1 = CoverageReporter.getBodyCoverage(it.getCoverage(), bodies);
          _builder.append(_bodyCoverage_1, "  ");
          _builder.newLineIfNotEmpty();
          _builder.append("</li>");
          _builder.newLine();
        }
      }
      _builder.append("</ul>");
      _builder.newLine();
      _builder.append("<h2>Detailed coverage</h2>");
      _builder.newLine();
      {
        final Function1<PQuery, String> _function_1 = (PQuery it_1) -> {
          return it_1.getFullyQualifiedName();
        };
        List<PQuery> _sortBy_1 = IterableExtensions.<PQuery, String>sortBy(Iterables.<PQuery>filter(it.getCoverage().getElementCoverage().keySet(), PQuery.class), _function_1);
        for(final PQuery query_1 : _sortBy_1) {
          _builder.append("<h3 id=\"");
          String _fullyQualifiedName_2 = query_1.getFullyQualifiedName();
          _builder.append(_fullyQualifiedName_2);
          _builder.append("\"/>pattern ");
          String _fullyQualifiedName_3 = query_1.getFullyQualifiedName();
          _builder.append(_fullyQualifiedName_3);
          _builder.append("</h3>");
          _builder.newLineIfNotEmpty();
          {
            final Function1<CoverageContext<PTraceable>, Boolean> _function_2 = (CoverageContext<PTraceable> it_1) -> {
              PTraceable _element = it_1.getElement();
              return Boolean.valueOf(Objects.equal(_element, query_1));
            };
            Iterable<CoverageContext<PTraceable>> _filter = IterableExtensions.<CoverageContext<PTraceable>>filter(it.getCoverage().keySet(), _function_2);
            for(final CoverageContext<PTraceable> context : _filter) {
              _builder.append("<p>Model: ");
              Set<URI> _scope = context.getScope();
              _builder.append(_scope);
              _builder.append("</p>");
              _builder.newLineIfNotEmpty();
              _builder.append("<p>");
              _builder.newLine();
              int bodyIndex = 0;
              _builder.newLineIfNotEmpty();
              {
                Set<PBody> _bodies = query_1.getDisjunctBodies().getBodies();
                boolean _hasElements = false;
                for(final PBody body : _bodies) {
                  if (!_hasElements) {
                    _hasElements = true;
                  } else {
                    _builder.appendImmediate(" or ", "             ");
                  }
                  _builder.append("             ");
                  _builder.append(" ");
                  final Function1<PConstraint, Boolean> _function_3 = (PConstraint it_1) -> {
                    return Boolean.valueOf((!(it_1 instanceof ExportedParameter)));
                  };
                  final Iterable<PConstraint> constraints = IterableExtensions.<PConstraint>filter(body.getConstraints(), _function_3);
                  _builder.newLineIfNotEmpty();
                  _builder.append("             ");
                  _builder.append("<div><span ");
                  String _attributes_1 = CoverageReporter.getAttributes(it.getCoverage(), body, constraints, context.getScope());
                  _builder.append(_attributes_1, "             ");
                  _builder.append(">Body #");
                  int _plusPlus = bodyIndex++;
                  _builder.append(_plusPlus, "             ");
                  _builder.append(" {</span>");
                  _builder.newLineIfNotEmpty();
                  {
                    for(final PConstraint constraint : constraints) {
                      _builder.append("             ");
                      _builder.append("<div ");
                      String _attributes_2 = CoverageReporter.getAttributes(it.getCoverage(), constraint, Collections.<PTraceable>unmodifiableList(CollectionLiterals.<PTraceable>newArrayList()), context.getScope());
                      _builder.append(_attributes_2, "             ");
                      _builder.append("\">");
                      _builder.newLineIfNotEmpty();
                      _builder.append("             ");
                      _builder.append(constraint, "             ");
                      _builder.newLineIfNotEmpty();
                      _builder.append("             ");
                      _builder.append("</div>");
                      _builder.newLine();
                    }
                  }
                  _builder.append("             ");
                  _builder.append("}</div>");
                  _builder.newLine();
                }
              }
            }
          }
          _builder.append("</p>");
          _builder.newLine();
        }
      }
      _builder.append("<h2>Legend</h2>");
      _builder.newLine();
      _builder.append("<p>");
      _builder.newLine();
      _builder.append("\t");
      String _legend = CoverageReporter.getLegend(CoverageState.COVERED, true);
      _builder.append(_legend, "\t");
      _builder.newLineIfNotEmpty();
      _builder.append("\t");
      String _legend_1 = CoverageReporter.getLegend(CoverageState.COVERED, false);
      _builder.append(_legend_1, "\t");
      _builder.newLineIfNotEmpty();
      _builder.append("\t");
      String _legend_2 = CoverageReporter.getLegend(CoverageState.NOT_COVERED, false);
      _builder.append(_legend_2, "\t");
      _builder.newLineIfNotEmpty();
      _builder.append("\t");
      String _legend_3 = CoverageReporter.getLegend(CoverageState.NOT_REPRESENTED, true);
      _builder.append(_legend_3, "\t");
      _builder.newLineIfNotEmpty();
      _builder.append("\t");
      String _legend_4 = CoverageReporter.getLegend(CoverageState.NOT_REPRESENTED_UNKNOWN_REASON, true);
      _builder.append(_legend_4, "\t");
      _builder.newLineIfNotEmpty();
      _builder.append("</p>");
      _builder.newLine();
      _builder.append("</body></html>");
      final String content = _builder.toString();
      Files.write(content, file, Charsets.UTF_8);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private static CharSequence getBodyCoverage(final CoverageInfo<?> coverage, final Iterable<PBody> bodies) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _formatPercent = CoverageReporter.formatPercent(coverage.getCoveragePercent(bodies));
    _builder.append(_formatPercent);
    _builder.newLineIfNotEmpty();
    _builder.append("        ");
    _builder.append("(");
    final Function1<PBody, Boolean> _function = (PBody body) -> {
      CoverageState _get = coverage.getElementCoverage().get(body);
      return Boolean.valueOf(Objects.equal(_get, CoverageState.COVERED));
    };
    int _size = IterableExtensions.size(IterableExtensions.<PBody>filter(bodies, _function));
    _builder.append(_size, "        ");
    _builder.append("/");
    int _size_1 = IterableExtensions.size(bodies);
    _builder.append(_size_1, "        ");
    _builder.append(" bodies covered)");
    return _builder;
  }
  
  private static String getAttributes(final CoverageInfo<?> coverage, final PTraceable traceable, final Iterable<? extends PTraceable> children) {
    String _xblockexpression = null;
    {
      CoverageState _elvis = null;
      CoverageState _get = coverage.getElementCoverage().get(traceable);
      if (_get != null) {
        _elvis = _get;
      } else {
        _elvis = CoverageState.NOT_REPRESENTED_UNKNOWN_REASON;
      }
      final CoverageState state = _elvis;
      final Function1<PTraceable, Boolean> _function = (PTraceable it) -> {
        CoverageState _get_1 = coverage.getElementCoverage().get(it);
        return Boolean.valueOf((!Objects.equal(_get_1, CoverageState.NOT_COVERED)));
      };
      final boolean childrenCovered = IterableExtensions.forall(children, _function);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("style=\"");
      String _style = CoverageReporter.getStyle(state, childrenCovered);
      _builder.append(_style);
      _builder.append("\" title=\"");
      String _title = CoverageReporter.getTitle(state, childrenCovered);
      _builder.append(_title);
      _builder.append("\"");
      _xblockexpression = _builder.toString();
    }
    return _xblockexpression;
  }
  
  private static String getAttributes(final CoverageInfo<?> coverage, final PTraceable traceable, final Iterable<? extends PTraceable> children, final Set<URI> scope) {
    String _xblockexpression = null;
    {
      CoverageState _elvis = null;
      CoverageContext<PTraceable> _coverageContext = new CoverageContext<PTraceable>(traceable, scope);
      CoverageState _get = coverage.get(_coverageContext);
      if (_get != null) {
        _elvis = _get;
      } else {
        _elvis = CoverageState.NOT_REPRESENTED_UNKNOWN_REASON;
      }
      final CoverageState state = _elvis;
      final Function1<PTraceable, Boolean> _function = (PTraceable it) -> {
        CoverageContext<PTraceable> _coverageContext_1 = new CoverageContext<PTraceable>(it, scope);
        CoverageState _get_1 = coverage.get(_coverageContext_1);
        return Boolean.valueOf((!Objects.equal(_get_1, CoverageState.NOT_COVERED)));
      };
      final boolean childrenCovered = IterableExtensions.forall(children, _function);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("style=\"");
      String _style = CoverageReporter.getStyle(state, childrenCovered);
      _builder.append(_style);
      _builder.append("\" title=\"");
      String _title = CoverageReporter.getTitle(state, childrenCovered);
      _builder.append(_title);
      _builder.append("\"");
      _xblockexpression = _builder.toString();
    }
    return _xblockexpression;
  }
  
  private static String getLegend(final CoverageState state, final boolean childrenCovered) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<span style=\"");
    String _style = CoverageReporter.getStyle(state, childrenCovered);
    _builder.append(_style);
    _builder.append("\">");
    String _title = CoverageReporter.getTitle(state, childrenCovered);
    _builder.append(_title);
    _builder.append("</span><br/>");
    return _builder.toString();
  }
  
  private static String getTitle(final CoverageState state, final boolean childrenCovered) {
    String _switchResult = null;
    if (state != null) {
      switch (state) {
        case COVERED:
          String _xifexpression = null;
          if (childrenCovered) {
            _xifexpression = "Covered";
          } else {
            _xifexpression = "Partially covered";
          }
          _switchResult = _xifexpression;
          break;
        case NOT_COVERED:
          _switchResult = "Not covered";
          break;
        case NOT_REPRESENTED:
          _switchResult = "Not represented";
          break;
        case NOT_REPRESENTED_UNKNOWN_REASON:
          _switchResult = "Not represented by error - please report";
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  private static String getStyle(final CoverageState state, final boolean childrenCovered) {
    String _switchResult = null;
    if (state != null) {
      switch (state) {
        case COVERED:
          String _xifexpression = null;
          if (childrenCovered) {
            _xifexpression = "background:springgreen";
          } else {
            _xifexpression = "background:yellow";
          }
          _switchResult = _xifexpression;
          break;
        case NOT_COVERED:
          _switchResult = "background:tomato";
          break;
        case NOT_REPRESENTED:
          _switchResult = "color:lightgray";
          break;
        case NOT_REPRESENTED_UNKNOWN_REASON:
          _switchResult = "color:red";
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
}
