/**
 * Copyright (c) 2010-2016, Abel Hegedus, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Abel Hegedus - initial API and implementation
 */
package org.eclipse.viatra.query.tooling.ui.queryregistry;

import com.google.common.base.Objects;
import com.google.common.collect.Maps;
import java.util.Arrays;
import java.util.Map;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistry;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistryEntry;
import org.eclipse.viatra.query.runtime.registry.IRegistryView;
import org.eclipse.viatra.query.runtime.registry.IRegistryViewFactory;
import org.eclipse.viatra.query.runtime.registry.view.AbstractRegistryView;
import org.eclipse.viatra.query.tooling.ui.queryregistry.QueryRegistryTreeEntry;
import org.eclipse.viatra.query.tooling.ui.queryregistry.QueryRegistryTreeInputChange;
import org.eclipse.viatra.query.tooling.ui.queryregistry.QueryRegistryTreePackage;
import org.eclipse.viatra.query.tooling.ui.queryregistry.QueryRegistryTreeSource;
import org.eclipse.viatra.query.tooling.ui.queryregistry.QueryRegistryTreeViewListener;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * @author Abel Hegedus
 */
@SuppressWarnings("all")
public class QueryRegistryTreeInput {
  @Accessors(AccessorType.PUBLIC_GETTER)
  private Map<String, QueryRegistryTreeSource> sources = Maps.<String, QueryRegistryTreeSource>newTreeMap();
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private IQuerySpecificationRegistry registry;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private IRegistryView view;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private QueryRegistryTreeViewListener listener;
  
  public QueryRegistryTreeInput(final IQuerySpecificationRegistry registry) {
    this.registry = registry;
    QueryRegistryTreeViewListener _queryRegistryTreeViewListener = new QueryRegistryTreeViewListener();
    this.listener = _queryRegistryTreeViewListener;
    this.listener.input = this;
    final IRegistryViewFactory _function = new IRegistryViewFactory() {
      @Override
      public IRegistryView createView(final IQuerySpecificationRegistry it) {
        return new AbstractRegistryView(registry, true) {
          @Override
          protected boolean isEntryRelevant(final IQuerySpecificationRegistryEntry entry) {
            return true;
          }
        };
      }
    };
    IRegistryView _createView = registry.createView(_function);
    this.view = _createView;
    Iterable<IQuerySpecificationRegistryEntry> _entries = this.view.getEntries();
    final Procedure1<IQuerySpecificationRegistryEntry> _function_1 = new Procedure1<IQuerySpecificationRegistryEntry>() {
      @Override
      public void apply(final IQuerySpecificationRegistryEntry it) {
        String _sourceIdentifier = it.getSourceIdentifier();
        final QueryRegistryTreeInputChange source = QueryRegistryTreeInput.this.getOrCreateSource(_sourceIdentifier);
        QueryRegistryTreeSource _source = source.getSource();
        String _fullyQualifiedName = it.getFullyQualifiedName();
        String _packageName = QueryRegistryTreeInput.this.getPackageName(_fullyQualifiedName);
        final QueryRegistryTreeInputChange treePackage = QueryRegistryTreeInput.this.getOrCreatePackage(_source, _packageName);
        QueryRegistryTreePackage _pckg = treePackage.getPckg();
        final QueryRegistryTreeEntry entry = new QueryRegistryTreeEntry(_pckg, it);
        QueryRegistryTreePackage _pckg_1 = treePackage.getPckg();
        Map<String, QueryRegistryTreeEntry> _entries = _pckg_1.getEntries();
        String _fullyQualifiedName_1 = it.getFullyQualifiedName();
        _entries.put(_fullyQualifiedName_1, entry);
      }
    };
    IterableExtensions.<IQuerySpecificationRegistryEntry>forEach(_entries, _function_1);
    this.view.addViewListener(this.listener);
  }
  
  public QueryRegistryTreeInputChange addEntryToInput(final IQuerySpecificationRegistryEntry entry) {
    String _sourceIdentifier = entry.getSourceIdentifier();
    final QueryRegistryTreeInputChange source = this.getOrCreateSource(_sourceIdentifier);
    QueryRegistryTreeSource _source = source.getSource();
    String _fullyQualifiedName = entry.getFullyQualifiedName();
    String _packageName = this.getPackageName(_fullyQualifiedName);
    final QueryRegistryTreeInputChange treePackage = this.getOrCreatePackage(_source, _packageName);
    final QueryRegistryTreePackage treePckg = treePackage.getPckg();
    final QueryRegistryTreeEntry treeEntry = new QueryRegistryTreeEntry(treePckg, entry);
    Map<String, QueryRegistryTreeEntry> _entries = treePckg.getEntries();
    final boolean emptyPckg = _entries.isEmpty();
    QueryRegistryTreePackage _pckg = treePackage.getPckg();
    Map<String, QueryRegistryTreeEntry> _entries_1 = _pckg.getEntries();
    String _fullyQualifiedName_1 = entry.getFullyQualifiedName();
    _entries_1.put(_fullyQualifiedName_1, treeEntry);
    boolean _or = false;
    boolean _isPckgAffected = treePackage.isPckgAffected();
    if (_isPckgAffected) {
      _or = true;
    } else {
      _or = emptyPckg;
    }
    boolean _isSourceAffected = source.isSourceAffected();
    QueryRegistryTreeSource _source_1 = source.getSource();
    return new QueryRegistryTreeInputChange(true, treeEntry, _or, treePckg, _isSourceAffected, _source_1);
  }
  
  public QueryRegistryTreeInputChange removeEntry(final IQuerySpecificationRegistryEntry entry) {
    String _sourceIdentifier = entry.getSourceIdentifier();
    final QueryRegistryTreeInputChange sourceDTO = this.getOrCreateSource(_sourceIdentifier);
    final QueryRegistryTreeSource source = sourceDTO.getSource();
    QueryRegistryTreeSource _source = sourceDTO.getSource();
    String _fullyQualifiedName = entry.getFullyQualifiedName();
    String _packageName = this.getPackageName(_fullyQualifiedName);
    final QueryRegistryTreeInputChange treePackageDTO = this.getOrCreatePackage(_source, _packageName);
    final QueryRegistryTreePackage treePckg = treePackageDTO.getPckg();
    Map<String, QueryRegistryTreeEntry> _entries = treePckg.getEntries();
    String _fullyQualifiedName_1 = entry.getFullyQualifiedName();
    final QueryRegistryTreeEntry treeEntry = _entries.remove(_fullyQualifiedName_1);
    Map<String, QueryRegistryTreeEntry> _entries_1 = treePckg.getEntries();
    final boolean emptyPckg = _entries_1.isEmpty();
    if (emptyPckg) {
      Map<String, QueryRegistryTreePackage> _packages = source.getPackages();
      String _packageName_1 = treePckg.getPackageName();
      _packages.remove(_packageName_1);
    }
    Map<String, QueryRegistryTreePackage> _packages_1 = source.getPackages();
    final boolean emptySource = _packages_1.isEmpty();
    if (emptySource) {
      String _sourceIdentifier_1 = source.getSourceIdentifier();
      this.sources.remove(_sourceIdentifier_1);
    }
    boolean _or = false;
    boolean _isPckgAffected = treePackageDTO.isPckgAffected();
    if (_isPckgAffected) {
      _or = true;
    } else {
      _or = emptyPckg;
    }
    boolean _or_1 = false;
    boolean _isSourceAffected = sourceDTO.isSourceAffected();
    if (_isSourceAffected) {
      _or_1 = true;
    } else {
      _or_1 = emptySource;
    }
    return new QueryRegistryTreeInputChange(true, treeEntry, _or, treePckg, _or_1, source);
  }
  
  public QueryRegistryTreeInputChange getOrCreateSource(final String sourceIdentifier) {
    final QueryRegistryTreeSource existingSource = this.sources.get(sourceIdentifier);
    boolean _equals = Objects.equal(existingSource, null);
    if (_equals) {
      final QueryRegistryTreeSource newSource = new QueryRegistryTreeSource(this, sourceIdentifier);
      this.sources.put(sourceIdentifier, newSource);
      return new QueryRegistryTreeInputChange(false, null, false, null, true, newSource);
    } else {
      return new QueryRegistryTreeInputChange(false, null, false, null, false, existingSource);
    }
  }
  
  public QueryRegistryTreeInputChange getOrCreatePackage(final QueryRegistryTreeSource source, final String packageName) {
    Map<String, QueryRegistryTreePackage> _packages = source.getPackages();
    final QueryRegistryTreePackage existingPackage = _packages.get(packageName);
    boolean _equals = Objects.equal(existingPackage, null);
    if (_equals) {
      final QueryRegistryTreePackage newPackage = new QueryRegistryTreePackage(source, packageName);
      Map<String, QueryRegistryTreePackage> _packages_1 = source.getPackages();
      _packages_1.put(packageName, newPackage);
      return new QueryRegistryTreeInputChange(false, null, true, newPackage, false, null);
    } else {
      return new QueryRegistryTreeInputChange(false, null, false, existingPackage, false, null);
    }
  }
  
  public String getPackageName(final String fullyQualifiedName) {
    int _lastIndexOf = fullyQualifiedName.lastIndexOf(".");
    return fullyQualifiedName.substring(0, _lastIndexOf);
  }
  
  protected TreePath _getTreePath(final QueryRegistryTreePackage pckg) {
    QueryRegistryTreeSource _parent = pckg.getParent();
    return new TreePath(new Object[] { _parent, pckg });
  }
  
  protected TreePath _getTreePath(final QueryRegistryTreeEntry entry) {
    QueryRegistryTreePackage _parent = entry.getParent();
    TreePath _treePath = this.getTreePath(_parent);
    return _treePath.createChildPath(entry);
  }
  
  public TreePath getTreePath(final Object entry) {
    if (entry instanceof QueryRegistryTreeEntry) {
      return _getTreePath((QueryRegistryTreeEntry)entry);
    } else if (entry instanceof QueryRegistryTreePackage) {
      return _getTreePath((QueryRegistryTreePackage)entry);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(entry).toString());
    }
  }
  
  @Pure
  public Map<String, QueryRegistryTreeSource> getSources() {
    return this.sources;
  }
  
  @Pure
  public IQuerySpecificationRegistry getRegistry() {
    return this.registry;
  }
  
  @Pure
  public IRegistryView getView() {
    return this.view;
  }
  
  @Pure
  public QueryRegistryTreeViewListener getListener() {
    return this.listener;
  }
}
