/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.notifications;

import static com.google.common.collect.ImmutableList.of;
import static org.eclipse.e4.core.contexts.ContextInjectionFactory.make;
import static org.eclipse.epp.internal.logging.aeri.ide.IDEWorkflow.TOPIC_BASE;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.IDEConstants.BUNDLE_ID;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.Servers.sort;

import java.util.List;
import java.util.Set;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.core.di.extensions.EventTopic;
import org.eclipse.e4.core.services.events.IEventBroker;
import org.eclipse.epp.internal.logging.aeri.ide.ILogEventGroup;
import org.eclipse.epp.internal.logging.aeri.ide.IServerDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages;
import org.eclipse.epp.logging.aeri.core.SystemControl;
import org.eclipse.epp.logging.aeri.core.util.Logs;
import org.eclipse.mylyn.commons.notifications.ui.NotificationsUi;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;

@SuppressWarnings({ "restriction" })
@Singleton
public class MylynNotificationsSupport {

    public static final String CTX_STATE_NOTIFICATION_IN_PROGRESS = BUNDLE_ID + ".di.notification-in-progress"; //$NON-NLS-1$

    /**
     * Internal Messages for opening and closing notifications
     */
    public static final String INTERNAL_SYSTEM_NOTIFICATION_CLOSED = TOPIC_BASE + "/notifications/closed"; //$NON-NLS-1$
    public static final String INTERNAL_NOTIFICATION_OPENED = TOPIC_BASE + "/notifications/opened"; //$NON-NLS-1$

    private Set<Notification> displayed = Sets.newConcurrentHashSet();
    private IEclipseContext context;

    @Inject
    public MylynNotificationsSupport(IEventBroker broker, IEclipseContext context) {
        this.context = context;
    }

    @Inject
    @Optional
    public void notificationClosed(@EventTopic(INTERNAL_SYSTEM_NOTIFICATION_CLOSED) Notification event) {
        displayed.remove(event);
        context.modify(CTX_STATE_NOTIFICATION_IN_PROGRESS, !displayed.isEmpty());
    }

    @Inject
    @Optional
    public void notificationOpened(@EventTopic(INTERNAL_NOTIFICATION_OPENED) Notification event) {
        displayed.add(event);
        context.modify(CTX_STATE_NOTIFICATION_IN_PROGRESS, !displayed.isEmpty());
    }

    public boolean isNotificationInProgress() {
        return !displayed.isEmpty();
    }

    public void showConfigureSystemNotification() {
        testIsNotificationInProgress();
        SetupSystemNotification notification = make(SetupSystemNotification.class, context);
        NotificationsUi.getService().notify(of(notification));
    }

    public void showConfigureNewServersNotification(List<IServerDescriptor> servers) {
        testIsNotificationInProgress();
        SetupNewServersNotification notification = make(SetupNewServersNotification.class, context);
        notification.setServers(sort(servers));
        NotificationsUi.getService().notify(of(notification));
    }

    public void showServerResponsesNotification(ILogEventGroup group) {
        testIsNotificationInProgress();
        IEclipseContext child = context.createChild("endpoint-responses-notification"); //$NON-NLS-1$
        child.set(ILogEventGroup.class, group);
        ServerResponsesNotification notification = make(ServerResponsesNotification.class, child);
        NotificationsUi.getService().notify(of(notification));
    }

    public void showNewEventLoggedNotification(ILogEventGroup group) {
        testIsNotificationInProgress();
        Class<NewEventNotification> clazz = NewEventNotification.class;
        IEclipseContext child = context.createChild("new-log-events-notification"); //$NON-NLS-1$
        child.set(ILogEventGroup.class, group);
        NewEventNotification notification = make(clazz, child);
        NotificationsUi.getService().notify(of(notification));
    }

    private void testIsNotificationInProgress() {
        if (SystemControl.isDebug() && isNotificationInProgress()) {
            Logs.log(LogMessages.DEBUG_NOTIFICATION_IN_PROGRESS, ImmutableList.of(displayed));
        }
    }

}
