/**
 * Copyright (c) 2010-2014, Zoltan Ujhelyi, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Zoltan Ujhelyi - initial API and implementation
 */
package org.eclipse.viatra.query.patternlanguage.typing;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.logging.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.viatra.query.patternlanguage.helper.CorePatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.AggregatedValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.BoolValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.CheckConstraint;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.CompareConstraint;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.CompareFeature;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.EntityType;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Expression;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.FunctionEvaluationValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.JavaType;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.ListValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.NumberValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PathExpressionConstraint;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PathExpressionHead;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PathExpressionTail;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Pattern;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PatternCall;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PatternCompositionConstraint;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.StringValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Type;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.TypeCheckConstraint;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.ValueReference;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Variable;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.VariableReference;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.VariableValue;
import org.eclipse.viatra.query.patternlanguage.typing.ITypeSystem;
import org.eclipse.viatra.query.patternlanguage.typing.TypeInformation;
import org.eclipse.viatra.query.patternlanguage.typing.judgements.ConditionalJudgement;
import org.eclipse.viatra.query.patternlanguage.typing.judgements.ParameterTypeJudgement;
import org.eclipse.viatra.query.patternlanguage.typing.judgements.TypeConformJudgement;
import org.eclipse.viatra.query.patternlanguage.typing.judgements.TypeJudgement;
import org.eclipse.viatra.query.patternlanguage.typing.judgements.XbaseExpressionTypeJudgement;
import org.eclipse.viatra.query.patternlanguage.util.AggregatorUtil;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;
import org.eclipse.viatra.query.runtime.matchers.context.common.JavaTransitiveInstancesKey;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.XNumberLiteral;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.typesystem.IBatchTypeResolver;
import org.eclipse.xtext.xbase.typesystem.computation.NumberLiterals;

/**
 * @author Zoltan Ujhelyi
 * @since 1.3
 */
@SuppressWarnings("all")
public class PatternLanguageTypeRules {
  @Inject
  private ITypeSystem typeSystem;
  
  @Inject
  private IBatchTypeResolver typeResolver;
  
  @Inject
  private NumberLiterals literals;
  
  @Inject
  private Logger logger;
  
  protected void _inferTypes(final Pattern pattern, final TypeInformation information) {
    EList<Variable> _parameters = pattern.getParameters();
    final Procedure1<Variable> _function = new Procedure1<Variable>() {
      @Override
      public void apply(final Variable parameter) {
        Type _type = parameter.getType();
        boolean _isValidType = PatternLanguageTypeRules.this.typeSystem.isValidType(_type);
        if (_isValidType) {
          Type _type_1 = parameter.getType();
          final IInputKey typeKey = PatternLanguageTypeRules.this.typeSystem.extractTypeDescriptor(_type_1);
          information.declareType(parameter, typeKey);
        }
        Set<Variable> _localReferencesOfParameter = CorePatternLanguageHelper.getLocalReferencesOfParameter(parameter);
        final Procedure1<Variable> _function = new Procedure1<Variable>() {
          @Override
          public void apply(final Variable ref) {
            information.provideType(new TypeConformJudgement(ref, parameter) {
              @Override
              public Set<Expression> getDependingExpressions() {
                return Collections.<Expression>unmodifiableSet(CollectionLiterals.<Expression>newHashSet());
              }
            });
            TypeConformJudgement _typeConformJudgement = new TypeConformJudgement(parameter, ref);
            information.provideType(_typeConformJudgement);
          }
        };
        IterableExtensions.<Variable>forEach(_localReferencesOfParameter, _function);
      }
    };
    IterableExtensions.<Variable>forEach(_parameters, _function);
  }
  
  protected void _inferTypes(final CheckConstraint constraint, final TypeInformation information) {
  }
  
  protected void _inferTypes(final CompareConstraint constraint, final TypeInformation information) {
    boolean _and = false;
    boolean _and_1 = false;
    CompareFeature _feature = constraint.getFeature();
    boolean _equals = Objects.equal(_feature, CompareFeature.EQUALITY);
    if (!_equals) {
      _and_1 = false;
    } else {
      ValueReference _leftOperand = constraint.getLeftOperand();
      boolean _notEquals = (!Objects.equal(_leftOperand, null));
      _and_1 = _notEquals;
    }
    if (!_and_1) {
      _and = false;
    } else {
      ValueReference _rightOperand = constraint.getRightOperand();
      boolean _notEquals_1 = (!Objects.equal(_rightOperand, null));
      _and = _notEquals_1;
    }
    if (_and) {
      ValueReference _leftOperand_1 = constraint.getLeftOperand();
      ValueReference _rightOperand_1 = constraint.getRightOperand();
      TypeConformJudgement _typeConformJudgement = new TypeConformJudgement(_leftOperand_1, _rightOperand_1);
      information.provideType(_typeConformJudgement);
      ValueReference _rightOperand_2 = constraint.getRightOperand();
      ValueReference _leftOperand_2 = constraint.getLeftOperand();
      TypeConformJudgement _typeConformJudgement_1 = new TypeConformJudgement(_rightOperand_2, _leftOperand_2);
      information.provideType(_typeConformJudgement_1);
    }
  }
  
  protected void _inferTypes(final PatternCompositionConstraint constraint, final TypeInformation information) {
    boolean _isNegative = constraint.isNegative();
    boolean _not = (!_isNegative);
    if (_not) {
      final PatternCall call = constraint.getCall();
      this.inferCallTypes(call, information);
    }
  }
  
  private void inferCallTypes(final PatternCall call, final TypeInformation information) {
    final Pattern pattern = call.getPatternRef();
    for (int i = 0; (i < Math.min(call.getParameters().size(), pattern.getParameters().size())); i++) {
      EList<ValueReference> _parameters = call.getParameters();
      ValueReference _get = _parameters.get(i);
      EList<Variable> _parameters_1 = pattern.getParameters();
      Variable _get_1 = _parameters_1.get(i);
      ParameterTypeJudgement _parameterTypeJudgement = new ParameterTypeJudgement(_get, _get_1);
      information.provideType(_parameterTypeJudgement);
    }
  }
  
  protected void _inferTypes(final TypeCheckConstraint constraint, final TypeInformation information) {
    final EntityType constraintType = constraint.getType();
    boolean _and = false;
    if (!(constraintType instanceof JavaType)) {
      _and = false;
    } else {
      boolean _isValidType = this.typeSystem.isValidType(constraintType);
      _and = _isValidType;
    }
    if (_and) {
      final IInputKey sourceType = this.typeSystem.extractTypeDescriptor(constraintType);
      boolean _notEquals = (!Objects.equal(sourceType, null));
      if (_notEquals) {
        VariableReference _var = constraint.getVar();
        TypeJudgement _typeJudgement = new TypeJudgement(_var, sourceType);
        information.provideType(_typeJudgement);
      }
    }
  }
  
  protected void _inferTypes(final PathExpressionConstraint constraint, final TypeInformation information) {
    PathExpressionHead _head = constraint.getHead();
    Type _type = _head.getType();
    boolean _isValidType = this.typeSystem.isValidType(_type);
    boolean _not = (!_isValidType);
    if (_not) {
      return;
    }
    PathExpressionHead _head_1 = constraint.getHead();
    Type _type_1 = _head_1.getType();
    final IInputKey sourceType = this.typeSystem.extractTypeDescriptor(_type_1);
    PathExpressionHead _head_2 = constraint.getHead();
    PathExpressionTail tail = _head_2.getTail();
    while ((!Objects.equal(tail.getTail(), null))) {
      PathExpressionTail _tail = tail.getTail();
      tail = _tail;
    }
    Type _type_2 = tail.getType();
    boolean _isValidType_1 = this.typeSystem.isValidType(_type_2);
    boolean _not_1 = (!_isValidType_1);
    if (_not_1) {
      return;
    }
    Type _type_3 = tail.getType();
    final IInputKey targetType = this.typeSystem.extractTypeDescriptor(_type_3);
    boolean _and = false;
    boolean _notEquals = (!Objects.equal(sourceType, null));
    if (!_notEquals) {
      _and = false;
    } else {
      boolean _notEquals_1 = (!Objects.equal(targetType, null));
      _and = _notEquals_1;
    }
    if (_and) {
      PathExpressionHead _head_3 = constraint.getHead();
      VariableReference _src = _head_3.getSrc();
      TypeJudgement _typeJudgement = new TypeJudgement(_src, sourceType);
      information.provideType(_typeJudgement);
      PathExpressionHead _head_4 = constraint.getHead();
      ValueReference _dst = _head_4.getDst();
      TypeJudgement _typeJudgement_1 = new TypeJudgement(_dst, targetType);
      information.provideType(_typeJudgement_1);
    }
  }
  
  protected void _inferTypes(final AggregatedValue reference, final TypeInformation information) {
    PatternCall _call = reference.getCall();
    this.inferCallTypes(_call, information);
    boolean _or = false;
    boolean _equals = Objects.equal(reference, null);
    if (_equals) {
      _or = true;
    } else {
      JvmDeclaredType _aggregator = reference.getAggregator();
      boolean _equals_1 = Objects.equal(_aggregator, null);
      _or = _equals_1;
    }
    if (_or) {
      return;
    }
    final List<VariableValue> values = AggregatorUtil.getAllAggregatorVariables(reference);
    int _size = values.size();
    boolean _equals_2 = (_size == 0);
    if (_equals_2) {
      boolean _mustHaveAggregatorVariables = AggregatorUtil.mustHaveAggregatorVariables(reference);
      if (_mustHaveAggregatorVariables) {
        return;
      }
      JvmDeclaredType _aggregator_1 = reference.getAggregator();
      final List<JvmType> returnTypes = AggregatorUtil.getReturnTypes(_aggregator_1);
      boolean _or_1 = false;
      boolean _equals_3 = Objects.equal(returnTypes, null);
      if (_equals_3) {
        _or_1 = true;
      } else {
        int _size_1 = returnTypes.size();
        boolean _notEquals = (_size_1 != 1);
        _or_1 = _notEquals;
      }
      if (_or_1) {
        JvmDeclaredType _aggregator_2 = reference.getAggregator();
        String _simpleName = _aggregator_2.getSimpleName();
        String _format = String.format("Return type for aggregator %s is non uniquely specified.", _simpleName);
        this.logger.warning(_format);
        return;
      }
      final JvmType returnType = returnTypes.get(0);
      String _identifier = returnType.getIdentifier();
      JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(_identifier);
      TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
      information.provideType(_typeJudgement);
    } else {
      boolean _or_2 = false;
      int _size_2 = values.size();
      boolean _notEquals_1 = (_size_2 != 1);
      if (_notEquals_1) {
        _or_2 = true;
      } else {
        boolean _mustHaveAggregatorVariables_1 = AggregatorUtil.mustHaveAggregatorVariables(reference);
        boolean _not = (!_mustHaveAggregatorVariables_1);
        _or_2 = _not;
      }
      if (_or_2) {
        return;
      }
      JvmDeclaredType _aggregator_3 = reference.getAggregator();
      final List<JvmType> parameterTypes = AggregatorUtil.getParameterTypes(_aggregator_3);
      JvmDeclaredType _aggregator_4 = reference.getAggregator();
      final List<JvmType> returnTypes_1 = AggregatorUtil.getReturnTypes(_aggregator_4);
      boolean _or_3 = false;
      boolean _equals_4 = Objects.equal(returnTypes_1, null);
      if (_equals_4) {
        _or_3 = true;
      } else {
        int _size_3 = returnTypes_1.size();
        int _size_4 = parameterTypes.size();
        boolean _notEquals_2 = (_size_3 != _size_4);
        _or_3 = _notEquals_2;
      }
      if (_or_3) {
        JvmDeclaredType _aggregator_5 = reference.getAggregator();
        String _identifier_1 = _aggregator_5.getIdentifier();
        String _format_1 = String.format(
          "Incorrect aggregator type annotation for aggregator %s: Different number of parameters and return types", _identifier_1);
        this.logger.warning(_format_1);
        return;
      }
      for (int i = 0; (i < returnTypes_1.size()); i++) {
        JvmType _get = returnTypes_1.get(i);
        String _identifier_2 = _get.getIdentifier();
        JavaTransitiveInstancesKey _javaTransitiveInstancesKey_1 = new JavaTransitiveInstancesKey(_identifier_2);
        PatternCall _call_1 = reference.getCall();
        EList<ValueReference> _parameters = _call_1.getParameters();
        int _aggregateVariableIndex = AggregatorUtil.getAggregateVariableIndex(reference);
        ValueReference _get_1 = _parameters.get(_aggregateVariableIndex);
        JvmType _get_2 = parameterTypes.get(i);
        String _identifier_3 = _get_2.getIdentifier();
        JavaTransitiveInstancesKey _javaTransitiveInstancesKey_2 = new JavaTransitiveInstancesKey(_identifier_3);
        ConditionalJudgement _conditionalJudgement = new ConditionalJudgement(reference, _javaTransitiveInstancesKey_1, _get_1, _javaTransitiveInstancesKey_2);
        information.provideType(_conditionalJudgement);
      }
    }
  }
  
  protected void _inferTypes(final Expression reference, final TypeInformation information) {
  }
  
  protected void _inferTypes(final FunctionEvaluationValue reference, final TypeInformation information) {
    XExpression _expression = reference.getExpression();
    XbaseExpressionTypeJudgement _xbaseExpressionTypeJudgement = new XbaseExpressionTypeJudgement(reference, _expression, this.typeResolver);
    information.provideType(_xbaseExpressionTypeJudgement);
  }
  
  protected void _inferTypes(final BoolValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(Boolean.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  /**
   * @since 1.5
   */
  protected void _inferTypes(final NumberValue reference, final TypeInformation information) {
    boolean _and = false;
    XNumberLiteral _value = reference.getValue();
    boolean _notEquals = (!Objects.equal(_value, null));
    if (!_notEquals) {
      _and = false;
    } else {
      XNumberLiteral _value_1 = reference.getValue();
      boolean _eIsProxy = _value_1.eIsProxy();
      boolean _not = (!_eIsProxy);
      _and = _not;
    }
    if (_and) {
      XNumberLiteral _value_2 = reference.getValue();
      final Class<? extends Number> type = this.literals.getJavaType(_value_2);
      JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(type);
      TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
      information.provideType(_typeJudgement);
    }
  }
  
  protected void _inferTypes(final ListValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(List.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  protected void _inferTypes(final StringValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(String.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  protected void _inferTypes(final VariableValue reference, final TypeInformation information) {
  }
  
  public void inferTypes(final EObject reference, final TypeInformation information) {
    if (reference instanceof AggregatedValue) {
      _inferTypes((AggregatedValue)reference, information);
      return;
    } else if (reference instanceof BoolValue) {
      _inferTypes((BoolValue)reference, information);
      return;
    } else if (reference instanceof FunctionEvaluationValue) {
      _inferTypes((FunctionEvaluationValue)reference, information);
      return;
    } else if (reference instanceof ListValue) {
      _inferTypes((ListValue)reference, information);
      return;
    } else if (reference instanceof NumberValue) {
      _inferTypes((NumberValue)reference, information);
      return;
    } else if (reference instanceof StringValue) {
      _inferTypes((StringValue)reference, information);
      return;
    } else if (reference instanceof VariableValue) {
      _inferTypes((VariableValue)reference, information);
      return;
    } else if (reference instanceof CheckConstraint) {
      _inferTypes((CheckConstraint)reference, information);
      return;
    } else if (reference instanceof CompareConstraint) {
      _inferTypes((CompareConstraint)reference, information);
      return;
    } else if (reference instanceof PathExpressionConstraint) {
      _inferTypes((PathExpressionConstraint)reference, information);
      return;
    } else if (reference instanceof PatternCompositionConstraint) {
      _inferTypes((PatternCompositionConstraint)reference, information);
      return;
    } else if (reference instanceof TypeCheckConstraint) {
      _inferTypes((TypeCheckConstraint)reference, information);
      return;
    } else if (reference instanceof Expression) {
      _inferTypes((Expression)reference, information);
      return;
    } else if (reference instanceof Pattern) {
      _inferTypes((Pattern)reference, information);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(reference, information).toString());
    }
  }
}
