/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.logging.aeri.core.filters;

import static java.util.concurrent.TimeUnit.*;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.epp.logging.aeri.core.SystemControl;
import org.eclipse.epp.logging.aeri.core.util.Statuses;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

public class RecentlySeenFilter implements Predicate<IStatus> {

    private Cache<String, String> cache;

    public RecentlySeenFilter() {
        cache = CacheBuilder.newBuilder().concurrencyLevel(1).expireAfterAccess(1, HOURS).initialCapacity(30).maximumSize(500).build();
    }

    @VisibleForTesting
    public RecentlySeenFilter(long timoutInMillis) {
        cache = CacheBuilder.newBuilder().concurrencyLevel(1).expireAfterAccess(timoutInMillis, MILLISECONDS).initialCapacity(30)
                .maximumSize(500).build();
    }

    /**
     * Returns <code>true</code> if the status was never observed before or its last occurrence is more than one hour ago.
     */
    @Override
    public boolean apply(IStatus input) {
        if (SystemControl.isDebug()) {
            return true;
        }
        String fp = Statuses.newThrowableFingerprint(input.getException(), false, false);
        String val = cache.getIfPresent(fp);
        if (val == null) {
            cache.put(fp, "");
        }
        return val == null;
    }

}
