/**
 * Copyright (c) 2010-2016, Grill Balázs, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Grill Balázs - initial API and implementation
 */
package org.eclipse.viatra.query.runtime.localsearch.planner.cost.impl;

import com.google.common.base.Objects;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.viatra.query.runtime.localsearch.planner.cost.IConstraintEvaluationContext;
import org.eclipse.viatra.query.runtime.localsearch.planner.cost.ICostFunction;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;
import org.eclipse.viatra.query.runtime.matchers.context.IQueryMetaContext;
import org.eclipse.viatra.query.runtime.matchers.context.IQueryRuntimeContext;
import org.eclipse.viatra.query.runtime.matchers.context.InputKeyImplication;
import org.eclipse.viatra.query.runtime.matchers.planning.helpers.FunctionalDependencyHelper;
import org.eclipse.viatra.query.runtime.matchers.psystem.PConstraint;
import org.eclipse.viatra.query.runtime.matchers.psystem.PVariable;
import org.eclipse.viatra.query.runtime.matchers.psystem.analysis.QueryAnalyzer;
import org.eclipse.viatra.query.runtime.matchers.psystem.basicdeferred.ExportedParameter;
import org.eclipse.viatra.query.runtime.matchers.psystem.basicdeferred.TypeFilterConstraint;
import org.eclipse.viatra.query.runtime.matchers.psystem.basicenumerables.ConstantValue;
import org.eclipse.viatra.query.runtime.matchers.psystem.basicenumerables.PositivePatternCall;
import org.eclipse.viatra.query.runtime.matchers.psystem.basicenumerables.TypeConstraint;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PParameter;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PQuery;
import org.eclipse.viatra.query.runtime.matchers.tuple.Tuple;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;

/**
 * Cost function which calculates cost based on the cardinality of items in the runtime model
 * 
 * @author Grill Balázs
 * @since 1.4
 */
@SuppressWarnings("all")
public abstract class StatisticsBasedConstraintCostFunction implements ICostFunction {
  protected static double MAX_COST = 250.0;
  
  protected static double DEFAULT_COST = (StatisticsBasedConstraintCostFunction.MAX_COST - 100.0);
  
  public abstract long countTuples(final IConstraintEvaluationContext input, final IInputKey supplierKey);
  
  @Override
  public double apply(final IConstraintEvaluationContext input) {
    PConstraint _constraint = input.getConstraint();
    return this.calculateCost(_constraint, input);
  }
  
  protected double _calculateCost(final ConstantValue constant, final IConstraintEvaluationContext input) {
    return 0.0f;
  }
  
  protected double _calculateCost(final TypeConstraint constraint, final IConstraintEvaluationContext input) {
    final Collection<PVariable> freeMaskVariables = input.getFreeVariables();
    final Collection<PVariable> boundMaskVariables = input.getBoundVariables();
    IInputKey supplierKey = ((TypeConstraint) constraint).getSupplierKey();
    long arity = supplierKey.getArity();
    if ((arity == 1)) {
      return this.calculateUnaryConstraintCost(constraint, input);
    } else {
      if ((arity == 2)) {
        long edgeCount = this.countTuples(input, supplierKey);
        Tuple _variablesTuple = ((TypeConstraint) constraint).getVariablesTuple();
        Object _get = _variablesTuple.get(0);
        PVariable srcVariable = ((PVariable) _get);
        Tuple _variablesTuple_1 = ((TypeConstraint) constraint).getVariablesTuple();
        Object _get_1 = _variablesTuple_1.get(1);
        PVariable dstVariable = ((PVariable) _get_1);
        boolean isInverse = false;
        boolean _and = false;
        boolean _contains = freeMaskVariables.contains(srcVariable);
        if (!_contains) {
          _and = false;
        } else {
          boolean _contains_1 = boundMaskVariables.contains(dstVariable);
          _and = _contains_1;
        }
        if (_and) {
          isInverse = true;
        }
        double _calculateBinaryExtendCost = this.calculateBinaryExtendCost(supplierKey, srcVariable, dstVariable, isInverse, edgeCount, input);
        double _xifexpression = (double) 0;
        if (isInverse) {
          _xifexpression = 1.0;
        } else {
          _xifexpression = 0.0;
        }
        return (_calculateBinaryExtendCost + _xifexpression);
      } else {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Cost calculation for arity ");
        _builder.append(arity, "");
        _builder.append(" is not implemented yet");
        throw new RuntimeException(_builder.toString());
      }
    }
  }
  
  protected double calculateBinaryExtendCost(final IInputKey supplierKey, final PVariable srcVariable, final PVariable dstVariable, final boolean isInverse, final long edgeCount, final IConstraintEvaluationContext input) {
    final Collection<PVariable> freeMaskVariables = input.getFreeVariables();
    final Collection<PVariable> boundMaskVariables = input.getBoundVariables();
    final PConstraint constraint = input.getConstraint();
    IQueryRuntimeContext _runtimeContext = input.getRuntimeContext();
    IQueryMetaContext metaContext = _runtimeContext.getMetaContext();
    final QueryAnalyzer queryAnalyzer = input.getQueryAnalyzer();
    Collection<InputKeyImplication> implications = metaContext.getImplications(supplierKey);
    double srcCount = (-1);
    double dstCount = (-1);
    for (final InputKeyImplication implication : implications) {
      {
        List<Integer> impliedIndices = implication.getImpliedIndices();
        boolean _and = false;
        int _size = impliedIndices.size();
        boolean _equals = (_size == 1);
        if (!_equals) {
          _and = false;
        } else {
          boolean _contains = impliedIndices.contains(Integer.valueOf(0));
          _and = _contains;
        }
        if (_and) {
          IInputKey _impliedKey = implication.getImpliedKey();
          long _countTuples = this.countTuples(input, _impliedKey);
          srcCount = _countTuples;
        } else {
          boolean _and_1 = false;
          int _size_1 = impliedIndices.size();
          boolean _equals_1 = (_size_1 == 1);
          if (!_equals_1) {
            _and_1 = false;
          } else {
            boolean _contains_1 = impliedIndices.contains(Integer.valueOf(1));
            _and_1 = _contains_1;
          }
          if (_and_1) {
            IInputKey _impliedKey_1 = implication.getImpliedKey();
            long _countTuples_1 = this.countTuples(input, _impliedKey_1);
            dstCount = _countTuples_1;
          }
        }
      }
    }
    boolean _and = false;
    boolean _contains = freeMaskVariables.contains(srcVariable);
    if (!_contains) {
      _and = false;
    } else {
      boolean _contains_1 = freeMaskVariables.contains(dstVariable);
      _and = _contains_1;
    }
    if (_and) {
      return (dstCount * srcCount);
    } else {
      double srcNodeCount = (-1);
      double dstNodeCount = (-1);
      if (isInverse) {
        srcNodeCount = dstCount;
        dstNodeCount = srcCount;
      } else {
        srcNodeCount = srcCount;
        dstNodeCount = dstCount;
      }
      if (((srcNodeCount > (-1)) && (edgeCount > (-1)))) {
        if ((srcNodeCount == 0)) {
          return 0;
        } else {
          return (((double) edgeCount) / srcNodeCount);
        }
      } else {
        if (((srcCount > (-1)) && (dstCount > (-1)))) {
          if ((srcCount != 0)) {
            return (dstNodeCount / srcNodeCount);
          } else {
            return 1.0f;
          }
        } else {
          final Map<Set<PVariable>, Set<PVariable>> functionalDependencies = queryAnalyzer.getFunctionalDependencies(Collections.<PConstraint>unmodifiableSet(CollectionLiterals.<PConstraint>newHashSet(constraint)), false);
          final Set<PVariable> impliedVariables = FunctionalDependencyHelper.<PVariable>closureOf(boundMaskVariables, functionalDependencies);
          boolean _and_1 = false;
          boolean _notEquals = (!Objects.equal(impliedVariables, null));
          if (!_notEquals) {
            _and_1 = false;
          } else {
            boolean _containsAll = impliedVariables.containsAll(freeMaskVariables);
            _and_1 = _containsAll;
          }
          if (_and_1) {
            return 1.0;
          } else {
            return StatisticsBasedConstraintCostFunction.DEFAULT_COST;
          }
        }
      }
    }
  }
  
  protected double calculateUnaryConstraintCost(final TypeConstraint constraint, final IConstraintEvaluationContext input) {
    Tuple _variablesTuple = constraint.getVariablesTuple();
    Object _get = _variablesTuple.get(0);
    PVariable variable = ((PVariable) _get);
    Collection<PVariable> _boundVariables = input.getBoundVariables();
    boolean _contains = _boundVariables.contains(variable);
    if (_contains) {
      return 0.9;
    } else {
      IInputKey _supplierKey = constraint.getSupplierKey();
      long _countTuples = this.countTuples(input, _supplierKey);
      return (_countTuples + StatisticsBasedConstraintCostFunction.DEFAULT_COST);
    }
  }
  
  protected double _calculateCost(final ExportedParameter exportedParam, final IConstraintEvaluationContext input) {
    return 0.0;
  }
  
  protected double _calculateCost(final TypeFilterConstraint exportedParam, final IConstraintEvaluationContext input) {
    return 0.0;
  }
  
  protected double _calculateCost(final PositivePatternCall patternCall, final IConstraintEvaluationContext input) {
    QueryAnalyzer _queryAnalyzer = input.getQueryAnalyzer();
    final Map<Set<PVariable>, Set<PVariable>> dependencies = _queryAnalyzer.getFunctionalDependencies(Collections.<PConstraint>unmodifiableSet(CollectionLiterals.<PConstraint>newHashSet(patternCall)), false);
    Collection<PVariable> _boundVariables = input.getBoundVariables();
    final Set<PVariable> boundOrImplied = FunctionalDependencyHelper.<PVariable>closureOf(_boundVariables, dependencies);
    PQuery _referredQuery = patternCall.getReferredQuery();
    final List<PParameter> parameters = _referredQuery.getParameters();
    double result = 1.0;
    for (int i = 0; (i < parameters.size()); i++) {
      {
        final PVariable variable = patternCall.getVariableInTuple(i);
        double _xifexpression = (double) 0;
        boolean _contains = boundOrImplied.contains(variable);
        if (_contains) {
          _xifexpression = 0.9;
        } else {
          double _xblockexpression = (double) 0;
          {
            PParameter _get = parameters.get(i);
            final IInputKey type = _get.getDeclaredUnaryType();
            double _xifexpression_1 = (double) 0;
            boolean _equals = Objects.equal(type, null);
            if (_equals) {
              _xifexpression_1 = StatisticsBasedConstraintCostFunction.DEFAULT_COST;
            } else {
              _xifexpression_1 = this.countTuples(input, type);
            }
            _xblockexpression = _xifexpression_1;
          }
          _xifexpression = _xblockexpression;
        }
        double _multiply = (result * _xifexpression);
        result = _multiply;
      }
    }
    return result;
  }
  
  /**
   * Default cost calculation strategy
   */
  protected double _calculateCost(final PConstraint constraint, final IConstraintEvaluationContext input) {
    Collection<PVariable> _freeVariables = input.getFreeVariables();
    boolean _isEmpty = _freeVariables.isEmpty();
    if (_isEmpty) {
      return 1.0;
    } else {
      return StatisticsBasedConstraintCostFunction.DEFAULT_COST;
    }
  }
  
  public double calculateCost(final PConstraint exportedParam, final IConstraintEvaluationContext input) {
    if (exportedParam instanceof ExportedParameter) {
      return _calculateCost((ExportedParameter)exportedParam, input);
    } else if (exportedParam instanceof TypeFilterConstraint) {
      return _calculateCost((TypeFilterConstraint)exportedParam, input);
    } else if (exportedParam instanceof ConstantValue) {
      return _calculateCost((ConstantValue)exportedParam, input);
    } else if (exportedParam instanceof PositivePatternCall) {
      return _calculateCost((PositivePatternCall)exportedParam, input);
    } else if (exportedParam instanceof TypeConstraint) {
      return _calculateCost((TypeConstraint)exportedParam, input);
    } else if (exportedParam != null) {
      return _calculateCost(exportedParam, input);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(exportedParam, input).toString());
    }
  }
}
