/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jst.j2ee.internal.provider;


import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.provider.ChangeNotifier;
import org.eclipse.emf.edit.provider.ComposeableAdapterFactory;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.Disposable;
import org.eclipse.emf.edit.provider.IChangeNotifier;
import org.eclipse.emf.edit.provider.IDisposable;
import org.eclipse.emf.edit.provider.IEditingDomainItemProvider;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.IItemPropertySource;
import org.eclipse.emf.edit.provider.INotifyChangedListener;
import org.eclipse.emf.edit.provider.IStructuredItemContentProvider;
import org.eclipse.emf.edit.provider.ITableItemLabelProvider;
import org.eclipse.emf.edit.provider.ITreeItemContentProvider;
import org.eclipse.jst.j2ee.client.internal.util.ClientAdapterFactory;


/**
 * This is the factory that is used to provide the interfaces needed to support
 * { @link org.eclipse.jface.viewers.Viewer}s. The adapters generated by this factory convert EMF
 * adapter notifications into calls to {@link #fireNotifyChanged fireNotifyChanged}. The adapters
 * also support property sheets, see {@link org.eclipse.ui.views.properties}. Note that most of the
 * adapters are shared among multiple instances.
 */
public class ClientItemProviderAdapterFactory extends ClientAdapterFactory implements ComposeableAdapterFactory, IChangeNotifier, IDisposable {
	/**
	 * This keeps track of the root adapter factory that delegates to this adapter factory.
	 */
	protected ComposedAdapterFactory parentAdapterFactory;
	/**
	 * This is used to implement {@link #IChangeNotifier}.
	 */
	protected IChangeNotifier changeNotifier = new ChangeNotifier();
	/**
	 * This keeps track of all the supported types checked by
	 * { @link #isFactoryForType isFactoryForType}.
	 */
	protected Collection supportedTypes = new ArrayList();
	/**
	 * This keeps track of the one adapter used for all
	 * { @link org.eclipse.jst.j2ee.internal.internal.client.ApplicationClient}instances.
	 */
	protected ApplicationClientItemProvider applicationClientItemProvider;
	protected Disposable disposable = new Disposable();

	/**
	 * This constructs an instance.
	 */
	public ClientItemProviderAdapterFactory() {
		supportedTypes.add(IStructuredItemContentProvider.class);
		supportedTypes.add(ITreeItemContentProvider.class);
		supportedTypes.add(IItemPropertySource.class);
		supportedTypes.add(IEditingDomainItemProvider.class);
		supportedTypes.add(IItemLabelProvider.class);
		supportedTypes.add(ITableItemLabelProvider.class);
	}

	@Override
	public Adapter adapt(Notifier target, Object adapterKey) {
		return super.adapt(target, this);
	}

	@Override
	public Object adapt(Object object, Object type) {
		// This is a kludge to deal with enumerators, which crash the doSwitch.
		//
		if (object instanceof EObject && ((EObject) object).eClass() == null) {
			return null;
		}

		if (isFactoryForType(type)) {
			Object adapter = super.adapt(object, type);
			if (!(type instanceof Class) || (((Class) type).isInstance(adapter))) {
				return adapter;
			}
		}

		return null;
	}

	@Override
	public Adapter adaptNew(Notifier target, Object adapterType) {
		Adapter adapter = super.adaptNew(target, adapterType);
		disposable.add(adapter);
		return adapter;
	}

	/**
	 * This adds a listener. <!-- begin-user-doc --> <!-- end-user-doc -->
	 * 
	 * @generated
	 */
	@Override
	public void addListener(INotifyChangedListener notifyChangedListener) {
		changeNotifier.addListener(notifyChangedListener);
	}

	/**
	 * This creates an adapter for a
	 * { @link org.eclipse.jst.j2ee.internal.internal.client.ApplicationClient}.
	 */
	@Override
	public Adapter createApplicationClientAdapter() {
		if (applicationClientItemProvider == null) {
			applicationClientItemProvider = new ApplicationClientItemProvider(this);
		}

		return applicationClientItemProvider;
	}

	@Override
	public void dispose() {
		disposable.dispose();
	}

	/**
	 * This delegates to {@link #changeNotifier}and to {@link #parentAdapterFactory}.
	 */
	/*
	 * public void fireNotifyChanged(Object object, int eventType, Object feature, Object oldValue,
	 * Object newValue, int index) { changeNotifier.fireNotifyChanged(object, eventType, feature,
	 * oldValue, newValue, index);
	 * 
	 * if (parentAdapterFactory != null) { parentAdapterFactory.fireNotifyChanged(object, eventType,
	 * feature, oldValue, newValue, index); } }
	 */
	/**
	 * This returns the root adapter factory that contains this factory.
	 */
	@Override
	public ComposeableAdapterFactory getRootAdapterFactory() {
		return parentAdapterFactory == null ? this : parentAdapterFactory.getRootAdapterFactory();
	}

	@Override
	public boolean isFactoryForType(Object type) {
		return super.isFactoryForType(type) || supportedTypes.contains(type);
	}

	/**
	 * This removes a listener.
	 * 
	 * @generated
	 */
	@Override
	public void removeListener(INotifyChangedListener notifyChangedListener) {
		changeNotifier.removeListener(notifyChangedListener);
	}

	/**
	 * This delegates to {@link #changeNotifier}and to {@link #parentAdapterFactory}. <!--
	 * begin-user-doc --> <!-- end-user-doc -->
	 * 
	 * @generated
	 */
	@Override
	public void fireNotifyChanged(Notification notification) {
		changeNotifier.fireNotifyChanged(notification);

		if (parentAdapterFactory != null) {
			parentAdapterFactory.fireNotifyChanged(notification);
		}
	}

	/**
	 * This sets the composed adapter factory that contains this factory.
	 */
	@Override
	public void setParentAdapterFactory(ComposedAdapterFactory parentAdapterFactory) {
		this.parentAdapterFactory = parentAdapterFactory;
	}
}
