# -*- coding: utf-8 -*- #
# frozen_string_literal: true

require 'open-uri'

VIML_SYNTAX_URI = "https://raw.githubusercontent.com/vim/vim/master/runtime/syntax/vim.vim"
VIML_KEYWORDS_FILE = "./lib/rouge/lexers/viml/keywords.rb"

namespace :builtins do
  task :viml do
    generator = Rouge::Tasks::Builtins::VimL.new

    input    = URI.open(VIML_SYNTAX_URI) { |f| f.read }
    keywords = generator.extract_keywords(input)

    output = generator.render_output(keywords)

    File.write(VIML_KEYWORDS_FILE, output)
  end
end

module Rouge
  module Tasks
    module Builtins
      class VimL
        def extract_keywords(input)
          keywords = Hash.new { |h,k| h[k] = Set.new }

          input.each_line do |line|
            line =~ %r(^
              (?: syn[ ]keyword[ ](vim(?:AutoEvent|Command|FuncName|Option))[ ]contained )
              (.*)
            $)x

            next unless $1

            name = $1
            functions = $2

            key = case name
                  when "vimAutoEvent"
                    :auto
                  when "vimCommand"
                    :command
                  when "vimFuncName"
                    :function
                  when "vimOption"
                    :option
                  end

            functions.scan(/([\w:]+)(?:\[(\w+)\])?/) do
              keywords[key].merge [$1, "#{$1}#{$2}"]
            end
          end

          keywords
        end

        def render_output(keywords, &b)
          return enum_for(:render_output, keywords).to_a.join("\n") unless b

          yield   "# -*- coding: utf-8 -*- #"
          yield   "# frozen_string_literal: true"
          yield   ""
          yield   "# DO NOT EDIT"
          yield   "# This file is automatically generated by `rake builtins:viml`."
          yield   "# See tasks/builtins/viml.rake for more info."
          yield   ""
          yield   "module Rouge"
          yield   "  module Lexers"
          yield   "    class VimL"
          yield   "      def self.keywords"
          yield   "        @keywords ||= {}.tap do |kw|"
          keywords.each do |k,v|
            yield "          kw[#{k.inspect}] = Set.new #{v.to_a.inspect}"
          end
          yield   "        end"
          yield   "      end"
          yield   "    end"
          yield   "  end"
          yield   "end"
        end
      end
    end
  end
end
