/* Libreswan config file parser (confread.c)
 * Copyright (C) 2001-2002 Mathieu Lafon - Arkoon Network Security
 * Copyright (C) 2004 Xelerance Corporation
 * Copyright (C) 2006-2008 Michael Richardson <mcr@xelerance.com>
 * Copyright (C) 2007 Ken Bantoft <ken@xelerance.com>
 * Copyright (C) 2006-2012 Paul Wouters <paul@xelerance.com>
 * Copyright (C) 2010 Michael Smith <msmith@cbnco.com>
 * Copyright (C) 2010 Tuomo Soini <tis@foobar.fi>
 * Copyright (C) 2012-2019 Paul Wouters <pwouters@redhat.com>
 * Copyright (C) 2012 Avesh Agarwal <avagarwa@redhat.com>
 * Copyright (C) 2012 Antony Antony <antony@phenome.org>
 * Copyright (C) 2013 Florian Weimer <fweimer@redhat.com>
 * Copyright (C) 2013 David McCullough <ucdevel@gmail.com>
 * Copyright (C) 2013-2019 D. Hugh Redelmeier <hugh@mimosa.com>
 * Copyright (C) 2016 Andrew Cagney <cagney@gnu.org>
 * Copyright (C) 2017-2018 Vukasin Karadzic <vukasin.karadzic@gmail.com>
 * Copyright (C) 2017 Mayank Totale <mtotale@gmail.com>
 * Copyright (C) 2020 Yulia Kuzovkova <ukuzovkova@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <https://www.gnu.org/licenses/gpl2.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <sys/socket.h>		/* for AF_UNSPEC */

#include "lswalloc.h"
#include "ip_address.h"
#include "ip_info.h"
#include "hunk.h"		/* for char_is_space() */
#include "ip_cidr.h"
#include "ttodata.h"
#include "config_setup.h"

#include "ipsecconf/parser.h"
#include "ipsecconf/confread.h"
#include "ipsecconf/interfaces.h"

#include "ipsecconf/keywords.h"
#include "ipsecconf/parser.h"	/* includes parser.tab.h generated by bison; requires keywords.h */

#include "whack.h" /* for DEFAULT_CTL_SOCKET */
#include "lswlog.h"

const char *rootdir;	/* when evaluating paths, prefix this to them */
const char *rootdir2;	/* or... try this one too */


static bool parse_ipsec_conf_config_conn(struct starter_config *cfg,
					 struct ipsec_conf *cfgp,
					 struct logger *logger);

static bool translate_conn(struct starter_conn *conn,
			   const struct ipsec_conf *cfgp,
			   struct section_list *sl,
			   enum keyword_set assigned_value,
			   struct logger *logger);

static struct starter_config *alloc_starter_config(void)
{
	struct starter_config *cfg = alloc_thing(struct starter_config, "starter_config cfg");

	TAILQ_INIT(&cfg->conns);

	/* ==== conn %default ==== */

	struct starter_conn *d = &cfg->conn_default;

	d->end[LEFT_END].leftright = "left";
	d->end[RIGHT_END].leftright = "right";

	d->state = STATE_LOADED;
	/* ==== end of conn %default ==== */

	return cfg;
}

/**
 * Take the list of key-value pairs, from ipsec.conf and in KW, and
 * turn them into an array in starter_conn.
 *
 * @param conn a connection definition
 * @param sl a section_list
 * @param assigned_value is set to either k_set, or k_default.
 *        k_default is used when we are loading a conn that should be
 *        considered to be a "default" value, and that replacing this
 *        value is considered acceptable.
 * @return bool TRUE if unsuccessful
 */

static bool translate_field(struct starter_conn *conn,
			    const struct ipsec_conf *cfgp,
			    const struct section_list *sl,
			    enum keyword_set assigned_value,
			    const struct keyval_entry *kw,
			    const char *leftright,
			    config_conn_values values,
			    struct logger *logger)
{
	bool ok = true;

	unsigned int field = kw->keyval.key->field;

	assert(kw->keyval.key != NULL);

	switch (kw->keyval.key->type) {
	case kt_also:
	{
		struct section_list *addin;
		const char *seeking = kw->keyval.val;
		TAILQ_FOREACH(addin, &cfgp->sections, next) {
			if (streq(seeking, addin->name)) {
				break;
			}
		}
		if (addin == NULL) {
			llog(RC_LOG, logger,
			     "cannot find conn '%s' needed by conn '%s'",
			     seeking, conn->name);
			ok = false;
			break;
		}
		/* translate things, but do not replace earlier settings! */
		ok &= translate_conn(conn, cfgp, addin, k_set, logger);
		break;
	}
	case kt_string:
		/* all treated as strings for now, even loose enums */
		if (values[field].set == k_set) {
			llog(RC_LOG, logger,
			     "duplicate key '%s%s' in conn %s while processing def %s",
			     leftright, kw->keyval.key->keyname,
			     conn->name,
			     sl->name);

			/* only fatal if we try to change values */
			if (kw->keyval.val == NULL ||
			    values[field].string == NULL ||
			    !streq(kw->keyval.val, values[field].string)) {
				ok = false;
				break;
			}
		}
		pfreeany(values[field].string);

		if (kw->keyval.val == NULL) {
			llog(RC_LOG, logger, "invalid %s value",
			     kw->keyval.key->keyname);
			ok = false;
			break;
		}

		values[field].string = clone_str(kw->keyval.val,
						 "kt_idtype kw->keyval.val");
		values[field].set = assigned_value;
		break;

	case kt_appendstring:
	case kt_appendlist:
		/* implicitly, this field can have multiple values */
		if (values[field].string == NULL) {
			values[field].string = clone_str(kw->keyval.val,
							 "kt_appendlist kw->keyval.val");
		} else {
			char *s = values[field].string;
			size_t old_len = strlen(s);	/* excludes '\0' */
			size_t new_len = strlen(kw->keyval.val);
			char *n = alloc_bytes(old_len + 1 + new_len + 1, "kt_appendlist");

			memcpy(n, s, old_len);
			n[old_len] = ' ';
			memcpy(n + old_len + 1, kw->keyval.val, new_len + 1);	/* includes '\0' */
			values[field].string = n;
			pfree(s);
		}
		values[field].set = true;
		break;

	case kt_sparse_name:
	case kt_unsigned:
		/* all treated as a number for now */
		if (values[field].set == k_set) {
			llog(RC_LOG, logger,
			     "duplicate key '%s%s' in conn %s while processing def %s",
			     leftright, kw->keyval.key->keyname,
			     conn->name,
			     sl->name);

			/* only fatal if we try to change values */
			if (values[field].option != (int)kw->number) {
				ok = false;
				break;
			}
		}

		values[field].option = kw->number;
		values[field].set = assigned_value;
		break;

	case kt_seconds:
		/* all treated as a number for now */
		if (values[field].set == k_set) {
			llog(RC_LOG, logger,
			     "duplicate key '%s%s' in conn %s while processing def %s",
			     leftright, kw->keyval.key->keyname,
			     conn->name,
			     sl->name);

			/* only fatal if we try to change values */
			if (deltatime_cmp(values[field].deltatime, !=, kw->deltatime)) {
				ok = false;
				break;
			}
		}

		values[field].deltatime = kw->deltatime;
		values[field].set = assigned_value;
		break;

	case kt_obsolete:
		break;
	}

	return ok;
}

static bool translate_leftright(struct starter_conn *conn,
				const struct ipsec_conf *cfgp,
				const struct section_list *sl,
				enum keyword_set assigned_value,
				const struct keyval_entry *kw,
				struct starter_end *this,
				struct logger *logger)
{
	return translate_field(conn, cfgp, sl, assigned_value, kw,
			       /*leftright*/this->leftright,
			       this->values, logger);
}

static bool translate_conn(struct starter_conn *conn,
			   const struct ipsec_conf *cfgp,
			   struct section_list *sl,
			   enum keyword_set assigned_value,
			   struct logger *logger)
{
	if (sl->beenhere) {
		ldbg(logger, "ignore duplicate include");
		return true; /* ok */
	}

	sl->beenhere = true;

	/*
	 * Note: not all errors are considered serious (see above).
	 */
	bool ok = true;

	const struct keyval_entry *kw;
	TAILQ_FOREACH(kw, &sl->keyvals, next) {
		if ((kw->keyval.key->validity & kv_leftright) ||
		    (kw->keyval.key->validity & kv_both)) {
			if (kw->keyval.left) {
				ok &= translate_leftright(conn, cfgp, sl, assigned_value,
							  kw, &conn->end[LEFT_END],
							  logger);
			}
			if (kw->keyval.right) {
				ok &= translate_leftright(conn, cfgp, sl, assigned_value,
							  kw, &conn->end[RIGHT_END],
							  logger);
			}
		} else {
			ok &= translate_field(conn, cfgp, sl, assigned_value, kw,
					      /*leftright*/"",
					      conn->values,
					      logger);
		}
	}

	return ok;
}

static bool load_conn(struct starter_conn *conn,
		      const struct ipsec_conf *cfgp,
		      struct section_list *sl,
		      bool alsoprocessing,
		      bool defaultconn,
		      struct logger *logger)
{
	/* reset all of the "beenhere" flags */
	struct section_list *s;
	TAILQ_FOREACH(s, &cfgp->sections, next) {
		s->beenhere = false;
	}

	/*
	 * Turn all of the keyword/value pairs into options/strings in
	 * left/right.
	 *
	 * DANGER: returns false on success.
	 */
	if (!translate_conn(conn, cfgp, sl,
			    defaultconn ? k_default : k_set,
			    logger)) {
		return false;
	}

	if (conn->values[KSCF_ALSO].string != NULL &&
	    !alsoprocessing) {
		llog(RC_LOG, logger, "also= is not valid in section '%s'",
		     sl->name);
		return false;	/* error */
	}

	if (conn->values[KNCF_TYPE].set) {
		switch ((enum type_options)conn->values[KNCF_TYPE].option) {
		case KS_UNSET:
			bad_case(KS_UNSET);

		case KS_TUNNEL:
			break;

		case KS_TRANSPORT:
			break;

		case KS_PASSTHROUGH:
			conn->never_negotiate_shunt = SHUNT_PASS;
			break;

		case KS_DROP:
			conn->never_negotiate_shunt = SHUNT_DROP;
			break;

		case KS_REJECT:
			llog(RC_LOG, logger, "warning: type=%%reject implemented as type=%%drop");
			conn->never_negotiate_shunt = SHUNT_DROP;
			break;
		}
	}

	conn->negotiation_shunt = conn->values[KNCF_NEGOTIATIONSHUNT].option;
	conn->failure_shunt = conn->values[KNCF_FAILURESHUNT].option;

	return true;
}

static void copy_conn_default(struct starter_conn *conn,
			      const struct starter_conn *def)
{
	/* structure copy to start */
	*conn = *def;

	/* unlink it */
	conn->link.tqe_next = NULL;
	conn->link.tqe_prev = NULL;

	/* Unshare all strings */

	/*
	 * Note: string fields in struct starter_end and struct starter_conn
	 * should correspond to STR_FIELD calls in copy_conn_default() and confread_free_conn.
	 */

# define STR_FIELD(f)  { conn->f = clone_str(conn->f, #f); }

	STR_FIELD(name);

	for (unsigned i = 0; i < elemsof(conn->values); i++)
		STR_FIELD(values[i].string);

	/* handle starter_end strings */

# define STR_FIELD_END(f) { STR_FIELD(end[LEFT_END].f); STR_FIELD(end[RIGHT_END].f); }

	for (unsigned i = 0; i < elemsof(conn->end[LEFT_END].values); i++)
		STR_FIELD_END(values[i].string);

# undef STR_FIELD_END

# undef STR_FIELD
}

static struct starter_conn *alloc_add_conn(struct starter_config *cfg, const char *name)
{
	struct starter_conn *conn = alloc_thing(struct starter_conn, "add_conn starter_conn");

	copy_conn_default(conn, &cfg->conn_default);
	assert(conn->name == NULL);
	conn->name = clone_str(name, "add conn name");
	conn->state = STATE_FAILED;

	TAILQ_INSERT_TAIL(&cfg->conns, conn, link);
	return conn;
}

enum config_section { CONFIG_SETUP, CONFIG_CONN };

static void check_config_keywords(struct logger *logger,
				  const enum config_section section,
				  const struct keywords_def *keywords)
{
	static bool checked;
	if (checked) {
		return;
	}
	checked = true;

	if (LDBGP(DBG_TMI, logger)) {
		ITEMS_FOR_EACH(k, keywords) {
			if (k->keyname == NULL) {
				continue;
			}
			unsigned i = (k - keywords->item);
			LDBG_log(logger, "[%u] %s", i, k->keyname);
		}
	}

	enum { NAME, ALIAS, OBSOLETE } group = NAME;

	ITEMS_FOR_EACH(k, keywords) {

		/*
		 * Ignore gaps, happens when #ifdefs are at play.
		 */

		if (k->keyname == NULL) {
			continue;
		}

		bool ok = true;
		unsigned i = (k - keywords->item);

		switch (group) {
		case NAME:
			if (k->validity & kv_alias) {
				group = ALIAS;
				break;
			}
			if (k->field == KNCF_OBSOLETE) {
				group = OBSOLETE;
				break;
			}
			break;
		case ALIAS:
			if (k->field == KNCF_OBSOLETE) {
				group = OBSOLETE;
				break;
			}
			break;
		case OBSOLETE:
			break;
		}

		ok &= pexpect(k->validity & kv_alias ? group == ALIAS :
			      group == NAME || group == OBSOLETE);
		ok &= pexpect(k->field == KNCF_OBSOLETE ? group == OBSOLETE :
			      group == NAME || group == ALIAS);
		ok &= pexpect(k->type == kt_obsolete ? group == OBSOLETE :
			      group == NAME || group == ALIAS);

		ok &= pexpect(group == NAME ? i == k->field : i > k->field);
		ok &= pexpect(group == OBSOLETE ? k->sparse_names == NULL : true);

		switch (section) {
		case CONFIG_SETUP:
			ok &= pexpect((k->field >= CONFIG_SETUP_KEYWORD_FLOOR &&
				       k->field < CONFIG_SETUP_KEYWORD_ROOF) ||
				      k->field == KNCF_OBSOLETE);
			ok &= pexpect((k->validity & (kv_leftright | kv_both)) == LEMPTY);
			break;
		case CONFIG_CONN:
			ok &= pexpect((k->field >= CONFIG_CONN_KEYWORD_FLOOR &&
				       k->field < CONFIG_CONN_KEYWORD_ROOF) ||
				      k->field == KNCF_OBSOLETE);
			break;
		}

		/* above checked k->field in range; check things,
		 * notably aliases, point back to a real NAME */
		ok &= pexpect(k->field < keywords->len);
		ok &= pexpect(group == OBSOLETE ? keywords->item[k->field].keyname == NULL/*entry 0*/ :
			      keywords->item[k->field].keyname != NULL);
		ok &= pexpect(group == OBSOLETE ? keywords->item[k->field].field == 0/*entry 0*/ :
			      keywords->item[k->field].field == k->field);
		ok &= pexpect(group == OBSOLETE ? keywords->item[k->field].validity == 0/*entry 0*/ :
			      keywords->item[k->field].validity == (k->validity & ~kv_alias));

		if (!ok) {
			llog_pexpect(logger, HERE, "[%u:%u] '%s' (follows '%s') expecting %s-%s",
				     i, k->field,
				     k->keyname,
				     (i > 0 ? keywords->item[i-1].keyname : "???"),
				     (section == CONFIG_SETUP ? "setup" :
				      section == CONFIG_CONN ? "conn" :
				      "???"),
				     (group == NAME ? "name" :
				      group == ALIAS ? "alias" :
				      group == OBSOLETE ? "obsolete" :
				      "???"));
			break;
		}
	}
}

static void check_ipsec_conf_keywords(struct logger *logger)
{
	check_config_keywords(logger, CONFIG_SETUP, &config_setup_keywords);
	check_config_keywords(logger, CONFIG_CONN, &config_conn_keywords);
}


struct starter_config *confread_load(const char *file,
				     bool setuponly,
				     struct logger *logger,
				     unsigned verbosity)
{
	check_ipsec_conf_keywords(logger);

	/**
	 * Load file
	 */
	struct ipsec_conf *cfgp = load_ipsec_conf(file, logger, setuponly, verbosity);
	if (cfgp == NULL)
		return NULL;

	struct starter_config *cfg = alloc_starter_config();

	/**
	 * Load setup
	 */
	if (!parse_ipsec_conf_config_setup(cfgp, logger)) {
		pfree_ipsec_conf(&cfgp);
		confread_free(cfg);
		return NULL;
	}

	if (!setuponly) {
		if (!parse_ipsec_conf_config_conn(cfg, cfgp, logger)) {
			pfree_ipsec_conf(&cfgp);
			confread_free(cfg);
			return NULL;
		}
	}

	pfree_ipsec_conf(&cfgp);
	return cfg;
}

bool parse_ipsec_conf_config_conn(struct starter_config *cfg,
				  struct ipsec_conf *cfgp,
				  struct logger *logger)
{
	struct section_list *sconn;

	/*
	 * Load %default conn
	 *
	 * ??? is it correct to accept multiple %default conns?
	 *
	 * XXX: yes, apparently it's a feature
	 */
	TAILQ_FOREACH(sconn, &cfgp->sections, next) {
		if (streq(sconn->name, "%default")) {
			/*
			 * Is failing to load default conn
			 * fatal?
			 */
			ldbg(logger, "loading default conn");
			if (!load_conn(&cfg->conn_default,
				       cfgp, sconn,
				       /*also=*/false,
				       /*default conn*/true,
				       logger)) {
				break;
			}
		}
	}

	/*
	 * Load other conns
	 */
	TAILQ_FOREACH(sconn, &cfgp->sections, next) {
		if (streq(sconn->name, "%default")) {
			/* %default processed above */
			continue;
		}

		ldbg(logger, "loading conn %s", sconn->name);
		struct starter_conn *conn = alloc_add_conn(cfg, sconn->name);
		if (!load_conn(conn, cfgp, sconn,
			       /*also*/true,
			       /*default-conn*/false,
			       logger)) {
			/* ??? should caller not log perrl? */
			continue;
		}

		conn->state = STATE_LOADED;
	}
	return true;
}

struct starter_config *confread_argv(const char *name,
				     char *argv[], int start,
				     struct logger *logger)
{
	check_ipsec_conf_keywords(logger);

	/**
	 * Load file
	 */
	struct ipsec_conf *cfgp = argv_ipsec_conf(name, argv, start, logger);
	if (cfgp == NULL)
		return NULL;

	struct starter_config *cfg = alloc_starter_config();
	if (cfg == NULL) {
		pfree_ipsec_conf(&cfgp);
		return NULL;
	}

	/*
	 * Load other conns
	 */

	if (!parse_ipsec_conf_config_conn(cfg, cfgp, logger)) {
		pfree_ipsec_conf(&cfgp);
		confread_free(cfg);
	}

	pfree_ipsec_conf(&cfgp);
	return cfg;
}

static void confread_free_conn_content(struct starter_conn *conn)
{
	/* Free all strings */

	/*
	 * Note: string fields in struct starter_end and struct starter_conn
	 * should correspond to STR_FIELD calls in copy_conn_default() and confread_free_conn.
	 */

# define STR_FIELD(f)  { pfreeany(conn->f); }

	STR_FIELD(name);

	for (unsigned i = 0; i < elemsof(conn->values); i++)
		STR_FIELD(values[i].string);

	/* handle starter_end strings */

# define STR_FIELD_END(f) { STR_FIELD(end[LEFT_END].f); STR_FIELD(end[RIGHT_END].f); }

	for (unsigned i = 0; i < elemsof(conn->end[LEFT_END].values); i++)
		STR_FIELD_END(values[i].string);

# undef STR_FIELD_END

# undef STR_FIELD
}

void confread_free(struct starter_config *cfg)
{
	confread_free_conn_content(&cfg->conn_default);

	struct starter_conn *conn;
	while ((conn = TAILQ_FIRST(&cfg->conns)) != NULL) {
		TAILQ_REMOVE(&cfg->conns, conn, link);
		confread_free_conn_content(conn);
		pfree(conn);
	}

	pfree(cfg);
}
