/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

/* we just get included from bseparser.c
 */


/* --- parser tables --- */
static	BseParserSymbol		sample_binary_appendix_symbols[] =
{
  { "value-block",		parse_value_block,	JOB_ID_NONE },
};
static	BseParserSymbol		sample_value_block_symbols[] =
{
  { "big-endian",		parse_flag,		JOB_ID_BIG_ENDIAN },
  { "little-endian",		parse_flag,		JOB_ID_LITTLE_ENDIAN },
  { "byte-size",		parse_number,		JOB_ID_BLOCK_BYTE_SIZE },
  { "recording-note",		parse_note,		JOB_ID_REC_NOTE },
  { "n-values",			parse_number,		JOB_ID_BLOCK_N_VALUES },
};
static	BseParserSymbol		sample_sample_symbols[] =
{
  { "blurb",			parse_string,		JOB_ID_SAMPLE_BLURB },
  { "type-blurb",		parse_string,		JOB_ID_SAMPLE_TYPE_BLURB },
  { "created",                  parse_date,             JOB_ID_SAMPLE_CREATED },
  { "modified",                 parse_date,             JOB_ID_SAMPLE_MODIFIED },
  { "n-channels",		parse_number,		JOB_ID_SAMPLE_N_CHANNELS },
  { "recording-freq",		parse_number,		JOB_ID_SAMPLE_REC_FREQ },
  { "octave-shift",		parse_number,		JOB_ID_OCTAVE_SHIFT },
  { "munk",			parse_munk,		JOB_ID_NONE },
};
static	BseParserSymbol		sample_munk_symbols[] =
{
  { "loop",			parse_loop,		JOB_ID_NONE },
  { "block",			parse_block,		JOB_ID_NONE },
};


/* --- functions --- */
static BseErrorType
read_block (int		   fd,
	    BseParserBlock *pblock)
{
  static const guint buffer_len = BSE_DFL_LOAD_BUFFER_SIZE;
  guchar buffer[buffer_len];
  register guint i;
  register guint left;
  register BseSampleValue *v;
  register BseValueBlock *vblock;
  register BseErrorType error;

  g_assert (pblock->vblock == NULL);
  vblock = bse_value_block_alloc (pblock->n_values);
  
  left = vblock->n_values * pblock->byte_size;
  
  v = vblock->values;
  error = BSE_ERROR_NONE;
  while (left)
    {
      register guint n;
      
      n = buffer_len;
      if (n > left)
	n = left;
      
      if (read (fd, buffer, n) != n)
	{
	  error = BSE_ERROR_FILE_TOO_SHORT;
	  break;
	}
      
      left -= n;
      
      if (pblock->byte_size == 1)
	for (i = 0; i < n; i++)
	  *(v++) = buffer[i] << 8;
      else if (pblock->byte_size == 2)
	{
	  if (pblock->big_endian)
	    for (i = 0; i < n - 1; i += 2)
	      {
		*(v++) = ((buffer[i] << 8) | buffer[i + 1]);
		/* *(v++) = ((buffer[i + 1] << 8) | buffer[i]); */
	      }
	  else
	    for (i = 0; i < n - 1; i += 2)
	      {
		*(v++) = ((buffer[i + 1] << 8) | buffer[i]);
		/* *(v++) = ((buffer[i] << 8) | buffer[i + 1]); */
	      }
	}
      else
	{
	  error = BSE_ERROR_SUB_HEADER_CORRUPT;
	  break;
	}
    }

  if (error != BSE_ERROR_NONE)
    bse_value_block_free (vblock);
  else
    {
      pblock->vblock = bse_value_block_setup (vblock,
					      pblock->identifier,
					      pblock->possible_loop_begin,
					      pblock->possible_loop_end);
    }
  
  return error;
}

/* parse a "(anytype-sample" construct by setting up the sample
 * structure and giving control to the symbol parser again.
 */
static guint
parse_sample (GScanner		*scanner,
	      BseParserSymbol	*symbol,
	      guint		job_id,
	      gpointer		struct_data)
{
  register BseSample	*sample;
  register guint	expected_token;
  register gchar	*sample_name;
  register gboolean	sample_valid;
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_STRING)
    return G_TOKEN_STRING;
  
  sample_name = scanner->value.v_string;
  bse_sample_name_make_valid (sample_name);
  sample_name = parser_check_sample_name (scanner, sample_name);

  if (!sample_name)
    {
      parser_skip_statement (scanner, 1);

      return G_TOKEN_NONE;
    }
  
  switch (job_id)
    {
    case	JOB_ID_NOTE_MUNKS:
      sample = bse_sample_new (BSE_SAMPLE_NOTE_MUNKS, sample_name, scanner->input_name);
      break;
      
    case	JOB_ID_EFFECT_MUNKS:
      sample = bse_sample_new (BSE_SAMPLE_EFFECT_MUNKS, sample_name, scanner->input_name);
      break;
      
    default:
      g_assert_not_reached ();
      sample = NULL;
    }
  g_free (sample_name);
  sample_name = NULL;
  
  /* we use the internal pointer as GList* of munk definitions
   */
  g_dataset_id_set_data (sample, KEYID_MUNK_QUEUE, NULL);
  g_dataset_id_set_data (sample, KEYID_NOTE_SHIFT, 0);
  
  expected_token = parser_parse_rest (scanner, sample_sample_symbols, sample);
  
  sample_valid = expected_token != G_TOKEN_ERROR;
  if (!sample->n_channels)
    {
      g_scanner_error (scanner, "sample without n-channels definition");
      sample_valid = FALSE;
    }
  else if (!sample->recording_freq)
    {
      g_scanner_error (scanner, "sample without recording-freq definition");
      sample_valid = FALSE;
    }
  else if (!g_dataset_id_get_data (sample, KEYID_MUNK_QUEUE))
    {
      g_scanner_error (scanner, "sample without munk definitions");
      sample_valid = FALSE;
    }
  
  if (sample_valid)
    PDATA (scanner)->new_samples = g_slist_prepend (PDATA (scanner)->new_samples, sample);
  else
    {
      register BseParserMunk *pmunk;
      
      pmunk = g_dataset_id_get_data (sample, KEYID_MUNK_QUEUE);
      g_dataset_id_remove_data (sample, KEYID_MUNK_QUEUE);
      while (pmunk)
	{
	  register BseParserMunk *next;

	  next = pmunk->next;
	  g_free (pmunk);
	  pmunk = next;
	}
      
      bse_sample_unref (sample);
    }
  
  return expected_token;
}

static	guint
parse_binary_appendix (GScanner		  *scanner,
		       BseParserSymbol	  *symbol,
		       guint		   job_id,
		       gpointer		   struct_data)
{
  register guint	   expected_token;
  
  g_assert (PDATA (scanner)->in_appendix_definition == FALSE);
  
  if (PDATA (scanner)->binary_pblock_list)
    {
      g_scanner_warn (scanner, "binary appendices already defined");
      parser_skip_statement (scanner, 1);
      return G_TOKEN_NONE;
    }
  
  PDATA (scanner)->in_appendix_definition = TRUE;
  
  expected_token = parser_parse_rest (scanner, sample_binary_appendix_symbols, NULL);
  
  PDATA (scanner)->in_appendix_definition = FALSE;
  
  if (!PDATA (scanner)->binary_pblock_list)
    g_scanner_warn (scanner, "no binary appendices defined");
  
  return expected_token;
}

static	guint
parse_value_block (GScanner		  *scanner,
		   BseParserSymbol	  *symbol,
		   guint		  job_id,
		   gpointer		  struct_data)
{
  register BseParserBlock *pblock;
  register guint	   expected_token;
  gchar                    *string;

  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_STRING)
    return G_TOKEN_STRING;

  string = g_strconcat (scanner->input_name, "::", scanner->value.v_string, NULL);
  if (bse_value_block_get_closest (string, 0, 0, 0))
    {
      g_scanner_warn (scanner,
		      "value block \"%s\" already exists",
		      string);
      g_free (string);
      parser_skip_statement (scanner, 1);

      return G_TOKEN_NONE;
    }

  pblock = g_new0 (BseParserBlock, 1);
  if (PDATA (scanner)->in_appendix_definition)
    {
      guint nth;

      nth = g_slist_length (PDATA (scanner)->binary_pblock_list);
      nth++;
      pblock->block_type = BSE_BLOCK_BINARY_APPENDIX;
      pblock->nth_appendix = nth;

      PDATA (scanner)->binary_pblock_list = g_slist_append (PDATA (scanner)->binary_pblock_list, pblock);
    }
  else
    {
      pblock->block_type = BSE_BLOCK_ZERO;
      pblock->nth_appendix = 0;
    }
  pblock->recording_note = BSE_KAMMER_NOTE;
  pblock->n_values = 0;
  pblock->byte_size = 0;
  pblock->big_endian = FALSE;
  pblock->vblock = NULL;
  pblock->identifier = string;
  pblock->possible_loop_begin = 0;
  pblock->possible_loop_end = 0;

  PDATA (scanner)->pblock_list = g_slist_prepend (PDATA (scanner)->pblock_list, pblock);
  
  expected_token = parser_parse_rest (scanner, sample_value_block_symbols, pblock);
  
  return expected_token;
}

static	guint
parse_munk (GScanner		*scanner,
	    BseParserSymbol	*symbol,
	    guint		job_id,
	    gpointer		struct_data)
{
  register BseSample *sample;
  register BseParserMunk *pmunk;
  register guint expected_token;
  register gboolean valid_munk;
  guint note;
  
  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);
  sample = struct_data;
  
  if (!sample->n_channels)
    {
      g_scanner_error (scanner, "munk definition preceeds n-channels definition");
      parser_skip_statement (scanner, 1);
      
      return G_TOKEN_NONE;
    }

  expected_token = parser_scan_note (scanner, &note, NULL);
  if (expected_token != G_TOKEN_NONE)
    return expected_token;

  note = CLAMP (note + (gint) g_dataset_id_get_data (sample, KEYID_NOTE_SHIFT),
		BSE_MIN_NOTE,
		BSE_MAX_NOTE);
  
  pmunk = g_new (BseParserMunk, 1);
  pmunk->sample = sample;
  pmunk->note = note;
  pmunk->loop_begin = 0;
  pmunk->loop_end = 0;
  pmunk->block_offset = 0;
  pmunk->n_values = 0;
  pmunk->pblock = NULL;
  pmunk->next = NULL;
  
  expected_token = parser_parse_rest (scanner, sample_munk_symbols, pmunk);
  
  valid_munk = expected_token != G_TOKEN_ERROR;
  if (!pmunk->pblock || !pmunk->n_values)
    {
      g_scanner_error (scanner, "munk without value definition");
      valid_munk = FALSE;
    }
  else if (pmunk->block_offset >= pmunk->pblock->n_values / sample->n_channels)
    {
      g_scanner_error (scanner, "munk offset (%d) exceeds block size (%d)",
		       pmunk->block_offset,
		       pmunk->pblock->n_values / sample->n_channels);
      valid_munk = FALSE;
    }
  else if (pmunk->loop_begin > pmunk->pblock->n_values / sample->n_channels - pmunk->block_offset ||
	   pmunk->loop_end > pmunk->pblock->n_values / sample->n_channels - pmunk->block_offset)
    {
      g_scanner_warn (scanner, "munk loop [%d, %d] exceeds block size (%d)",
		      pmunk->loop_begin,
		      pmunk->loop_end,
		      pmunk->pblock->n_values / sample->n_channels - pmunk->block_offset);
      pmunk->loop_begin = 0;
      pmunk->loop_end = 0;
      pmunk->pblock->possible_loop_begin = 0;
      pmunk->pblock->possible_loop_end = 0;
    }
  
  if (!valid_munk)
    g_free (pmunk);
  else
    {
      pmunk->next = g_dataset_id_get_data (sample, KEYID_MUNK_QUEUE);
      g_dataset_id_set_data (sample, KEYID_MUNK_QUEUE, pmunk);
    }
  
  return expected_token;
}

static guint
parse_block (GScanner		*scanner,
	     BseParserSymbol	*symbol,
	     guint		job_id,
	     gpointer		struct_data)
{
  register BseParserMunk *pmunk;
  register BseParserBlock *pblock;
  register gchar *identifier;
  register guint block_offset;
  register guint n_values;
  register GSList *slist;
  
  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);
  pmunk = struct_data;
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_STRING)
    return G_TOKEN_STRING;

  identifier = g_strconcat (scanner->input_name, "::", scanner->value.v_string, NULL);

  pblock = NULL;
  for (slist = PDATA (scanner)->pblock_list; slist; slist = slist->next)
    {
      register BseParserBlock *tmp;

      tmp = slist->data;
      if (g_str_equal (tmp->identifier, identifier))
	{
	  pblock = tmp;
	  break;
	}
    }
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_INT &&
      scanner->token != ')')
    {
      g_free (identifier);
      return ')';
    }
  if (scanner->token != G_TOKEN_INT)
    {
      block_offset = 0;
      n_values = 0;
    }
  else
    {
      block_offset = scanner->value.v_int;
      
      g_scanner_get_next_token (scanner);
      if (scanner->token != G_TOKEN_INT &&
	  scanner->token != ')')
	{
	  g_free (identifier);
	  return ')';
	}
      if (scanner->token == G_TOKEN_INT)
	n_values = scanner->value.v_int;
      else
	n_values = 0;

      g_scanner_get_next_token (scanner);
    }
  if (n_values == 0 && pblock)
    n_values = pblock->n_values / pmunk->sample->n_channels;

  if (!pblock)
    g_scanner_warn (scanner, "munk refers to undefined block: `%s'", identifier);
  else if (pblock->byte_size != 2)
    {
      g_scanner_warn (scanner, "munk refers to block with invalid byte size (%d)", pblock->byte_size);
      pblock = NULL;
    }
  else if (block_offset >= pblock->n_values / pmunk->sample->n_channels)
    {
      g_scanner_warn (scanner, "munk offset (%d) exceeds block size (%d)",
		      block_offset,
		      pblock->n_values / pmunk->sample->n_channels);
      pblock = NULL;
    }
  else if (n_values > pblock->n_values / pmunk->sample->n_channels - block_offset)
    {
      g_scanner_warn (scanner, "munk size (%d) exceeds block size (%d)",
		      n_values,
		      pblock->n_values / pmunk->sample->n_channels - block_offset);
      n_values = pblock->n_values / pmunk->sample->n_channels - block_offset;
    }
  g_free (identifier);
  
  if (pblock)
    {
      pmunk->block_offset = 0;
      pmunk->block_offset = block_offset;
      pmunk->n_values = n_values * pmunk->sample->n_channels;
      pmunk->pblock = pblock;
      pblock->possible_loop_begin = pmunk->loop_begin;
      pblock->possible_loop_end = pmunk->loop_end;
    }
  
  if (scanner->token != ')')
    return ')';
  
  return G_TOKEN_NONE;
}

static guint
parse_loop (GScanner		*scanner,
	    BseParserSymbol	*symbol,
	    guint		job_id,
	    gpointer		struct_data)
{
  register BseParserMunk *pmunk;
  register gint begin;
  register gint end;
  
  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);
  pmunk = struct_data;
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_INT)
    return G_TOKEN_INT;
  
  g_scanner_peek_next_token (scanner);
  if (scanner->next_token != G_TOKEN_INT)
    {
      g_scanner_get_next_token (scanner);
      
      return G_TOKEN_INT;
    }
  
  begin = scanner->value.v_int;
  g_scanner_get_next_token (scanner);
  end = scanner->value.v_int;
  
  if (begin < 0 || end < 0 ||
      begin > end)
    g_scanner_error (scanner, "munk holds invalid loop definition [%d, %d]", begin, end);
  else
    {
      pmunk->loop_begin = begin;
      pmunk->loop_end = end;
      if (pmunk->pblock)
	{
	  pmunk->pblock->possible_loop_begin = begin;
	  pmunk->pblock->possible_loop_end = end;
	}
    }
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != ')')
    return ')';
  
  return G_TOKEN_NONE;
}
