/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#ifndef	__BSE_STREAM_H__
#define	__BSE_STREAM_H__

#include	<bse/bsebase.h>


#ifdef __cplusplus
extern "C" {
#pragma }
#endif /* __cplusplus */



/* Private, implementation dependant BseStream portion
 */
typedef	struct _BseStreamPrivate	BseStreamPrivate;


/* BSE Stream attributes
 */
struct _BseStreamAttribs
{
  guint			n_channels;
  guint			play_frequency;
  guint			record_frequency;
  guint			fragment_size;
};


/* BSE Stream
 */
struct _BseStream
{
  gchar			*type_name;

  guint			ref_count;

  gchar                 *name;

  /* BseStream flags (packed)
   */
  guint			opened : 1;
  guint			readable : 1;
  guint			writable : 1;
  guint                 suspended : 1;
  guint			can_block : 1;

  BseStreamPrivate      *private;

  /* Constant Values
   */
  guint max_channels;
  guint min_play_frequency;
  guint max_play_frequency;
  guint min_record_frequency;
  guint max_record_frequency;
  guint min_fragment_size;
  guint	max_fragment_size;

  /* Stream attributes
   */
  BseStreamAttribs	attribs;

  /* Implementation dependant functions
   */
  BseErrorType  (*open)                 (BseStream       *stream,
					 gboolean         read_access,
					 gboolean         write_access);
  void          (*close)                (BseStream	 *stream);
  BseErrorType  (*start)                (BseStream       *stream);
  BseErrorType  (*suspend)              (BseStream       *stream);
  gboolean      (*would_block)          (BseStream       *stream,
					 guint            n_values);
  BseErrorType  (*read)                 (BseStream       *stream,
					 guint            n_values,
					 BseSampleValue  *values);
  BseErrorType  (*write)                (BseStream       *stream,
					 guint            n_values,
					 BseSampleValue  *values);
  BseErrorType  (*set_attribs)          (BseStream       *stream,
					 BseStreamAttribMask mask,
					 BseStreamAttribs *attribs);
};



/* --- prototypes -- */
BseErrorType	bse_stream_open			(BseStream	 *stream,
						 const gchar	 *name,
						 gboolean	  read_access,
						 gboolean	  write_access);
void		bse_stream_ref			(BseStream	 *stream);
void		bse_stream_unref		(BseStream	 *stream);
BseErrorType	bse_stream_start		(BseStream	 *stream);
BseErrorType	bse_stream_suspend		(BseStream	 *stream);
gboolean	bse_stream_would_block		(BseStream	 *stream,
						 guint		  n_values);
BseErrorType	bse_stream_read			(BseStream	 *stream,
						 guint		  n_values,
						 BseSampleValue	 *values);
BseErrorType	bse_stream_write		(BseStream	 *stream,
						 guint		  n_values,
						 BseSampleValue	 *values);
BseErrorType	bse_stream_set_attribs		(BseStream	 *stream,
						 BseStreamAttribMask mask,
						 BseStreamAttribs *attribs);

BseStream*      bse_null_stream_new		(void);







#ifdef __cplusplus
#pragma {
}
#endif /* __cplusplus */

#endif /* __BSE_STREAM_H__ */
