/* BEAST - Bedevilled Audio System
 * Copyright (C) 2002 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * A copy of the GNU Lesser General Public License should ship along
 * with this library; if not, see http://www.gnu.org/copyleft/.
 */
#include "bstasciipixbuf.h"


/* --- prototypes --- */
static GdkPixbuf*	ascii_pixbuf_create	(void);


/* --- variables --- */
static GdkPixbuf *ascii_pixbuf = NULL;


/* --- functions --- */
void
bst_ascii_pixbuf_ref (void)
{
  if (!ascii_pixbuf)
    {
      ascii_pixbuf = ascii_pixbuf_create ();
      g_object_set_data_full (G_OBJECT (ascii_pixbuf), "location", &ascii_pixbuf, (GDestroyNotify) g_nullify_pointer);
    }
  else
    g_object_ref (ascii_pixbuf);
}

static void
bst_ascii_pixbuf_child_destroy (guint8  *pixels,
				gpointer data)
{
  GdkPixbuf *pixbuf = data;
  
  g_return_if_fail (pixbuf == ascii_pixbuf);

  g_object_unref (pixbuf);
}

GdkPixbuf*
bst_ascii_pixbuf_new (gchar character,
		      guint char_width,
		      guint char_height)
{
  GdkPixbuf *pixbuf;
  guint width, i;
  guint8 *pixels;

  g_return_val_if_fail (character >= 32 && character <= 126, NULL);

  bst_ascii_pixbuf_ref ();
  width = gdk_pixbuf_get_width (ascii_pixbuf);
  width /= 126 - 32 + 1;
  pixels = gdk_pixbuf_get_pixels (ascii_pixbuf);
  i = (character - 32) * width * gdk_pixbuf_get_n_channels (ascii_pixbuf);
  pixbuf = gdk_pixbuf_new_from_data (pixels + i, GDK_COLORSPACE_RGB,
				     gdk_pixbuf_get_has_alpha (ascii_pixbuf),
				     gdk_pixbuf_get_bits_per_sample (ascii_pixbuf),
				     width,
				     gdk_pixbuf_get_height (ascii_pixbuf),
				     gdk_pixbuf_get_rowstride (ascii_pixbuf),
				     bst_ascii_pixbuf_child_destroy, g_object_ref (ascii_pixbuf));
  if (!char_width)
    char_width = width;
  if (!char_height)
    char_height = gdk_pixbuf_get_height (pixbuf);
  if (char_width != width || char_height != gdk_pixbuf_get_height (pixbuf))
    {
      GdkPixbuf *tmp_pixbuf = pixbuf;

      pixbuf = gdk_pixbuf_scale_simple (pixbuf, char_width, char_height, GDK_INTERP_HYPER);
      g_object_unref (tmp_pixbuf);
    }
  bst_ascii_pixbuf_unref ();
  
  return pixbuf;
}

void
bst_ascii_pixbuf_unref (void)
{
  g_return_if_fail (ascii_pixbuf != NULL);

  g_object_unref (ascii_pixbuf);
}


/* --- image data --- */

/* inlined pixbuf image containing character bitmaps for ascii 32 up to ascii 126.
 * created with the gimp, by drawing a text string:
 *   for i in `seq 32 126` ; do OCT=`printf '\%o' $i` ; echo -ne "$OCT" ; done ; echo
 * without anti-aliasing, using the font fixed/misc/medium[C]/24Pt.
 * further mods where cutting the slash out of the 0 and cropping the image
 * into an apropriate size.
 * image size is width=95*12 by height=24, characters are 12 pixels wide
 */


/* GdkPixbuf RGB C-Source image dump 1-byte-run-length-encoded */

static const guint8 ascii_font_pixdata[] = 
{ ""
  /* Pixbuf magic (0x47646b50) */
  "GdkP"
  /* length: header (24) + pixel_data (16635) */
  "\0\0A\23"
  /* pixdata_type (0x2010002) */
  "\2\1\0\2"
  /* rowstride (4560) */
  "\0\0\21\320"
  /* width (1140) */
  "\0\0\4t"
  /* height (24) */
  "\0\0\0\30"
  /* pixel_data: */
  "\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0"
  "\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377"
  "\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0"
  "\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0"
  "\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\327\0\0\0\0\202\0\0\0\377\377"
  "\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0"
  "\377\0\0\0\0\377\0\0\0\0\372\0\0\0\0\202\0\0\0\377\366\0\0\0\0\202\0"
  "\0\0\377\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0"
  "\260\0\0\0\0\202\0\0\0\377\377\0\0\0\0\377\0\0\0\0\317\0\0\0\0\202\0"
  "\0\0\377\270\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\267\0\0"
  "\0\0\204\0\0\0\377\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\360"
  "\0\0\0\0\205\0\0\0\377\222\0\0\0\0\205\0\0\0\377\212\0\0\0\0\1\0\0\0"
  "\377\226\0\0\0\0\202\0\0\0\377\377\0\0\0\0\377\0\0\0\0\305\0\0\0\0\203"
  "\0\0\0\377\210\0\0\0\0\202\0\0\0\377\210\0\0\0\0\203\0\0\0\377\211\0"
  "\0\0\0\203\0\0\0\377\204\0\0\0\0\1\0\0\0\377\222\0\0\0\0\202\0\0\0\377"
  "\210\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202"
  "\0\0\0\377\202\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\207\0"
  "\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\207\0\0\0\0\203\0\0\0"
  "\377\210\0\0\0\0\204\0\0\0\377\213\0\0\0\0\202\0\0\0\377\206\0\0\0\0"
  "\202\0\0\0\377\313\0\0\0\0\202\0\0\0\377\206\0\0\0\0\204\0\0\0\377\211"
  "\0\0\0\0\202\0\0\0\377\211\0\0\0\0\205\0\0\0\377\207\0\0\0\0\205\0\0"
  "\0\377\211\0\0\0\0\202\0\0\0\377\206\0\0\0\0\210\0\0\0\377\206\0\0\0"
  "\0\205\0\0\0\377\205\0\0\0\0\211\0\0\0\377\205\0\0\0\0\204\0\0\0\377"
  "\210\0\0\0\0\205\0\0\0\377\303\0\0\0\0\204\0\0\0\377\210\0\0\0\0\205"
  "\0\0\0\377\207\0\0\0\0\205\0\0\0\377\205\0\0\0\0\206\0\0\0\377\210\0"
  "\0\0\0\205\0\0\0\377\205\0\0\0\0\207\0\0\0\377\205\0\0\0\0\211\0\0\0"
  "\377\203\0\0\0\0\211\0\0\0\377\205\0\0\0\0\205\0\0\0\377\205\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\204\0\0\0\0\206\0\0\0\377\214"
  "\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377"
  "\205\0\0\0\0\205\0\0\0\377\205\0\0\0\0\207\0\0\0\377\207\0\0\0\0\205"
  "\0\0\0\377\205\0\0\0\0\207\0\0\0\377\207\0\0\0\0\205\0\0\0\377\204\0"
  "\0\0\0\212\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\202\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\203\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205"
  "\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0"
  "\0\377\202\0\0\0\0\212\0\0\0\377\205\0\0\0\0\205\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\213\0\0\0\0\205\0\0\0\377\211\0\0\0\0\203\0\0\0\377"
  "\224\0\0\0\0\202\0\0\0\377\224\0\0\0\0\202\0\0\0\377\235\0\0\0\0\202"
  "\0\0\0\377\223\0\0\0\0\203\0\0\0\377\221\0\0\0\0\202\0\0\0\377\215\0"
  "\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\226\0\0\0\0\202\0\0\0"
  "\377\377\0\0\0\0\263\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377"
  "\212\0\0\0\0\202\0\0\0\377\207\0\0\0\0\205\0\0\0\377\202\0\0\0\0\202"
  "\0\0\0\377\222\0\0\0\0\202\0\0\0\377\210\0\0\0\0\202\0\0\0\377\202\0"
  "\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\206\0\0\0\377\204\0\0\0\0\205\0\0\0\377\202\0\0\0\0"
  "\202\0\0\0\377\206\0\0\0\0\205\0\0\0\377\211\0\0\0\0\202\0\0\0\377\212"
  "\0\0\0\0\202\0\0\0\377\210\0\0\0\0\202\0\0\0\377\312\0\0\0\0\202\0\0"
  "\0\377\205\0\0\0\0\206\0\0\0\377\207\0\0\0\0\203\0\0\0\377\210\0\0\0"
  "\0\207\0\0\0\377\205\0\0\0\0\207\0\0\0\377\210\0\0\0\0\202\0\0\0\377"
  "\206\0\0\0\0\210\0\0\0\377\205\0\0\0\0\207\0\0\0\377\204\0\0\0\0\211"
  "\0\0\0\377\204\0\0\0\0\206\0\0\0\377\206\0\0\0\0\207\0\0\0\377\301\0"
  "\0\0\0\206\0\0\0\377\206\0\0\0\0\207\0\0\0\377\205\0\0\0\0\207\0\0\0"
  "\377\204\0\0\0\0\207\0\0\0\377\206\0\0\0\0\207\0\0\0\377\204\0\0\0\0"
  "\210\0\0\0\377\204\0\0\0\0\211\0\0\0\377\203\0\0\0\0\211\0\0\0\377\204"
  "\0\0\0\0\207\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\204\0\0\0\0\206\0\0\0\377\214\0\0\0\0\202\0\0\0\377\203\0\0\0"
  "\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377"
  "\212\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203"
  "\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0\0\207\0\0\0\377\204\0"
  "\0\0\0\210\0\0\0\377\205\0\0\0\0\207\0\0\0\377\204\0\0\0\0\210\0\0\0"
  "\377\205\0\0\0\0\207\0\0\0\377\203\0\0\0\0\212\0\0\0\377\203\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\206"
  "\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\202\0\0\0"
  "\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\202\0\0\0\0\212\0\0\0\377"
  "\205\0\0\0\0\202\0\0\0\377\210\0\0\0\0\202\0\0\0\377\216\0\0\0\0\202"
  "\0\0\0\377\210\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\223\0\0"
  "\0\0\202\0\0\0\377\224\0\0\0\0\202\0\0\0\377\235\0\0\0\0\202\0\0\0\377"
  "\222\0\0\0\0\204\0\0\0\377\221\0\0\0\0\202\0\0\0\377\215\0\0\0\0\202"
  "\0\0\0\377\212\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\215\0"
  "\0\0\0\202\0\0\0\377\336\0\0\0\0\202\0\0\0\377\322\0\0\0\0\202\0\0\0"
  "\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\207\0\0\0\0"
  "\1\0\0\0\377\202\0\0\0\0\206\0\0\0\377\222\0\0\0\0\202\0\0\0\377\210"
  "\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0"
  "\0\377\202\0\0\0\0\202\0\0\0\377\204\0\0\0\0\211\0\0\0\377\202\0\0\0"
  "\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\206"
  "\0\0\0\0\203\0\0\0\377\1\0\0\0\0\203\0\0\0\377\210\0\0\0\0\202\0\0\0"
  "\377\211\0\0\0\0\203\0\0\0\377\210\0\0\0\0\203\0\0\0\377\212\0\0\0\0"
  "\202\0\0\0\377\274\0\0\0\0\202\0\0\0\377\205\0\0\0\0\203\0\0\0\377\202"
  "\0\0\0\0\203\0\0\0\377\205\0\0\0\0\204\0\0\0\377\207\0\0\0\0\202\0\0"
  "\0\377\204\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0"
  "\0\203\0\0\0\377\206\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377"
  "\212\0\0\0\0\203\0\0\0\377\204\0\0\0\0\202\0\0\0\377\211\0\0\0\0\202"
  "\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0"
  "\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\277\0\0\0\0\203\0\0\0"
  "\377\202\0\0\0\0\203\0\0\0\377\204\0\0\0\0\203\0\0\0\377\204\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203"
  "\0\0\0\0\202\0\0\0\377\203\0\0\0\0\203\0\0\0\377\204\0\0\0\0\203\0\0"
  "\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0"
  "\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377"
  "\212\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\216\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\202\0\0\0\377\212\0\0\0\0\203\0\0\0\377\203\0\0\0\0"
  "\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\204\0\0\0\0\202\0\0\0\377\203"
  "\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0"
  "\0\377\204\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0"
  "\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\203\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\203\0\0\0\377\206\0\0\0\0\202"
  "\0\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\202\0"
  "\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\203\0\0\0\0"
  "\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\213"
  "\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\211\0\0\0\0\202\0\0"
  "\0\377\215\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\203\0\0\0"
  "\0\202\0\0\0\377\222\0\0\0\0\204\0\0\0\377\222\0\0\0\0\202\0\0\0\377"
  "\235\0\0\0\0\202\0\0\0\377\222\0\0\0\0\202\0\0\0\377\223\0\0\0\0\202"
  "\0\0\0\377\242\0\0\0\0\202\0\0\0\377\215\0\0\0\0\202\0\0\0\377\336\0"
  "\0\0\0\202\0\0\0\377\322\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0"
  "\377\212\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\204\0\0\0\0"
  "\203\0\0\0\377\223\0\0\0\0\202\0\0\0\377\210\0\0\0\0\202\0\0\0\377\202"
  "\0\0\0\0\202\0\0\0\377\205\0\0\0\0\211\0\0\0\377\202\0\0\0\0\202\0\0"
  "\0\377\202\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\202\0\0\0"
  "\0\205\0\0\0\377\1\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\203"
  "\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0"
  "\0\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0"
  "\0\202\0\0\0\377\260\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377"
  "\204\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0"
  "\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0"
  "\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377"
  "\207\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\211\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0"
  "\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\206\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\216\0\0\0\0"
  "\202\0\0\0\377\216\0\0\0\0\202\0\0\0\377\213\0\0\0\0\1\0\0\0\377\205"
  "\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0"
  "\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377"
  "\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\212\0"
  "\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\216\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\206"
  "\0\0\0\0\202\0\0\0\377\212\0\0\0\0\204\0\0\0\377\1\0\0\0\0\204\0\0\0"
  "\377\203\0\0\0\0\204\0\0\0\377\203\0\0\0\0\202\0\0\0\377\203\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205"
  "\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0"
  "\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377"
  "\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202"
  "\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0"
  "\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0"
  "\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\212\0\0\0\0"
  "\203\0\0\0\377\205\0\0\0\0\202\0\0\0\377\211\0\0\0\0\202\0\0\0\377\215"
  "\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\221\0\0\0\0\204\0\0\0\377\211\0\0\0\0\204\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\235\0\0\0\0\202\0\0\0\377\222\0\0\0\0\202\0\0\0\377"
  "\223\0\0\0\0\202\0\0\0\377\242\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202"
  "\0\0\0\377\207\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\326\0"
  "\0\0\0\202\0\0\0\377\322\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0"
  "\377\212\0\0\0\0\202\0\0\0\377\242\0\0\0\0\202\0\0\0\377\223\0\0\0\0"
  "\211\0\0\0\377\202\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\207"
  "\0\0\0\0\203\0\0\0\377\1\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0"
  "\377\202\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\212\0\0\0\0"
  "\203\0\0\0\377\212\0\0\0\0\203\0\0\0\377\205\0\0\0\0\202\0\0\0\377\202"
  "\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0"
  "\0\377\257\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\204\0\0\0"
  "\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\216\0\0\0\0\202\0\0\0\377"
  "\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0"
  "\0\0\377\205\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\217\0\0"
  "\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377"
  "\204\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\205\0\0\0\0\204"
  "\0\0\0\377\210\0\0\0\0\204\0\0\0\377\213\0\0\0\0\203\0\0\0\377\220\0"
  "\0\0\0\203\0\0\0\377\217\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0"
  "\377\202\0\0\0\0\205\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204"
  "\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0"
  "\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377"
  "\206\0\0\0\0\202\0\0\0\377\216\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202"
  "\0\0\0\377\1\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\212\0\0"
  "\0\0\211\0\0\0\377\203\0\0\0\0\204\0\0\0\377\203\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\203\0\0\0\0\202\0\0\0\377\215\0\0\0\0\202\0\0\0\377\207\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204"
  "\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\205\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\206\0\0\0\0"
  "\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\212\0\0\0\0\203\0\0\0\377\206"
  "\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\214\0\0\0\0\202\0\0"
  "\0\377\241\0\0\0\0\202\0\0\0\377\211\0\0\0\0\206\0\0\0\377\204\0\0\0"
  "\0\202\0\0\0\377\202\0\0\0\0\203\0\0\0\377\207\0\0\0\0\206\0\0\0\377"
  "\206\0\0\0\0\204\0\0\0\377\1\0\0\0\0\202\0\0\0\377\205\0\0\0\0\205\0"
  "\0\0\377\210\0\0\0\0\202\0\0\0\377\211\0\0\0\0\203\0\0\0\377\202\0\0"
  "\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\202\0\0\0\0\203\0\0\0\377"
  "\210\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202"
  "\0\0\0\377\203\0\0\0\0\203\0\0\0\377\207\0\0\0\0\202\0\0\0\377\206\0"
  "\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\1\0\0\0\0\203\0\0\0\377"
  "\204\0\0\0\0\202\0\0\0\377\202\0\0\0\0\203\0\0\0\377\207\0\0\0\0\205"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\202\0\0\0\0\203\0\0\0\377\207\0"
  "\0\0\0\203\0\0\0\377\202\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0"
  "\377\202\0\0\0\0\203\0\0\0\377\206\0\0\0\0\207\0\0\0\377\205\0\0\0\0"
  "\206\0\0\0\377\205\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\202"
  "\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0"
  "\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\211\0\0\0\377\206\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202"
  "\0\0\0\377\212\0\0\0\0\202\0\0\0\377\242\0\0\0\0\202\0\0\0\377\224\0"
  "\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\205\0\0\0\0\205\0\0\0"
  "\377\213\0\0\0\0\202\0\0\0\377\210\0\0\0\0\205\0\0\0\377\223\0\0\0\0"
  "\202\0\0\0\377\214\0\0\0\0\202\0\0\0\377\205\0\0\0\0\203\0\0\0\377\1"
  "\0\0\0\0\202\0\0\0\377\1\0\0\0\0\203\0\0\0\377\206\0\0\0\0\202\0\0\0"
  "\377\257\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\204\0\0\0\0"
  "\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\215\0\0\0\0\202\0\0\0\377\212"
  "\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\207\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1\0\0\0\0\204"
  "\0\0\0\377\212\0\0\0\0\202\0\0\0\377\206\0\0\0\0\206\0\0\0\377\205\0"
  "\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\205\0\0\0\0\204\0\0\0"
  "\377\210\0\0\0\0\204\0\0\0\377\211\0\0\0\0\203\0\0\0\377\205\0\0\0\0"
  "\212\0\0\0\377\205\0\0\0\0\203\0\0\0\377\214\0\0\0\0\202\0\0\0\377\205"
  "\0\0\0\0\202\0\0\0\377\1\0\0\0\0\206\0\0\0\377\203\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\203\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205"
  "\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0"
  "\0\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\216\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\204\0\0\0\377\210\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202"
  "\0\0\0\377\1\0\0\0\0\203\0\0\0\377\1\0\0\0\0\202\0\0\0\377\203\0\0\0"
  "\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\203"
  "\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0"
  "\0\377\204\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\203\0\0\0\377"
  "\203\0\0\0\0\203\0\0\0\377\214\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0"
  "\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\206\0\0\0\0\206"
  "\0\0\0\377\211\0\0\0\0\203\0\0\0\377\207\0\0\0\0\202\0\0\0\377\212\0"
  "\0\0\0\202\0\0\0\377\214\0\0\0\0\202\0\0\0\377\253\0\0\0\0\202\0\0\0"
  "\377\204\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\1\0\0\0\0\205"
  "\0\0\0\377\205\0\0\0\0\207\0\0\0\377\205\0\0\0\0\210\0\0\0\377\204\0"
  "\0\0\0\207\0\0\0\377\205\0\0\0\0\206\0\0\0\377\206\0\0\0\0\205\0\0\0"
  "\377\1\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\1\0\0\0\0\205"
  "\0\0\0\377\207\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\207\0"
  "\0\0\0\202\0\0\0\377\202\0\0\0\0\203\0\0\0\377\210\0\0\0\0\202\0\0\0"
  "\377\206\0\0\0\0\212\0\0\0\377\203\0\0\0\0\202\0\0\0\377\1\0\0\0\0\205"
  "\0\0\0\377\205\0\0\0\0\207\0\0\0\377\204\0\0\0\0\202\0\0\0\377\1\0\0"
  "\0\0\205\0\0\0\377\205\0\0\0\0\205\0\0\0\377\1\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\1\0\0\0\0\205\0\0\0\377\204\0\0\0\0\211\0"
  "\0\0\377\204\0\0\0\0\206\0\0\0\377\205\0\0\0\0\202\0\0\0\377\205\0\0"
  "\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\211\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\213\0\0\0\0\202\0\0\0\377\213\0\0\0\0\202\0\0\0\377\241\0\0\0\0"
  "\202\0\0\0\377\224\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\206"
  "\0\0\0\0\206\0\0\0\377\210\0\0\0\0\202\0\0\0\377\210\0\0\0\0\205\0\0"
  "\0\377\224\0\0\0\0\202\0\0\0\377\214\0\0\0\0\202\0\0\0\377\207\0\0\0"
  "\0\206\0\0\0\377\210\0\0\0\0\202\0\0\0\377\256\0\0\0\0\202\0\0\0\377"
  "\207\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202"
  "\0\0\0\377\214\0\0\0\0\202\0\0\0\377\210\0\0\0\0\204\0\0\0\377\206\0"
  "\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\205\0\0\0\0\210\0\0\0"
  "\377\204\0\0\0\0\210\0\0\0\377\210\0\0\0\0\202\0\0\0\377\207\0\0\0\0"
  "\206\0\0\0\377\206\0\0\0\0\210\0\0\0\377\206\0\0\0\0\202\0\0\0\377\212"
  "\0\0\0\0\202\0\0\0\377\210\0\0\0\0\204\0\0\0\377\206\0\0\0\0\212\0\0"
  "\0\377\206\0\0\0\0\204\0\0\0\377\211\0\0\0\0\202\0\0\0\377\206\0\0\0"
  "\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\203"
  "\0\0\0\0\211\0\0\0\377\203\0\0\0\0\207\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0"
  "\0\207\0\0\0\377\205\0\0\0\0\207\0\0\0\377\205\0\0\0\0\202\0\0\0\377"
  "\212\0\0\0\0\211\0\0\0\377\206\0\0\0\0\202\0\0\0\377\216\0\0\0\0\202"
  "\0\0\0\377\203\0\0\0\0\203\0\0\0\377\211\0\0\0\0\202\0\0\0\377\212\0"
  "\0\0\0\202\0\0\0\377\1\0\0\0\0\203\0\0\0\377\1\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0"
  "\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0"
  "\0\0\210\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\210\0\0\0\377\205\0\0\0\0\206\0\0\0\377\210\0\0\0\0\202"
  "\0\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\204\0"
  "\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\202\0\0\0\0\1\0\0\0\377\202\0\0\0\0\202\0\0\0\377\206\0\0\0\0\203"
  "\0\0\0\377\210\0\0\0\0\204\0\0\0\377\211\0\0\0\0\203\0\0\0\377\210\0"
  "\0\0\0\202\0\0\0\377\213\0\0\0\0\202\0\0\0\377\213\0\0\0\0\202\0\0\0"
  "\377\261\0\0\0\0\202\0\0\0\377\203\0\0\0\0\204\0\0\0\377\202\0\0\0\0"
  "\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\204\0\0\0\0\202\0\0\0\377\203"
  "\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0"
  "\0\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0\0\206\0\0\0\377\205\0\0\0"
  "\0\203\0\0\0\377\202\0\0\0\0\204\0\0\0\377\203\0\0\0\0\204\0\0\0\377"
  "\202\0\0\0\0\203\0\0\0\377\206\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202"
  "\0\0\0\377\207\0\0\0\0\202\0\0\0\377\1\0\0\0\0\203\0\0\0\377\211\0\0"
  "\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377"
  "\202\0\0\0\0\202\0\0\0\377\203\0\0\0\0\204\0\0\0\377\203\0\0\0\0\202"
  "\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0"
  "\0\0\0\204\0\0\0\377\202\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0"
  "\377\202\0\0\0\0\204\0\0\0\377\203\0\0\0\0\204\0\0\0\377\203\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\206\0\0\0\0\1\0\0\0\377\206"
  "\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0"
  "\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\211\0\0\0\0\202\0\0\0\377\205\0\0\0\0\203\0\0\0\377\213\0"
  "\0\0\0\202\0\0\0\377\213\0\0\0\0\203\0\0\0\377\240\0\0\0\0\202\0\0\0"
  "\377\224\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\210\0\0\0\0"
  "\205\0\0\0\377\207\0\0\0\0\202\0\0\0\377\207\0\0\0\0\205\0\0\0\377\225"
  "\0\0\0\0\202\0\0\0\377\214\0\0\0\0\202\0\0\0\377\211\0\0\0\0\202\0\0"
  "\0\377\206\0\0\0\0\212\0\0\0\377\216\0\0\0\0\212\0\0\0\377\222\0\0\0"
  "\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377"
  "\207\0\0\0\0\202\0\0\0\377\213\0\0\0\0\202\0\0\0\377\211\0\0\0\0\205"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\213\0"
  "\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0"
  "\377\207\0\0\0\0\202\0\0\0\377\206\0\0\0\0\203\0\0\0\377\202\0\0\0\0"
  "\203\0\0\0\377\206\0\0\0\0\204\0\0\0\377\1\0\0\0\0\202\0\0\0\377\232"
  "\0\0\0\0\204\0\0\0\377\232\0\0\0\0\204\0\0\0\377\206\0\0\0\0\202\0\0"
  "\0\377\207\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\202\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\211\0\0\0\377\203\0\0\0\0\210\0\0\0\377\204"
  "\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\207\0\0\0\377\205\0\0\0\0\207\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\203\0\0\0\0\204\0\0\0\377\203\0\0\0\0\211\0\0\0\377"
  "\206\0\0\0\0\202\0\0\0\377\216\0\0\0\0\202\0\0\0\377\203\0\0\0\0\204"
  "\0\0\0\377\210\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\202\0"
  "\0\0\0\1\0\0\0\377\202\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377"
  "\202\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0"
  "\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\207\0\0\0\377\205\0\0"
  "\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\207\0\0\0\377"
  "\207\0\0\0\0\206\0\0\0\377\207\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\202\0"
  "\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1\0\0\0\0\203\0\0\0\377"
  "\1\0\0\0\0\202\0\0\0\377\206\0\0\0\0\203\0\0\0\377\211\0\0\0\0\202\0"
  "\0\0\377\211\0\0\0\0\203\0\0\0\377\211\0\0\0\0\202\0\0\0\377\213\0\0"
  "\0\0\202\0\0\0\377\213\0\0\0\0\202\0\0\0\377\255\0\0\0\0\206\0\0\0\377"
  "\203\0\0\0\0\203\0\0\0\377\204\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\206\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\204\0\0\0\0"
  "\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\204\0\0\0\0\202\0\0\0\377\206"
  "\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\207\0\0\0\0\205\0\0"
  "\0\377\212\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\202\0\0\0"
  "\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\203\0\0\0\0\203\0\0\0\377"
  "\204\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\203\0\0\0\0\203\0\0\0\377\204\0\0\0\0\202\0\0\0\377\203\0"
  "\0\0\0\202\0\0\0\377\204\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0"
  "\377\204\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\215\0\0\0\0"
  "\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203"
  "\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0"
  "\0\377\205\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\210"
  "\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\213\0\0\0\0\202\0\0"
  "\0\377\213\0\0\0\0\202\0\0\0\377\241\0\0\0\0\202\0\0\0\377\224\0\0\0"
  "\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\210\0\0\0\0\202\0\0\0\377"
  "\202\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\207\0\0\0\0\203"
  "\0\0\0\377\202\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\221\0\0"
  "\0\0\202\0\0\0\377\214\0\0\0\0\202\0\0\0\377\207\0\0\0\0\206\0\0\0\377"
  "\204\0\0\0\0\212\0\0\0\377\216\0\0\0\0\212\0\0\0\377\221\0\0\0\0\202"
  "\0\0\0\377\210\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\207\0"
  "\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\216\0\0\0\0\202\0\0\0"
  "\377\203\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\214\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206"
  "\0\0\0\0\202\0\0\0\377\206\0\0\0\0\203\0\0\0\377\204\0\0\0\0\203\0\0"
  "\0\377\212\0\0\0\0\202\0\0\0\377\232\0\0\0\0\204\0\0\0\377\232\0\0\0"
  "\0\204\0\0\0\377\206\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377"
  "\1\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0"
  "\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0"
  "\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377"
  "\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202"
  "\0\0\0\377\212\0\0\0\0\202\0\0\0\377\203\0\0\0\0\204\0\0\0\377\203\0"
  "\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0"
  "\377\216\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202"
  "\0\0\0\377\207\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0"
  "\377\1\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0"
  "\377\213\0\0\0\0\203\0\0\0\377\206\0\0\0\0\202\0\0\0\377\207\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\202"
  "\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1\0\0\0\0\203\0\0\0"
  "\377\1\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202"
  "\0\0\0\377\210\0\0\0\0\202\0\0\0\377\210\0\0\0\0\203\0\0\0\377\212\0"
  "\0\0\0\202\0\0\0\377\214\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0"
  "\377\253\0\0\0\0\210\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205"
  "\0\0\0\0\202\0\0\0\377\203\0\0\0\0\211\0\0\0\377\206\0\0\0\0\202\0\0"
  "\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0"
  "\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377"
  "\212\0\0\0\0\202\0\0\0\377\207\0\0\0\0\204\0\0\0\377\213\0\0\0\0\202"
  "\0\0\0\377\206\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\202\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205"
  "\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\213\0\0\0\0\206\0\0"
  "\0\377\210\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377"
  "\205\0\0\0\0\202\0\0\0\377\202\0\0\0\0\1\0\0\0\377\202\0\0\0\0\202\0"
  "\0\0\377\206\0\0\0\0\203\0\0\0\377\206\0\0\0\0\202\0\0\0\377\205\0\0"
  "\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\211\0\0\0\0\202\0\0\0\377"
  "\212\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\242\0\0\0\0\202"
  "\0\0\0\377\222\0\0\0\0\211\0\0\0\377\207\0\0\0\0\202\0\0\0\377\202\0"
  "\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1\0\0\0\0\203\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\204\0\0\0\377\221\0\0\0\0\203"
  "\0\0\0\377\212\0\0\0\0\203\0\0\0\377\205\0\0\0\0\203\0\0\0\377\1\0\0"
  "\0\0\202\0\0\0\377\1\0\0\0\0\203\0\0\0\377\206\0\0\0\0\202\0\0\0\377"
  "\255\0\0\0\0\202\0\0\0\377\210\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202"
  "\0\0\0\377\207\0\0\0\0\202\0\0\0\377\211\0\0\0\0\202\0\0\0\377\217\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\211\0\0\0\377\212\0\0\0\0\202\0\0\0"
  "\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0"
  "\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\212"
  "\0\0\0\0\202\0\0\0\377\234\0\0\0\0\204\0\0\0\377\206\0\0\0\0\212\0\0"
  "\0\377\206\0\0\0\0\204\0\0\0\377\210\0\0\0\0\202\0\0\0\377\207\0\0\0"
  "\0\202\0\0\0\377\1\0\0\0\0\206\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205"
  "\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377"
  "\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\216\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0"
  "\377\206\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\203\0\0\0\0\204\0\0\0\377\203"
  "\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0"
  "\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0"
  "\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\214\0\0\0\0\202\0\0\0\377"
  "\206\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\1\0\0\0\377\202\0\0\0\0\1\0\0\0\377\206\0\0\0"
  "\0\204\0\0\0\377\1\0\0\0\0\204\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1"
  "\0\0\0\0\202\0\0\0\377\210\0\0\0\0\202\0\0\0\377\207\0\0\0\0\203\0\0"
  "\0\377\213\0\0\0\0\202\0\0\0\377\214\0\0\0\0\202\0\0\0\377\212\0\0\0"
  "\0\202\0\0\0\377\252\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202"
  "\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0"
  "\0\0\0\211\0\0\0\377\206\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0"
  "\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\207"
  "\0\0\0\0\205\0\0\0\377\212\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0"
  "\0\377\202\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\203\0\0\0"
  "\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377"
  "\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0"
  "\0\0\0\202\0\0\0\377\214\0\0\0\0\206\0\0\0\377\207\0\0\0\0\202\0\0\0"
  "\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\204\0\0\0\0"
  "\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1"
  "\0\0\0\0\203\0\0\0\377\1\0\0\0\0\202\0\0\0\377\206\0\0\0\0\203\0\0\0"
  "\377\206\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0"
  "\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\212"
  "\0\0\0\0\202\0\0\0\377\266\0\0\0\0\211\0\0\0\377\203\0\0\0\0\202\0\0"
  "\0\377\202\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\204\0\0\0"
  "\0\202\0\0\0\377\1\0\0\0\0\205\0\0\0\377\202\0\0\0\0\202\0\0\0\377\204"
  "\0\0\0\0\203\0\0\0\377\223\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0"
  "\0\377\206\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\202\0\0\0"
  "\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377"
  "\226\0\0\0\0\202\0\0\0\377\210\0\0\0\0\202\0\0\0\377\211\0\0\0\0\202"
  "\0\0\0\377\204\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\210\0"
  "\0\0\0\202\0\0\0\377\211\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\203\0\0\0\0\211\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\205"
  "\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0"
  "\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0\0\204\0\0\0\377"
  "\204\0\0\0\0\212\0\0\0\377\205\0\0\0\0\203\0\0\0\377\223\0\0\0\0\202"
  "\0\0\0\377\202\0\0\0\0\204\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212"
  "\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0"
  "\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\212\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\204\0\0\0\377\203\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\204\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204"
  "\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\206\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\205\0\0\0\0\204\0\0\0\377\206\0\0\0\0\204\0\0\0\377"
  "\1\0\0\0\0\204\0\0\0\377\204\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0"
  "\0\0\377\207\0\0\0\0\202\0\0\0\377\206\0\0\0\0\203\0\0\0\377\214\0\0"
  "\0\0\202\0\0\0\377\215\0\0\0\0\202\0\0\0\377\211\0\0\0\0\202\0\0\0\377"
  "\252\0\0\0\0\202\0\0\0\377\204\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203"
  "\0\0\0\377\204\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0"
  "\0\0\0\202\0\0\0\377\204\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0"
  "\377\215\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0"
  "\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206"
  "\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0"
  "\0\377\1\0\0\0\0\203\0\0\0\377\211\0\0\0\0\202\0\0\0\377\206\0\0\0\0"
  "\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\203"
  "\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0"
  "\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\203\0\0\0\377\204\0\0\0"
  "\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\203\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\221\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202"
  "\0\0\0\377\207\0\0\0\0\202\0\0\0\377\204\0\0\0\0\203\0\0\0\377\205\0"
  "\0\0\0\204\0\0\0\377\206\0\0\0\0\202\0\0\0\377\1\0\0\0\0\203\0\0\0\377"
  "\1\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0"
  "\0\377\205\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\205\0\0\0"
  "\0\202\0\0\0\377\213\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377"
  "\212\0\0\0\0\202\0\0\0\377\267\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\211\0\0\0\377\205\0\0\0\0\202\0\0\0\377\1\0\0"
  "\0\0\202\0\0\0\377\1\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377"
  "\204\0\0\0\0\202\0\0\0\377\224\0\0\0\0\203\0\0\0\377\210\0\0\0\0\203"
  "\0\0\0\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\211\0"
  "\0\0\0\204\0\0\0\377\224\0\0\0\0\204\0\0\0\377\207\0\0\0\0\202\0\0\0"
  "\377\211\0\0\0\0\203\0\0\0\377\202\0\0\0\0\203\0\0\0\377\207\0\0\0\0"
  "\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\204"
  "\0\0\0\0\203\0\0\0\377\210\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\204\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0"
  "\0\203\0\0\0\377\205\0\0\0\0\202\0\0\0\377\207\0\0\0\0\203\0\0\0\377"
  "\204\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\203"
  "\0\0\0\377\205\0\0\0\0\204\0\0\0\377\210\0\0\0\0\204\0\0\0\377\213\0"
  "\0\0\0\203\0\0\0\377\220\0\0\0\0\203\0\0\0\377\225\0\0\0\0\203\0\0\0"
  "\377\211\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0"
  "\202\0\0\0\377\204\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203"
  "\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\203\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0"
  "\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377"
  "\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202"
  "\0\0\0\377\204\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0"
  "\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0"
  "\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203"
  "\0\0\0\0\202\0\0\0\377\212\0\0\0\0\203\0\0\0\377\202\0\0\0\0\204\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0\0\0"
  "\0\202\0\0\0\377\204\0\0\0\0\203\0\0\0\377\206\0\0\0\0\202\0\0\0\377"
  "\207\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\205\0\0\0\0\204"
  "\0\0\0\377\206\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\204\0"
  "\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0"
  "\377\206\0\0\0\0\202\0\0\0\377\215\0\0\0\0\202\0\0\0\377\215\0\0\0\0"
  "\202\0\0\0\377\211\0\0\0\0\202\0\0\0\377\252\0\0\0\0\202\0\0\0\377\203"
  "\0\0\0\0\204\0\0\0\377\203\0\0\0\0\204\0\0\0\377\202\0\0\0\0\203\0\0"
  "\0\377\203\0\0\0\0\203\0\0\0\377\204\0\0\0\0\202\0\0\0\377\203\0\0\0"
  "\0\202\0\0\0\377\203\0\0\0\0\204\0\0\0\377\203\0\0\0\0\202\0\0\0\377"
  "\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\207\0\0\0\0\203"
  "\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0"
  "\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0"
  "\377\207\0\0\0\0\202\0\0\0\377\202\0\0\0\0\203\0\0\0\377\210\0\0\0\0"
  "\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\202"
  "\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0"
  "\0\204\0\0\0\377\202\0\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377"
  "\202\0\0\0\0\204\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\207\0"
  "\0\0\0\203\0\0\0\377\202\0\0\0\0\204\0\0\0\377\205\0\0\0\0\204\0\0\0"
  "\377\206\0\0\0\0\204\0\0\0\377\1\0\0\0\0\204\0\0\0\377\204\0\0\0\0\202"
  "\0\0\0\377\203\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\204\0"
  "\0\0\0\203\0\0\0\377\204\0\0\0\0\202\0\0\0\377\214\0\0\0\0\202\0\0\0"
  "\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\242\0\0\0\0"
  "\202\0\0\0\377\223\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\206"
  "\0\0\0\0\207\0\0\0\377\205\0\0\0\0\202\0\0\0\377\202\0\0\0\0\205\0\0"
  "\0\377\203\0\0\0\0\211\0\0\0\377\223\0\0\0\0\202\0\0\0\377\210\0\0\0"
  "\0\202\0\0\0\377\213\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377"
  "\211\0\0\0\0\204\0\0\0\377\224\0\0\0\0\204\0\0\0\377\206\0\0\0\0\202"
  "\0\0\0\377\213\0\0\0\0\206\0\0\0\377\205\0\0\0\0\210\0\0\0\377\204\0"
  "\0\0\0\211\0\0\0\377\204\0\0\0\0\207\0\0\0\377\211\0\0\0\0\202\0\0\0"
  "\377\206\0\0\0\0\207\0\0\0\377\205\0\0\0\0\207\0\0\0\377\205\0\0\0\0"
  "\202\0\0\0\377\211\0\0\0\0\210\0\0\0\377\205\0\0\0\0\207\0\0\0\377\206"
  "\0\0\0\0\204\0\0\0\377\210\0\0\0\0\204\0\0\0\377\215\0\0\0\0\202\0\0"
  "\0\377\216\0\0\0\0\202\0\0\0\377\216\0\0\0\0\202\0\0\0\377\210\0\0\0"
  "\0\210\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377"
  "\203\0\0\0\0\210\0\0\0\377\205\0\0\0\0\207\0\0\0\377\204\0\0\0\0\210"
  "\0\0\0\377\204\0\0\0\0\211\0\0\0\377\203\0\0\0\0\202\0\0\0\377\213\0"
  "\0\0\0\207\0\0\0\377\204\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0"
  "\377\204\0\0\0\0\206\0\0\0\377\206\0\0\0\0\207\0\0\0\377\204\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\211\0\0\0\377\203"
  "\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0"
  "\0\377\204\0\0\0\0\203\0\0\0\377\204\0\0\0\0\207\0\0\0\377\204\0\0\0"
  "\0\202\0\0\0\377\213\0\0\0\0\207\0\0\0\377\204\0\0\0\0\202\0\0\0\377"
  "\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\210\0\0\0\377\207\0\0\0\0\202"
  "\0\0\0\377\210\0\0\0\0\207\0\0\0\377\207\0\0\0\0\202\0\0\0\377\207\0"
  "\0\0\0\203\0\0\0\377\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\206\0\0\0\0"
  "\212\0\0\0\377\205\0\0\0\0\205\0\0\0\377\213\0\0\0\0\202\0\0\0\377\205"
  "\0\0\0\0\205\0\0\0\377\252\0\0\0\0\206\0\0\0\377\1\0\0\0\0\202\0\0\0"
  "\377\203\0\0\0\0\202\0\0\0\377\1\0\0\0\0\205\0\0\0\377\205\0\0\0\0\210"
  "\0\0\0\377\204\0\0\0\0\205\0\0\0\377\1\0\0\0\0\202\0\0\0\377\204\0\0"
  "\0\0\210\0\0\0\377\206\0\0\0\0\202\0\0\0\377\210\0\0\0\0\210\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202"
  "\0\0\0\377\212\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\203\0"
  "\0\0\0\203\0\0\0\377\207\0\0\0\0\204\0\0\0\377\204\0\0\0\0\202\0\0\0"
  "\377\202\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377\203\0\0\0\0"
  "\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\204\0\0\0\0\207\0\0\0\377\204"
  "\0\0\0\0\202\0\0\0\377\1\0\0\0\0\205\0\0\0\377\205\0\0\0\0\205\0\0\0"
  "\377\1\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\212\0\0\0\0\211"
  "\0\0\0\377\206\0\0\0\0\204\0\0\0\377\206\0\0\0\0\205\0\0\0\377\1\0\0"
  "\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\207\0\0\0\0\203\0\0\0\377"
  "\203\0\0\0\0\203\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\204\0\0\0\0\210\0\0\0\377\203\0\0\0\0\211\0\0\0\377\206\0"
  "\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0\0"
  "\377\241\0\0\0\0\204\0\0\0\377\222\0\0\0\0\202\0\0\0\377\202\0\0\0\0"
  "\202\0\0\0\377\211\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\203"
  "\0\0\0\0\203\0\0\0\377\205\0\0\0\0\204\0\0\0\377\202\0\0\0\0\202\0\0"
  "\0\377\224\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\245\0\0\0"
  "\0\202\0\0\0\377\225\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377"
  "\214\0\0\0\0\204\0\0\0\377\206\0\0\0\0\210\0\0\0\377\204\0\0\0\0\211"
  "\0\0\0\377\205\0\0\0\0\205\0\0\0\377\212\0\0\0\0\202\0\0\0\377\207\0"
  "\0\0\0\205\0\0\0\377\207\0\0\0\0\205\0\0\0\377\206\0\0\0\0\202\0\0\0"
  "\377\212\0\0\0\0\206\0\0\0\377\207\0\0\0\0\205\0\0\0\377\210\0\0\0\0"
  "\202\0\0\0\377\213\0\0\0\0\202\0\0\0\377\254\0\0\0\0\204\0\0\0\377\210"
  "\0\0\0\0\207\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\207\0\0\0\377\207\0\0\0\0\205\0\0\0\377\205\0\0\0"
  "\0\207\0\0\0\377\205\0\0\0\0\211\0\0\0\377\203\0\0\0\0\202\0\0\0\377"
  "\214\0\0\0\0\205\0\0\0\377\205\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\204\0\0\0\0\206\0\0\0\377\207\0\0\0\0\205\0\0\0\377\205\0"
  "\0\0\0\202\0\0\0\377\206\0\0\0\0\1\0\0\0\377\203\0\0\0\0\211\0\0\0\377"
  "\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\202\0\0\0\377\205\0\0\0\0\205\0\0\0\377\205\0"
  "\0\0\0\202\0\0\0\377\214\0\0\0\0\210\0\0\0\377\202\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\202\0\0\0\377\204\0\0\0\0\206\0\0\0\377\210\0\0\0\0"
  "\202\0\0\0\377\211\0\0\0\0\205\0\0\0\377\210\0\0\0\0\202\0\0\0\377\207"
  "\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0"
  "\0\377\205\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\206\0\0\0"
  "\0\212\0\0\0\377\205\0\0\0\0\205\0\0\0\377\213\0\0\0\0\202\0\0\0\377"
  "\205\0\0\0\0\205\0\0\0\377\253\0\0\0\0\204\0\0\0\377\202\0\0\0\0\202"
  "\0\0\0\377\203\0\0\0\0\202\0\0\0\377\202\0\0\0\0\203\0\0\0\377\207\0"
  "\0\0\0\206\0\0\0\377\206\0\0\0\0\203\0\0\0\377\202\0\0\0\0\202\0\0\0"
  "\377\205\0\0\0\0\206\0\0\0\377\207\0\0\0\0\202\0\0\0\377\211\0\0\0\0"
  "\204\0\0\0\377\1\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205"
  "\0\0\0\0\202\0\0\0\377\206\0\0\0\0\202\0\0\0\377\212\0\0\0\0\202\0\0"
  "\0\377\207\0\0\0\0\202\0\0\0\377\204\0\0\0\0\202\0\0\0\377\210\0\0\0"
  "\0\203\0\0\0\377\204\0\0\0\0\202\0\0\0\377\202\0\0\0\0\202\0\0\0\377"
  "\202\0\0\0\0\202\0\0\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202"
  "\0\0\0\377\205\0\0\0\0\205\0\0\0\377\205\0\0\0\0\202\0\0\0\377\202\0"
  "\0\0\0\203\0\0\0\377\207\0\0\0\0\203\0\0\0\377\202\0\0\0\0\202\0\0\0"
  "\377\203\0\0\0\0\202\0\0\0\377\213\0\0\0\0\207\0\0\0\377\210\0\0\0\0"
  "\203\0\0\0\377\207\0\0\0\0\203\0\0\0\377\202\0\0\0\0\202\0\0\0\377\206"
  "\0\0\0\0\202\0\0\0\377\207\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0"
  "\0\377\203\0\0\0\0\202\0\0\0\377\205\0\0\0\0\202\0\0\0\377\205\0\0\0"
  "\0\204\0\0\0\377\1\0\0\0\0\202\0\0\0\377\203\0\0\0\0\211\0\0\0\377\207"
  "\0\0\0\0\203\0\0\0\377\210\0\0\0\0\202\0\0\0\377\210\0\0\0\0\203\0\0"
  "\0\377\243\0\0\0\0\202\0\0\0\377\377\0\0\0\0\204\0\0\0\0\202\0\0\0\377"
  "\377\0\0\0\0\263\0\0\0\0\202\0\0\0\377\255\0\0\0\0\202\0\0\0\377\377"
  "\0\0\0\0\334\0\0\0\0\202\0\0\0\377\377\0\0\0\0\377\0\0\0\0\207\0\0\0"
  "\0\202\0\0\0\377\236\0\0\0\0\202\0\0\0\377\303\0\0\0\0\202\0\0\0\377"
  "\221\0\0\0\0\202\0\0\0\377\336\0\0\0\0\202\0\0\0\377\236\0\0\0\0\202"
  "\0\0\0\377\377\0\0\0\0\263\0\0\0\0\202\0\0\0\377\377\0\0\0\0\263\0\0"
  "\0\0\202\0\0\0\377\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\254\0\0\0\0\214"
  "\0\0\0\377\335\0\0\0\0\202\0\0\0\377\236\0\0\0\0\202\0\0\0\377\303\0"
  "\0\0\0\202\0\0\0\377\221\0\0\0\0\202\0\0\0\377\336\0\0\0\0\202\0\0\0"
  "\377\236\0\0\0\0\202\0\0\0\377\377\0\0\0\0\262\0\0\0\0\202\0\0\0\377"
  "\377\0\0\0\0\263\0\0\0\0\202\0\0\0\377\377\0\0\0\0\377\0\0\0\0\377\0"
  "\0\0\0\255\0\0\0\0\214\0\0\0\377\335\0\0\0\0\202\0\0\0\377\233\0\0\0"
  "\0\205\0\0\0\377\303\0\0\0\0\202\0\0\0\377\221\0\0\0\0\202\0\0\0\377"
  "\336\0\0\0\0\202\0\0\0\377\236\0\0\0\0\202\0\0\0\377\377\0\0\0\0\377"
  "\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0\377\0\0\0\0"
  "\377\0\0\0\0\377\0\0\0\0\232\0\0\0\0"};


static GdkPixbuf*
ascii_pixbuf_create (void)
{
  return gdk_pixbuf_new_from_inline (-1, ascii_font_pixdata, FALSE, NULL);
}
