#!/bin/bash
# This file is in the public domain.

set -eu

function clean_wallet() {
    rm -f "${WALLET_DB}"
    exit_cleanup
}


# Replace with 0 for nexus...
USE_FAKEBANK=1
if [ 1 = "$USE_FAKEBANK" ]
then
    ACCOUNT="exchange-account-2"
    BANK_FLAGS="-f -d x-taler-bank -u $ACCOUNT"
    BANK_URL="http://localhost:8082/"
else
    ACCOUNT="exchange-account-1"
    BANK_FLAGS="-ns -d iban -u $ACCOUNT"
    BANK_URL="http://localhost:18082/"
    echo -n "Testing for libeufin-bank"
    libeufin-bank --help >/dev/null </dev/null || exit_skip " MISSING"
    echo " FOUND"

fi

. setup.sh

echo -n "Testing for taler-harness"
taler-harness --help >/dev/null </dev/null || exit_skip " MISSING"
echo " FOUND"

# Launch exchange, merchant and bank.
setup -c "test_template.conf" \
      -r "merchant-exchange-default" \
      -em \
      $BANK_FLAGS
LAST_RESPONSE=$(mktemp -p "${TMPDIR:-/tmp}" test_response.conf-XXXXXX)
CONF="test_template.conf.edited"
WALLET_DB=$(mktemp -p "${TMPDIR:-/tmp}" test_wallet.json-XXXXXX)
EXCHANGE_URL="http://localhost:8081/"

# Install cleanup handler (except for kill -9)
trap clean_wallet EXIT

echo -n "First prepare wallet with coins ..."
rm -f "$WALLET_DB"
taler-wallet-cli \
    --no-throttle \
    --wallet-db="$WALLET_DB" \
    api \
    --expect-success 'withdrawTestBalance' \
  "$(jq -n '
    {
        amount: "TESTKUDOS:99",
        corebankApiBaseUrl: $BANK_URL,
        exchangeBaseUrl: $EXCHANGE_URL
    }' \
    --arg BANK_URL "${BANK_URL}" \
    --arg EXCHANGE_URL "$EXCHANGE_URL"
  )" 2>wallet-withdraw-1.err >wallet-withdraw-1.out
echo -n "."
# FIXME-MS: add logic to have nexus check immediately here.
# sleep 10
echo -n "."
# NOTE: once libeufin can do long-polling, we should
# be able to reduce the delay here and run wirewatch
# always in the background via setup
taler-exchange-wirewatch \
    -a "$ACCOUNT" \
    -L "INFO" \
    -c "$CONF" \
    -t &> taler-exchange-wirewatch.out
echo -n "."
taler-wallet-cli \
    --wallet-db="$WALLET_DB" \
    run-until-done \
    2>wallet-withdraw-finish-1.err \
    >wallet-withdraw-finish-1.out
echo " OK"

CURRENCY_COUNT=$(taler-wallet-cli --wallet-db="$WALLET_DB" balance | jq '.balances|length')
if [ "$CURRENCY_COUNT" = "0" ]
then
    exit_fail "Expected least one currency, withdrawal failed. check log."
fi

#
# CREATE INSTANCE FOR TESTING
#

echo -n "Configuring merchant instance ..."

STATUS=$(curl -H "Content-Type: application/json" -X POST \
    -H 'Authorization: Bearer secret-token:super_secret' \
    http://localhost:9966/management/instances \
    -d '{"auth":{"method":"external"},"id":"admin","name":"default","user_type":"business","address":{},"jurisdiction":{},"use_stefan":true,"default_wire_transfer_delay":{"d_us" : 50000000000},"default_pay_delay":{"d_us": 60000000000}}' \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "204" ]
then
    exit_fail "Expected '204 No content' response. Got instead $STATUS"
fi
echo "Ok"

echo -n "Configuring merchant bank account ..."

if [ 1 = "$USE_FAKEBANK" ]
then
    FORTYTHREE="payto://x-taler-bank/localhost/fortythree?receiver-name=fortythree"
else
    FORTYTHREE=$(get_payto_uri fortythree x)
fi
# add bank account address
STATUS=$(curl -H "Content-Type: application/json" -X POST \
    -H 'Authorization: Bearer secret-token:super_secret' \
    http://localhost:9966/private/accounts \
    -d '{"payto_uri":"'"$FORTYTHREE"'"}' \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "200" ]
then
    exit_fail "Expected '200 OK' response. Got instead $STATUS"
fi
echo "Ok"


echo -n "Creating Paivana template..."
TID="paivana"
STATUS=$(curl 'http://localhost:9966/private/templates' \
    -d '{"template_id":"paivana","template_description":"A Paivana template","template_contract":{"template_type":"paivana","summary":"The summary","choices":[{"amount":"TESTKUDOS:1"}]}}' \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

if [ "$STATUS" != "204" ]
then
    cat "$LAST_RESPONSE" >&2
    exit_fail "Expected 204, template created. got: $STATUS"
fi

echo "Checking template data ..."
STATUS=$(curl http://localhost:9966/templates/"$TID" \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

AMOUNT=$(jq -r .template_contract.choices[0].amount < "$LAST_RESPONSE")
if [ "$AMOUNT" != "TESTKUDOS:1" ]
then
    cat "$LAST_RESPONSE" >&2
    exit_fail "Expected TESTKUDOS:1. Got: $AMOUNT"
fi
echo " OK"

echo -n "Creating order using template..."
PAIVANA_ID="4321-4ZAXW7M5WW09QNZDF4MRAXBKS1321QQEBFVSWD68FCXHXS1G04A4ZAXW7M5WW09QNZDF4MRAXBKS1321QQEBFVSWD68FCXHXS1G04A0"
STATUS=$(curl 'http://localhost:9966/templates/'"$TID" \
              -d '{"template_type":"paivana","tip":"TESTKUDOS:0.1","website":"https://example.com/","paivana_id":"'"${PAIVANA_ID}"'"}' \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

if [ "$STATUS" != "200" ]
then
    cat "$LAST_RESPONSE" >&2
    exit_fail "Expected 200, order created. got: $STATUS"
fi

ORDER_ID=$(jq -r .order_id < "$LAST_RESPONSE")
TOKEN=$(jq -r .token < "$LAST_RESPONSE")

if [ "$TOKEN" == "null" ]
then
    cat "$LAST_RESPONSE" >&2
    exit_fail "token should not be null, got: $TOKEN"
fi

echo "OK"

echo -n "Fetching order details ..."
STATUS=$(curl "http://localhost:9966/private/orders/${ORDER_ID}?session_id=${PAIVANA_ID}" \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

if [ "$STATUS" != "200" ]
then
    cat "$LAST_RESPONSE" >&2
    exit_fail "Expected 200, getting order info. got: $STATUS"
fi

PRICE=$(jq -r .total_amount < "$LAST_RESPONSE")
if [ "$PRICE" != "TESTKUDOS:1.1" ]
then
    cat "$LAST_RESPONSE" >&2
    exit_fail "Expected TESTKUDOS:1.1 (with tip) but got: $PRICE"
fi

PAY_URL=$(jq -e -r .taler_pay_uri < "$LAST_RESPONSE")
echo "OK: $PAY_URL"


NOW=$(date +%s)

echo -n "Pay first order ${PAY_URL} ..."
# echo "0" to tell wallet to use choice #0
echo "0" | \
  taler-wallet-cli \
      --no-throttle \
      --wallet-db="$WALLET_DB" \
      handle-uri "${PAY_URL}" \
      -y 2> wallet-pay1.err > wallet-pay1.log

taler-wallet-cli \
    --no-throttle \
    --wallet-db="$WALLET_DB" \
    run-until-done 2> wallet-finish-pay1.err > wallet-finish-pay1.log
NOW2=$(date +%s)
echo " OK (took $(( NOW2 - NOW )) secs )"

# Check payment status AND binding to session ID.
STATUS=$(curl "http://localhost:9966/private/orders/${ORDER_ID}?session_id=${PAIVANA_ID}" \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

if [ "$STATUS" != "200" ]
then
    cat "$LAST_RESPONSE" >&2
    exit_fail "Expected 200, after pay. got: $STATUS"
fi

ORDER_STATUS=$(jq -r .order_status < "$LAST_RESPONSE")

if [ "$ORDER_STATUS" != "paid" ]
then
    cat "$LAST_RESPONSE" >&2
    exit_fail "Order status should be 'paid'. got: $ORDER_STATUS"
fi

echo "TEST PASSED"
exit 0
