RCS_ID("$Id: FFActionHandler.m 610 2007-08-26 19:40:40Z ravemax $")

#import "FFActionHandler.h"
#import <stdlib.h>
#import <sys/time.h>
#import "FFPreferences.h"
#import "FFOptions.h"
#import "FFImageFile.h"
#import "FFImageList.h"
#import "FFImageListController.h"
#import "FFImageWindowTitled.h"
#import "FFImageWindowBorderless.h"
#import "FFFullscreenWindow.h"
#import "FFOSD.h"
#import "FFInfoPanel.h"
#import "FFGL.h"
#import "FFPageView.h"
#import "FFPage.h"
#import "FFVideoInfo.h"
#import "NSWindow_Additions.h"

@implementation FFActionHandler

#pragma mark -
#pragma mark Initialisation and cleanup

static BOOL _hideHack = FALSE; // ugly. FALSE = don't show when imagelist is activated

- (void)_registerForNotifications {
	NSNotificationCenter*   nc = [NSNotificationCenter defaultCenter];
	
	// FFImageListController (not the whole controller but the window
	[nc addObserver:self selector:@selector(imageListWinActivated:)
			   name:ImageListWinActivatedNotification object:nil];
	
	// FFImageList
	[nc addObserver:self selector:@selector(newImageSelected:)
			   name:NewImageSelectedNotification object:nil];
	[nc addObserver:self selector:@selector(imageListChanged:)
			   name:ImageListChangedNotification object:nil];
	[nc addObserver:self selector:@selector(imageListRearranged:)
			   name:ImageListRearrangedNotification object:nil];
	
	// Menu or app
	[nc addObserver:self selector:@selector(rotationChanged:)
			   name:RotationChangedNotification object:nil];
	[nc addObserver:self selector:@selector(scalingChanged:)
			   name:ScalingChangedNotification object:nil];
	[nc addObserver:self selector:@selector(pageModeChanged:)
			   name:PageModeChangedNotification object:nil];
	[nc addObserver:self selector:@selector(noBlowUpChanged:)
			   name:NoBlowUpChangedNotification object:nil];
	[nc addObserver:self selector:@selector(antialiasingChanged:) 
			   name:AntialiasingChangedNotification object:nil];
	[nc addObserver:self selector:@selector(screenModeChanged:)
			   name:ScreenModeChangedNotification object:nil];
	[nc addObserver:self selector:@selector(OSDVisibilityChanged:) 
			   name:OSDVisibilityChangedNotification object:nil];
	[nc addObserver:self selector:@selector(zoomMultiplierChanged:)
			   name:ZoomMultiplierChangedNotification object:nil];
	[nc addObserver:self selector:@selector(magnifyingLensVisibilityChanged:)
			   name:MagnifyingLensVisibilityChangedNotification object:nil];
	
	// FFPreferences
	[nc addObserver:self selector:@selector(ctrlForPanningEnOrDisabled:)
			   name:CtrlForPanningEnOrDisabledNotification object:nil];
	[nc addObserver:self selector:@selector(windowBackgroundChanged:)
			   name:WindowBackgroundChangedNotification object:nil];
	[nc addObserver:self selector:@selector(doublePageAlignmentChanged:)
			   name:DoublePageAlignmentChangedNotification object:nil];
}

- (void)_setupRingBuffer {
	int i;
	
	m_imageLock		= [[NSLock alloc] init];
	m_loadLock		= [[NSLock alloc] init];
	m_ringBufSize   = [m_prefs ringBufferSize];
	m_fetchDist		= [m_prefs fetchDistance];
	m_lastImgIdx	= 0;
	m_ringIdx		= 0;

	m_ringBuffer = (FFPage**)malloc(sizeof(FFPage*)*m_ringBufSize);
	for (i = 0; i < m_ringBufSize; i++) {
		m_ringBuffer[i] = [[FFPage alloc] initWithView:m_pageView];
		NSAssert(m_ringBuffer[i], @"Failed to alloc a ring buffer element");
	}	
}

- (void)_setupWindowAndView {
	// Must be called before creating the first OpenGL Context
	createGLPixelPipelineLock();
	
	// Create window & view
	if ([m_prefs borderlessWin])
		m_imageWindow = [[FFImageWindowBorderless alloc] initWithLevel:[m_prefs winLevel]];
	else
		m_imageWindow = [[FFImageWindowTitled alloc] init];
	
	m_pageView = [[FFPageView alloc]
					initWithOptions:m_opts
					backgroundColor:([m_prefs winBackgroundAuto] ? nil : [m_prefs winBackgroundColor])
						   lensSize:NSMakeSize([m_prefs magnifyingLensWidth], [m_prefs magnifyingLensHeight])
						   lensZoom:[m_prefs lensZoom]
				doublePageAlignment:[m_prefs doublePageAlignment]
						maxTileSize:[m_prefs maxTileSize]
					   eventHandler:self
							 andOSD:m_OSD];
	
	[m_pageView setAutoresizingMask:(NSViewWidthSizable | NSViewHeightSizable)];
	[m_imageWindow setContentView:m_pageView];
	[m_imageWindow setDelegate:self];
	[m_imageWindow useOptimizedDrawing:TRUE]; // no overlapping subviews
	[m_imageWindow setAlphaValue:[m_prefs imageWindowAlpha]];
	
	// Move the window to the screen on which FFView was lastly (?) terminated
	NSNumber* screenNum = [m_prefs screenNumber];
	if (screenNum != nil) {
		NSEnumerator*	en = [[NSScreen screens] objectEnumerator];
		NSScreen*		screen;
		
		while (screen = [en nextObject]) {
			if ([[[screen deviceDescription] objectForKey:@"NSScreenNumber"] isEqualToNumber:screenNum]) {
				[m_imageWindow setFrameOrigin:[screen frame].origin];
				break;
			}
		}
	}
}

- (id)initWithOptions:(FFOptions*)opts preferences:(FFPreferences*)prefs
		 andImageList:(FFImageList*)imgList {
	if ([super init]) {
		
		// Member vars
		m_opts				= [opts retain];
		m_prefs				= [prefs retain];
		m_imgList			= [imgList retain];
		m_ctrlForPanning	= [prefs ctrlForPanning];
		m_panningOn			= !m_ctrlForPanning;
		m_crossCursorSmall	= [[NSCursor alloc] initWithImage:[NSImage imageNamed:@"cross_cursor_small"]
													  hotSpot:NSMakePoint(3.0f, 3.0f)];
		
		// OSD
		m_OSD = [[FFOSD alloc] initWithOptions:m_opts];

		[self _setupWindowAndView];
		[self _registerForNotifications];
		[self _setupRingBuffer];
	}
	return self;
}

- (void)dealloc {
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	
	int i;
	for (i = 0; i < m_ringBufSize; i++)
		[m_ringBuffer[i] release];
	free(m_ringBuffer);
	
	[m_imageLock release];
	[m_loadLock release];
	[m_OSD release];
	[m_crossCursorSmall release];
	[m_imgList release];
	[m_prefs release];
	[m_opts release];	
	[m_imageWindow release];
	[m_pageView release];

	deleteGLPixelPipelineLock();
	
	[super dealloc];
}

- (void)cleanUp {
	// Store the current screen
	if ([m_imageWindow isPrimaryScreen])
		[m_prefs setScreenNumber:nil];
	else 
		[m_prefs setScreenNumber:[[[m_imageWindow screen] deviceDescription] objectForKey:@"NSScreenNumber"]];
}

#pragma mark -
#pragma mark Image handling thread methods

#define CYCLE_INDEX(OLD_INDEX, DISTANCE, SIZE) \
	((OLD_INDEX+DISTANCE+SIZE) % SIZE)

#define LOADING_FAILED			-1
#define LOADING_ALREADY_LOADED  0
#define LOADING_SUCCESSFULL		1

- (int)_loadImageWithIndex:(int)imgIdx intoRingIndex:(int)ringIdx {
	FFPage*		page	= m_ringBuffer[ringIdx];
	int			lret	= LOADING_SUCCESSFULL;
	NSString*	fn;

	FFLOG(2, @"_loadImageWithIndex:%d into:%d; page->fileindex=%u = %u",
		  imgIdx, ringIdx, page->fileIndex, [m_imgList fileIndexAtIndex:imgIdx])
	
	// Already loaded?
	if (page->fileIndex == [m_imgList fileIndexAtIndex:imgIdx])
		return LOADING_ALREADY_LOADED;
	
	// Get the filename (and extract)
	fn = [m_imgList fullPathAtIndex:imgIdx];
	if (fn == nil)
		return LOADING_FAILED;
	[fn retain]; // If the list is changed fast then this might become "nil" (= crash)
	
	// Try to load it
	NS_DURING
		[page loadImageWithFilename:fn andFileIndex:[m_imgList fileIndexAtIndex:imgIdx]];
	NS_HANDLER
		[FFInfoPanel runInFullscreen:[m_opts fullscreen]
						   withTitle:@"Image loader error"
							 message:[localException reason]
						 andOkButton:@"Ok"];
		
		lret =  LOADING_FAILED;
	NS_ENDHANDLER

	[fn release];
	return lret;
}

- (void)_prefetchImageWithDistance:(int)dist andNumberOfImages:(unsigned)numImg {
	int ringIdx = CYCLE_INDEX(m_ringIdx, dist, m_ringBufSize);
	FFLOG(2, @"prefetch - ringIdx=%d - aus m_ringIdx=%d, dist=%d, ringbufsize=%d", ringIdx, m_ringIdx, dist, m_ringBufSize)
		
	int imgIdx = CYCLE_INDEX(m_lastImgIdx, dist, numImg);
	
	 // A image was removed while loading
	if (imgIdx >= [m_imgList numberOfImages])
		return;
	
	// Try to prefetch the image
	(void)[self _loadImageWithIndex:imgIdx intoRingIndex:ringIdx];
}

- (void)imageLoaderThread:(id)_unused {
	NSAutoreleasePool*  pool;
	int					i;
	unsigned			noi;
	int					idx2nd, idxPrev;
	
	pool = [[NSAutoreleasePool alloc] init];	
	[m_imageLock lock];
	[m_loadLock lock];
	
	// Current image
	FFLOG(2, @"current image - ringidx=%d", m_ringIdx)
	if ([self _loadImageWithIndex:m_lastImgIdx intoRingIndex:m_ringIdx] == LOADING_FAILED) {
		[m_imageLock unlock]; // this terminates the display thread
		[m_imgList removeCurrentImage];
		[m_loadLock unlock]; // let the next load thead (if available) run
		[pool release];
		[NSThread exit];
	}

	// Second - and if Auto-single mode enabled the previous image
	noi	= [m_imgList numberOfImages];
	if ([m_opts pageMode] != OPT_MODE_SINGLE) {
		// Second image	
		idx2nd	= CYCLE_INDEX(m_ringIdx, +1, m_ringBufSize);
		if (m_lastImgIdx+1 < noi)
			(void)[self _loadImageWithIndex:(m_lastImgIdx + 1) intoRingIndex:idx2nd];
		else
			[m_ringBuffer[idx2nd] reset];
		
		// Previous image
		if ((m_lastImgIdx > 0) && ([m_prefs autoSinglePageAdjustment] > 0.0f)) {
			idxPrev	= CYCLE_INDEX(m_ringIdx, -1, m_ringBufSize);
			(void)[self _loadImageWithIndex:(m_lastImgIdx - 1) intoRingIndex:idxPrev];
		}
	}
	
	// Unblock display thread
	[m_imageLock unlock];
	
	// Prefetch
	if (noi > 1) {
		for (i = 1; i <= m_fetchDist; i++) {
			// image removed while prefetch
			[self _prefetchImageWithDistance: i andNumberOfImages:noi]; // Next image
			[self _prefetchImageWithDistance:-i andNumberOfImages:noi]; // Prev. image
		}
	}
	
	FFLOG(1, @"imgLoaderThread finished")
		
	[m_loadLock unlock];	
	
	[pool release];
	[NSThread exit];
}

- (void)_clearRingBuffer {
	int			i;
	for (i = 0; i < m_ringBufSize; i++)
		[m_ringBuffer[i] reset];
}

#define AUTO_SINGLE_PAGE_DETECTION(PAGE) \
	(PAGE->width * [m_prefs autoSinglePageAdjustment] >= PAGE->height)

#define PAGE_INFO_WITH_INDEX(PAGE, INDEX_ADJUST) \
	[pageInfo(PAGE) stringByAppendingFormat:@" [%d/%u]", \
		[m_imgList selectedImageIndex]+1+INDEX_ADJUST, [m_imgList numberOfImages]]

- (void)imageDisplayThread:(id)_unused {
	NSAutoreleasePool*  pool;
	
	// Wait till the loader is finished w/ the current image
	[m_imageLock lock];	[m_imageLock unlock]; 
	
	// Terminate the display thread if no current image is available	
	if (m_ringBuffer[m_ringIdx]->fileIndex == FILE_INDEX_UNDEFINED)
		[NSThread exit];
		
	pool = [[NSAutoreleasePool alloc] init];

	// Add the pages according to the page mode
	NSString*	selDisplayName	= [[m_imgList displayNameOfSelectedImage] string];
				m_singleImage	= ([m_opts pageMode] == OPT_MODE_SINGLE);

	[m_pageView removeAllPages];
	
	// Double page mode
	if (!m_singleImage) {
		NSString*	nextDisplayName = [[m_imgList displayNameOfNextImage] string];
		FFPage*		prevPage;
		
		// Only one image left
		if (nextDisplayName == nil)
			m_singleImage = TRUE;
		
		// Atleast 2 images left
		else {
			FFPage* selPage		= m_ringBuffer[m_ringIdx];
			FFPage* nextPage	= m_ringBuffer[CYCLE_INDEX(m_ringIdx, +1, m_ringBufSize)];

			// Auto detect a "double scan page"
			if (AUTO_SINGLE_PAGE_DETECTION(selPage) || AUTO_SINGLE_PAGE_DETECTION(nextPage))
				m_singleImage = TRUE;
			
			// Valid double page mode
			else {
				NSString* selInfo	= PAGE_INFO_WITH_INDEX(selPage, 0);
				NSString* nextInfo	= PAGE_INFO_WITH_INDEX(nextPage, 1);
				
				[m_imgList setSelectionIncrementer:2];
				
				if ([m_opts pageMode] == OPT_MODE_WESTERN) {
					[m_pageView addPage:selPage];
					[m_pageView addPage:nextPage];
					[m_OSD setFilename:selDisplayName
						   information:selInfo
							 filename2:nextDisplayName
					   andInformation2:nextInfo];
				} else { // Manga mode
					[m_pageView addPage:nextPage];
					[m_pageView addPage:selPage];
					[m_OSD setFilename:nextDisplayName
						   information:nextInfo
							 filename2:selDisplayName
					   andInformation2:selInfo];
				}
			}
		}
		
		// Set decrementer - 1 if the previous image is a "double page scan" else 2
		if ([m_imgList selectedImageIndex] > 0) {
			prevPage = m_ringBuffer[CYCLE_INDEX(m_ringIdx, -1, m_ringBufSize)];

			if ((prevPage != nil) && AUTO_SINGLE_PAGE_DETECTION(prevPage))
				[m_imgList setSelectionDecrementer:1];
			else
				[m_imgList setSelectionDecrementer:2];
		} else
			[m_imgList setSelectionDecrementer:2];
	
	// Single page mode
	} else
		[m_imgList setSelectionDecrementer:1];
	
	// Display only 1 image - by option (single page mode), only 1 image left or auto-detection
	if (m_singleImage) {
		[m_imgList setSelectionIncrementer:1];
		
		[m_pageView addPage:m_ringBuffer[m_ringIdx]];
		[m_OSD setFilename:selDisplayName
			andInformation:PAGE_INFO_WITH_INDEX(m_ringBuffer[m_ringIdx], 0)];
	}
	
	[m_pageView finishedAdding];
	
	// Set the title of the window to the display name of the 1st image
	[m_imageWindow setTitle:selDisplayName];
	
	// Adjust and show window - if required
	if (![m_opts fullscreen]) {
		NSSize	psize	= [m_pageView preferredSize];
		BOOL	aratio	= ([m_opts scaling] == OPT_FIT_TO_WINDOW);
		if (![m_imageWindow isVisible]) {
			[m_imageWindow adjustViewSize:psize refresh:REFRESH_NEVER 
						  keepAspectRatio:aratio andCenter:TRUE];
			[m_pageView setNeedsDisplay:TRUE];
			[m_imageWindow makeKeyAndOrderFront:self];
			_hideHack = TRUE;
		} else {
			if ([m_prefs dontResizeImageWindow])
				[m_pageView doLayoutAndDisplay];
			else
				[m_imageWindow adjustViewSize:psize refresh:REFRESH_ALWAYS
							  keepAspectRatio:(BOOL)aratio andCenter:FALSE];
		}	
	} else
		[m_pageView doLayoutAndDisplay];

	[pool release];
	[NSThread exit];
}

#pragma mark -
#pragma mark Delegate methods for the image window

- (BOOL)windowShouldClose:(id)sender {
	_hideHack = FALSE;
	[m_imageWindow orderOut:self];
	return FALSE;
}

- (BOOL)windowShouldZoom:(NSWindow*)sender toFrame:(NSRect)newFrame {
	[m_imageWindow adjustViewSize:[m_pageView preferredSize]
								  refresh:REFRESH_ONLY_WHEN_RESIZED
						  keepAspectRatio:([m_opts scaling] == OPT_FIT_TO_WINDOW)
								andCenter:FALSE];
	return FALSE;
}

#pragma mark -
#pragma mark Notificationhandlers - FFImageListController

- (void)imageListWinActivated:(NSNotification*)notification {
	if (![m_imageWindow isMiniaturized] && _hideHack)
		[m_imageWindow orderWindow:NSWindowBelow relativeTo:[[notification object] intValue]];
	_hideHack = TRUE;
}

#pragma mark -
#pragma mark Notificationhandlers - FFImageList

- (void)_imageLoader {
	int selIdx = [m_imgList selectedImageIndex];
	
	// Note: Calculation is not 100% correct
	FFLOG(1, @"_imageLoader. old:m_ringIdx=%d, selIdx=%d, lastImgIdx=%d, ringBufSize=%d",
		  m_ringIdx, selIdx, m_lastImgIdx, m_ringBufSize);
	
	m_ringIdx = CYCLE_INDEX(m_ringIdx, (selIdx - m_lastImgIdx), m_ringBufSize);
	
	// Complete out of the prefetched data
	if ((m_ringIdx < 0) || (m_ringIdx >= m_ringBufSize)) {
		FFLOG(1, @"* clear ... ringbuf - %d", m_ringIdx)
		[self _clearRingBuffer];
		m_ringIdx		= selIdx % m_ringBufSize;
		m_lastImgIdx	= selIdx;
		FFLOG(1, @" --- newringidx = %d", m_ringIdx)
	}
		
	m_lastImgIdx = selIdx;
	
	[NSThread detachNewThreadSelector:@selector(imageLoaderThread:) toTarget:self withObject:nil];
}

- (void)newImageSelected:(NSNotification*)notification {
	[self _imageLoader];
	
	FFLOG(8, @"newImageSelected")
	
	[NSThread detachNewThreadSelector:@selector(imageDisplayThread:) toTarget:self withObject:nil];
}

- (void)imageListChanged:(NSNotification*)notification {
	if ([m_imgList numberOfImages] > 0)
		[self _imageLoader];
	else {
		[m_imageWindow orderOut:self];		
		if ([m_opts fullscreen])
			[m_opts setFullscreen:FALSE];
		
		[self _clearRingBuffer];
		m_ringIdx		= 0;
		m_lastImgIdx	= 0;		
	}
}

- (void)_rearrangeCheckWithDistance:(int)dist numberOfImagesToRefetch:(unsigned*)refetchCnt {
	int			newRingIdx		= CYCLE_INDEX(m_ringIdx, dist, m_ringBufSize);
	int			imgIdx			= CYCLE_INDEX(m_lastImgIdx, dist, [m_imgList numberOfImages]);
	unsigned	listFileIndex	= [m_imgList fileIndexAtIndex:imgIdx];
	int			oldRingIdx;
	FFPage*		page;
	
	for (oldRingIdx = 0; oldRingIdx < m_ringBufSize; oldRingIdx++) {
		page = m_ringBuffer[oldRingIdx];
		if (page->fileIndex == listFileIndex) {
			if (newRingIdx != oldRingIdx) {
				m_ringBuffer[oldRingIdx] = m_ringBuffer[newRingIdx];
				m_ringBuffer[newRingIdx] = page;
			}
			(*refetchCnt)--;
			break;
		}
	}
}	

- (void)imageListRearranged:(NSNotification*)notification {
	BOOL needToRedraw;
	
	// Current image
	int selIdx		= [m_imgList selectedImageIndex];
	int newRingIdx  = CYCLE_INDEX(m_ringIdx, (selIdx - m_lastImgIdx), m_ringBufSize);
	
	FFLOG(1, @"lastIdx=%u, selIdx=%u : ringIdx=%u, newRingIdx=%u", m_lastImgIdx, selIdx, m_ringIdx, newRingIdx)	
		
	if (newRingIdx < 0)
		newRingIdx = selIdx % m_ringBufSize;
	
	FFPage*	t					= m_ringBuffer[newRingIdx];
	m_ringBuffer[newRingIdx]	= m_ringBuffer[m_ringIdx];
	m_ringBuffer[m_ringIdx]		= t;
	
	m_lastImgIdx	= selIdx;
	m_ringIdx		= newRingIdx;
	
	// 2nd image if not single page mode
	if ([m_opts pageMode] != OPT_MODE_SINGLE) {
		needToRedraw = TRUE; //XXX Optimize later
	} else
		needToRedraw = FALSE;
	
	// Check for the rest
	int i;
	unsigned refetchCnt = m_fetchDist << 1;
	
	for (i = 1; i <= m_fetchDist; i++) {
		[self _rearrangeCheckWithDistance: i numberOfImagesToRefetch:&refetchCnt];
		[self _rearrangeCheckWithDistance:-i numberOfImagesToRefetch:&refetchCnt];
	}
	
	FFLOG(1, @"left to refetch = %d", refetchCnt)
		
	if (needToRedraw) {
		[self newImageSelected:nil];
	} else if (refetchCnt > 0)
		[NSThread detachNewThreadSelector:@selector(imageLoaderThread:) toTarget:self withObject:nil];
}

#pragma mark -
#pragma mark Notificationhandlers - Menu or App

- (void)rotationChanged:(NSNotification*)notification {
	[m_OSD setRotation:[m_opts rotation]];
	if ([m_imgList numberOfImages] > 0) {
		if (![m_opts fullscreen] && ([m_opts scaling] == OPT_NO_SCALING)) {
			[m_imageWindow adjustViewSize:[m_pageView preferredSize] refresh:REFRESH_ALWAYS
						  keepAspectRatio:([m_opts scaling] == OPT_FIT_TO_WINDOW)
								andCenter:FALSE];
		} else
			[m_pageView doLayoutAndDisplay];
	}
}

- (void)scalingChanged:(NSNotification*)notification {
	[m_imageWindow setContentResizeIncrements:NSMakeSize(1.0f, 1.0f)]; // Fix
	[m_OSD setScaling:[m_opts scaling]];
	[m_pageView doLayoutAndDisplay];
}

- (void)pageModeChanged:(NSNotification*)notification {
	[self newImageSelected:nil]; // optimize?
}

- (void)noBlowUpChanged:(NSNotification*)notification {
	[m_OSD setNoBlowUp:[m_opts noBlowUp]];
	[m_pageView doLayoutAndDisplay];
}

- (void)antialiasingChanged:(NSNotification*)notification {
	[m_OSD setAntialiasing:[m_opts antialiasing]];
	[m_pageView setNeedsDisplay:TRUE];
}

- (void)_cyclicFullscreenCheck:(NSTimer*)timer {
	m_lastFullscreenCheck.tv_sec += 3;
	if (!m_mouseIsVisible)
		return;
	
	if (m_lastFullscreenCheck.tv_sec - m_lastMouseMove.tv_sec > 2) {
		m_mouseIsVisible = FALSE;
		[NSCursor setHiddenUntilMouseMoves:TRUE];
	}
}

- (void)_switchToFullscreen {
	FFLOG(2, @"Switch to fullscreen")

	// Window just as big as the current screen
	m_fsWin = [[FFFullscreenWindow alloc] initWithWindow:m_imageWindow 
												 andView:m_pageView
								  disableAutoShowMenuBar:[m_prefs disableAutoShowMenuBar]];
	if (m_fsWin != nil) {
		[m_imageWindow makeKeyWindow];
		[m_fsWin makeKeyAndOrderFront:self];
	}
	
	// Cursor hide stuff
	if ([m_prefs keepMouseHidden])
		[NSCursor hide];
	else {
		m_mouseIsVisible = TRUE;
		gettimeofday(&m_lastMouseMove, NULL);
		gettimeofday(&m_lastFullscreenCheck, NULL);
		m_cursorHideTimer = [NSTimer scheduledTimerWithTimeInterval:3.0 target:self selector:@selector(_cyclicFullscreenCheck:) userInfo:nil repeats:TRUE];	
	}
}

- (void)_switchToWindowedMode {
	FFLOG(2, @"Switch to windowed mode")	
		
	[m_fsWin close];

	// Reattach view
	[m_imageWindow setContentView:m_pageView];
	[m_imageWindow makeKeyWindow];
	
	// Cursor stuff
	if ([m_prefs keepMouseHidden])
		[NSCursor unhide];
	else {
		[NSCursor setHiddenUntilMouseMoves:FALSE];
		if ((m_cursorHideTimer != nil) && [m_cursorHideTimer isValid])
			[m_cursorHideTimer invalidate];
	}

	if ([m_imageWindow isVisible]) {
		[m_imageWindow adjustViewSize:[m_pageView preferredSize] refresh:REFRESH_ALWAYS
					  keepAspectRatio:([m_opts scaling] == OPT_FIT_TO_WINDOW)
							andCenter:FALSE];
	}
}

- (void)screenModeChanged:(NSNotification*)notification {
	if ([m_opts fullscreen])
		[self _switchToFullscreen];
	else
		[self _switchToWindowedMode];
}

- (void)OSDVisibilityChanged:(NSNotification*)notification {
	[m_OSD setVisibility:[m_opts isOSDVisible]];
	[m_pageView setNeedsDisplay:TRUE];
}

- (void)zoomMultiplierChanged:(NSNotification*)notification {
	[m_pageView doLayoutAndDisplay];
}

- (void)magnifyingLensVisibilityChanged:(NSNotification*)notification {
	if ([m_opts isMagnifyingLensVisible]) {
		[m_pageView setCursor:m_crossCursorSmall];
		[m_pageView showMagnifyingLensAtPoint:[m_imageWindow mouseLocationOutsideOfEventStream]];
	} else {
		[m_pageView setCursor:nil];
		[m_pageView hideMagnifyingLens];
		[m_pageView setNeedsDisplay:TRUE];
	}
}

#pragma mark -
#pragma mark Notificationhandlers - FFPreferences

- (void)ctrlForPanningEnOrDisabled:(NSNotification*)notification {
	m_ctrlForPanning = [m_prefs ctrlForPanning];
	m_panningOn = !m_ctrlForPanning;
}

- (void)windowBackgroundChanged:(NSNotification*)notification {
	[m_pageView setBackgroundColor:
		([m_prefs winBackgroundAuto] ? nil : [m_prefs winBackgroundColor])];
	if ([m_imgList numberOfImages] > 0)
		[m_pageView setNeedsDisplay:TRUE];
}

- (void)doublePageAlignmentChanged:(NSNotification*)notification {
	[m_pageView setDoublePageAlignment:[m_prefs doublePageAlignment]];
}

#pragma mark -
#pragma mark Events from the view

- (BOOL)acceptsFirstResponder   { return TRUE; }

- (void)mouseMoved:(NSEvent*)event {
	// Show cursor
	if ([m_opts fullscreen]) {
		gettimeofday(&m_lastMouseMove, NULL);
		m_mouseIsVisible = TRUE;
	}

	// Pan or move the lens (if visible)
	if (m_panningOn)
		[m_pageView panWithDeltaX:[event deltaX] andDeltaY:[event deltaY] relative:[m_prefs relativePanning]];
	else if ([m_opts isMagnifyingLensVisible])
		[m_pageView updateMagnifyingLensPoint:[event locationInWindow]];
}

- (void)flagsChanged:(NSEvent*)event {
	m_panningOn = (BOOL)(([event modifierFlags] & NSControlKeyMask) >> 18) ^ !m_ctrlForPanning;
}

- (void)rightMouseDown:(NSEvent*)event {
	m_panningOn = m_ctrlForPanning;
}

- (void)rightMouseUp:(NSEvent*)event {
	m_panningOn = !m_ctrlForPanning;
}

- (void)scrollWheel:(NSEvent*)event {
	if ([m_prefs scrollWheelAction] == PREF_SCROLL_WHEEL_PAN)
		(void)[m_pageView panWithDeltaX:([event deltaX]*[m_prefs mouseWheelSpeedup])
							  andDeltaY:([event deltaY]*[m_prefs mouseWheelSpeedup])
							   relative:[m_prefs relativePanning]];
	else {
		BOOL up = [event deltaY] > 0.0f;
		
		if ([m_prefs scrollWheelAction] == PREF_SCROLL_WHEEL_PAGE)
			up ? [self pageUp] : [self pageDown];
		else
			up ? [m_imgList selectPreviousImage] : [m_imgList selectNextImage];
	}
}

- (void)rightMouseDragged:(NSEvent*)event {
	[self mouseMoved:event];
}

- (void)pageClicked:(int)no {
	FFLOG(8, @"page Clicked: %d", no)
	
	if (m_singleImage)
		[m_imgList selectNextImage];
	else if ([m_opts pageMode] == OPT_MODE_WESTERN) {
		if (no == 0)
			[m_imgList selectPreviousImage];
		else
			[m_imgList selectNextImage];
	} else {
		if (no == 1)
			[m_imgList selectPreviousImage];
		else
			[m_imgList selectNextImage];
	}
}

#pragma mark -
#pragma mark Panning & Page

#define PAN_ONE_DIRECTION(DX, DY) \
	[m_pageView panWithDeltaX:DX andDeltaY:DY relative:[m_prefs relativePanning]]

- (void)panLeft			{ PAN_ONE_DIRECTION((float)[m_prefs normalCursorShift], 0.0f); }
- (void)panLeftAccel	{ PAN_ONE_DIRECTION((float)[m_prefs acceleratedCursorShift], 0.0f); }
- (void)panRight		{ PAN_ONE_DIRECTION(-(float)[m_prefs normalCursorShift], 0.0f); }
- (void)panRightAccel	{ PAN_ONE_DIRECTION(-(float)[m_prefs acceleratedCursorShift], 0.0f); }
- (void)panUp			{ PAN_ONE_DIRECTION(0.0f, (float)[m_prefs normalCursorShift]); }
- (void)panUpAccel		{ PAN_ONE_DIRECTION(0.0f, (float)[m_prefs acceleratedCursorShift]); }
- (void)panDown			{ PAN_ONE_DIRECTION(0.0f, - (float)[m_prefs normalCursorShift]); }
- (void)panDownAccel	{ PAN_ONE_DIRECTION(0.0f, - (float)[m_prefs acceleratedCursorShift]); }

- (void)pageUp {
	if (([m_opts scaling] == OPT_FIT_TO_WINDOW) || ![m_pageView pageUp])
		[m_imgList selectPreviousImage];
}

- (void)pageDown {
	if (([m_opts scaling] == OPT_FIT_TO_WINDOW) || ![m_pageView pageDown])
		[m_imgList selectNextImage];
}

#define CAN_PAGE_OR_MOVE(OBJCMSG) \
	(([m_imgList numberOfImages] > 0) && ([m_opts scaling] != OPT_FIT_TO_WINDOW) \
	 && [m_pageView OBJCMSG])

- (BOOL)canPageDown {	return CAN_PAGE_OR_MOVE(canPageDown); }
- (BOOL)canPageUp	{	return CAN_PAGE_OR_MOVE(canPageUp); }
- (BOOL)canMoveLeft { 	return CAN_PAGE_OR_MOVE(canMoveLeft); }
- (BOOL)canMoveRight {	return CAN_PAGE_OR_MOVE(canMoveRight); }

#pragma mark -
#pragma mark Misc. exported

- (void)showImageWindow {
	if ([m_imgList numberOfImages] > 0)
		[m_imageWindow makeKeyAndOrderFront:self];
}

- (NSWindow*)imageWindow {
	return m_imageWindow;
}

- (void)autoFullScreenIfPossible {
	if (([m_imgList numberOfImages] > 0) && ![m_imageWindow isVisible])
		[m_opts setFullscreen:TRUE];
}

- (void)showInternalStatus {
	int				i;
	FFPage*			page;

	NSLog(@"[FFView internal status, %@]", [NSDate date]); 
	
	// VRAM
	NSLog(@"VRAM size = %ld", VRAMSize());
	
	// Ring buffer
	NSLog(@"--- Ring buffer ---\n");
	for (i = 0; i < m_ringBufSize; i++) {
		page = m_ringBuffer[i];
		NSLog(@"%d : %@", i, ((page->fileIndex == FILE_INDEX_UNDEFINED) ?
							  @"<empty>" : [page description]));
	}
	NSLog(@"Ring index = %d\n", m_ringIdx);
	
	// View
	NSLog(@"\n%@", [m_pageView description]);
	NSLog(@"[end]");

	// Extended mode
#if 0
	NSFileManager*		fm = [NSFileManager defaultManager];
	NSString*			ddir = [@"~/Desktop/FFView_Debug_Out" stringByExpandingTildeInPath];
	BOOL				isDir;
	NSSize				size;
	NSMutableData*		mdata;
	NSBitmapImageRep*	rep;
	NSImage*			img;
	
	if (![fm fileExistsAtPath:ddir isDirectory:&isDir])
		[fm createDirectoryAtPath:ddir attributes:nil];

	size	= [m_pageView frame].size;
	
	rep = [[[NSBitmapImageRep alloc] 
			initWithBitmapDataPlanes:NULL
						  pixelsWide:(GLint)size.width
						  pixelsHigh:(GLint)size.height
					   bitsPerSample:8
					 samplesPerPixel:4
							hasAlpha:TRUE
							isPlanar:FALSE
					  colorSpaceName:NSCalibratedRGBColorSpace
						 bytesPerRow:(GLint)(size.width * 4)
						bitsPerPixel:32] autorelease];
	
	[m_pageView readBufferWithWidth:size.width andHeight:size.height
					   intoBuffer:[rep bitmapData]];
	
	img = [[NSImage alloc] initWithSize:size];
	[img addRepresentation:rep];
	[img setFlipped:TRUE];
	
	[[img TIFFRepresentationUsingCompression:NSTIFFCompressionJPEG factor:0.6] 
		writeToFile:[ddir stringByAppendingPathComponent:@"complete.tif"] atomically:FALSE];
	
	[img release];
#endif
}

@end

