% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assay-functions.R
\name{assay-functions}
\alias{assay-functions}
\alias{sparseAssay}
\alias{compactAssay}
\alias{disjoinAssay}
\alias{qreduceAssay}
\title{Create simplified representation of ragged assay data.}
\usage{
sparseAssay(
  x,
  i = 1,
  withDimnames = TRUE,
  background = NA_integer_,
  sparse = FALSE
)

compactAssay(
  x,
  i = 1,
  withDimnames = TRUE,
  background = NA_integer_,
  sparse = FALSE
)

disjoinAssay(
  x,
  simplifyDisjoin,
  i = 1,
  withDimnames = TRUE,
  background = NA_integer_
)

qreduceAssay(
  x,
  query,
  simplifyReduce,
  i = 1,
  withDimnames = TRUE,
  background = NA_integer_
)
}
\arguments{
\item{x}{A \code{RaggedExperiment} object}

\item{i}{integer(1) or character(1) name of assay to be
transformed.}

\item{withDimnames}{logical(1) include dimnames on the returned
matrix. When there are no explict rownames, these are
manufactured with \code{as.character(rowRanges(x))}; rownames
are always manufactured for \code{compactAssay()} and
\code{disjoinAssay()}.}

\item{background}{A value (default NA) for the returned matrix after
\code{*Assay} operations}

\item{sparse}{logical(1) whether to return a
\code{\link[Matrix]{sparseMatrix}} representation}

\item{simplifyDisjoin}{A \code{function} / functional operating on a
    \code{*List}, where the elements of the list are all within-sample
    assay values from ranges overlapping each disjoint range. For
    instance, to use the \code{simplifyDisjoin=mean} of overlapping ranges,
    where ranges are characterized by integer-valued scores, the
    entries are calculated as
    \preformatted{
                    a
    original: |-----------|
                        b
                   |----------|

                a    a, b   b
    disjoint: |----|------|---|

    values <- IntegerList(a, c(a, b), b)
    simplifyDisjoin(values)
    }}

\item{query}{\code{GRanges} providing regions over which reduction
is to occur.}

\item{simplifyReduce}{A \code{function} / functional accepting arguments
    \code{score}, \code{range}, and \code{qrange}:

    \itemize{

        \item{\code{score}} A \code{*List}, where each list element
            corresponds to a cell in the matrix to be returned by
            \code{qreduceAssay}. Vector elements correspond to
            ranges overlapping query. The \code{*List} objects
            support many vectorized mathematical operations, so
            \code{simplifyReduce} can be implemented efficiently.

        \item{\code{range}} A \code{GRangesList} instance,
            'parallel' to \code{score}. Each element of the list
            corresponds to a cell in the matrix to be returned by
            \code{qreduceAssay}. Each range in the element
            corresponds to the range for which the \code{score}
            element applies.

        \item{\code{qrange}} A \code{GRanges} instance with the
             same length as \code{unlist(score)}, providing the
             query range window to which the corresponding scores
             apply.

    }}
}
\value{
\code{sparseAssay()}: A matrix() with dimensions
    \code{dim(x)}. Elements contain the assay value for the \emph{i}th
    range and \emph{j}th sample. Use 'sparse=TRUE' to obtain
    a \code{\link[Matrix]{sparseMatrix}} assay representation.

\code{compactAssay()}: Samples with identical range are placed
    in the same row. Non-disjoint ranges are NOT collapsed. Use
    'sparse=TRUE' to obtain a \code{\link[Matrix]{sparseMatrix}} assay
    representation.

\code{disjoinAssay()}: A matrix with number of rows equal
    to number of disjoint ranges across all samples. Elements of
    the matrix are summarized by applying \code{simplifyDisjoin()} to
    assay values of overlapping ranges

\code{qreduceAssay()}: A matrix() with dimensions
    \code{length(query) x ncol(x)}. Elements contain assay
    values for the ith query range and jth sample, summarized
    according to the function \code{simplifyReduce}.
}
\description{
These methods transform \code{assay()} from the
    default (i.e., \code{sparseAssay()}) representation to various
    forms of more dense representation. \code{compactAssay()}
    collapses identical ranges across samples into a single
    row. \code{disjoinAssay()} creates disjoint (non-overlapping)
    regions, simplifies values within each sample in a
    user-specified manner, and returns a matrix of disjoint regions
    x samples.

This method transforms \code{assay()} from the default
    (i.e., \code{sparseAssay()}) representation to a reduced
    representation summarizing each original range overlapping
    ranges in \code{query}. Reduction in each cell can be tailored
    to indivdual needs using the \code{simplifyReduce} functional argument.
}
\examples{
re4 <- RaggedExperiment(GRangesList(
    GRanges(c(A = "chr1:1-10:-", B = "chr1:8-14:-", C = "chr2:15-18:+"),
        score = 3:5),
    GRanges(c(D = "chr1:1-10:-", E = "chr2:11-18:+"), score = 1:2)
), colData = DataFrame(id = 1:2))

query <- GRanges(c("chr1:1-14:-", "chr2:11-18:+"))

weightedmean <- function(scores, ranges, qranges)
{
    ## weighted average score per query range
    ## the weight corresponds to the size of the overlap of each
    ## overlapping subject range with the corresponding query range
    isects <- pintersect(ranges, qranges)
    sum(scores * width(isects)) / sum(width(isects))
}

qreduceAssay(re4, query, weightedmean)

\dontrun{
    ## Extended example: non-silent mutations, summarized by genic
    ## region
    suppressPackageStartupMessages({
        library(TxDb.Hsapiens.UCSC.hg19.knownGene)
        library(org.Hs.eg.db)
        library(GenomeInfoDb)
        library(MultiAssayExperiment)
        library(curatedTCGAData)
        library(TCGAutils)
    })

    ## TCGA MultiAssayExperiment with RaggedExperiment data
    mae <- curatedTCGAData("ACC", c("RNASeq2GeneNorm", "CNASNP", "Mutation"),
        version = "1.1.38", dry.run = FALSE)

    ## genomic coordinates
    gn <- genes(TxDb.Hsapiens.UCSC.hg19.knownGene)
    gn <- keepStandardChromosomes(granges(gn), pruning.mode="coarse")
    seqlevelsStyle(gn) <- "NCBI"
    genome(gn)
    gn <- unstrand(gn)

    ## reduce mutations, marking any genomic range with non-silent
    ## mutation as FALSE
    nonsilent <- function(scores, ranges, qranges)
        any(scores != "Silent")
    mre <- mae[["ACC_Mutation-20160128"]]
    seqlevelsStyle(rowRanges(mre)) <- "NCBI"
    ## hack to make genomes match
    genome(mre) <- paste0(correctBuild(unique(genome(mre)), "NCBI"), ".p13")
    mutations <- qreduceAssay(mre, gn, nonsilent, "Variant_Classification")
    genome(mre) <- correctBuild(unique(genome(mre)), "NCBI")

    ## reduce copy number
    re <- mae[["ACC_CNASNP-20160128"]]
    class(re)
    ## [1] "RaggedExperiment"
    seqlevelsStyle(re) <- "NCBI"
    genome(re) <- "GRCh37.p13"
    cn <- qreduceAssay(re, gn, weightedmean, "Segment_Mean")
    genome(re) <- "GRCh37"

    ## ALTERNATIVE
    ##
    ## TCGAutils helper function to convert RaggedExperiment objects to
    ## RangedSummarizedExperiment based on annotated gene ranges
    mae2 <- mae
    mae2[[1L]] <- re
    mae2[[2L]] <- mre
    qreduceTCGA(mae2)
}
}
