% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/apeglm.R
\name{apeglm}
\alias{apeglm}
\title{Approximate posterior estimation for GLM coefficients}
\usage{
apeglm(
  Y,
  x,
  log.lik,
  param = NULL,
  coef = NULL,
  mle = NULL,
  no.shrink = FALSE,
  interval.type = c("laplace", "HPD", "credible"),
  interval.level = 0.95,
  threshold = NULL,
  contrasts,
  weights = NULL,
  offset = NULL,
  flip.sign = TRUE,
  prior.control,
  multiplier = 1,
  ngrid = 50,
  nsd = 5,
  ngrid.nuis = 5,
  nsd.nuis = 2,
  log.link = TRUE,
  param.sd = NULL,
  method = c("general", "nbinomR", "nbinomCR", "nbinomC", "nbinomC*", "betabinR",
    "betabinCR", "betabinC", "betabinC*"),
  optim.method = "BFGS",
  bounds = c(-Inf, Inf)
)
}
\arguments{
\item{Y}{the observations, which can be a matrix or SummarizedExperiment,
with columns for samples and rows for "features" (e.g. genes in a genomic context).
If Y is a SummarizedExperiment, \code{apeglm} will return, in addition
to other list items, a GRanges or GRangesList \code{ranges} with the
estimated coefficients as metadata columns.}

\item{x}{design matrix, with intercept in the first column.
Continuous-valued columns should be centered and scaled to unit variance,
or at least set so that the scale (in SD) is not very large or very small}

\item{log.lik}{the log of likelihood function, specified by the user.
For Negative Binomial distribution, user can use \code{logLikNB} provided within the package.}

\item{param}{the other parameter(s) to be used in the likelihood function,
e.g. the dispersion parameter for a negative binomial distribution.
this can be a vector or a matrix (with columns as parameters)}

\item{coef}{(optional) the index of the coefficient for which
to generate posterior estimates (FSR, svalue, and intervals)}

\item{mle}{(optional) a 2 column matrix giving the MLE and its standard error
of \code{coef}. this will be used to adapt the scale of the prior (empirical Bayes).
This overrides the \code{prior.scale} specified by \code{prior.control}
and sets \code{no.shrink} to all coefficients other than \code{coef}.
Note that these MLE's and SE's should be on the natural log scale for a log link GLM.}

\item{no.shrink}{logical, if TRUE, apeglm won't perform shrinkage (default is FALSE)}

\item{interval.type}{(optional) can be "laplace", "HPD", or "credible", which specifies 
the type of Bayesian interval that the user wants to output; "laplace" represents the 
Laplace approximation of the posterior mode}

\item{interval.level}{(optional) default is 0.95}

\item{threshold}{(optional) a threshold for integrating posterior probabilities,
see details under 'Value'.
Note that this should be on the natural log scale for a log link GLM.}

\item{contrasts}{(optional) contrast matrix, same number of rows as \code{x}}

\item{weights}{(optional) weights matrix, same shape as \code{Y}}

\item{offset}{(optional) offsets matrix, same shape as \code{Y}.
Note that this should be on the natural log scale for a log link GLM.}

\item{flip.sign}{whether to flip the sign of threshold value
when MAP is negative, default is TRUE (threshold must then be positive)}

\item{prior.control}{see Details}

\item{multiplier}{a positive number, when the prior is adapted to the \code{mle}
matrix provided, this parameter connects the scale of the estimated distribution
of betas to the scale of the prior. the default value was chosen based on
FSR and error analysis of simulated data}

\item{ngrid}{the number of grid points for grid integration of intervals}

\item{nsd}{the number of SDs of the Laplace posterior approximation to set the
left and right edges of the grid around the MAP}

\item{ngrid.nuis}{the number of grid points for nuisance parameters}

\item{nsd.nuis}{the number of Laplace standard errors to set the
left and right edges of the grid around the MAP of the nuisance parameters}

\item{log.link}{whether the GLM has a log link (default = TRUE)}

\item{param.sd}{(optional) potential uncertainty measure on the parameter \code{param}.
this should only be a vector, used when \code{param} specifies a single parameter}

\item{method}{options for how apeglm will find the posterior mode and SD.
The default is "general" which allows the user to specify a likelihood
in a general way. Alternatives for faster performance with the Negative Binomial
likelihood are:
"nbinomR", "nbinomCR", and "nbinomC" / "nbinomC*"
These alternative methods should provide increasing speeds,
  respectively.
(Also for beta binomial, substitute "betabin" for "nbinom" in the
  above.)
From testing on RNA-seq data, the nbinom methods are roughly 5x, 10x and 50x faster than "general".
Note that "nbinomC" uses C++ to find the MAP for the coefficients,
but does not calculate or return the posterior SD or other quantities.
"nbinomC*" is the same as "nbinomC", but includes a random start for finding the MAP.
"nbinomCR" uses C++ to calculate the MAP and then estimates
the posterior SD in R, with the exception that if the MAP from C++
did not converge or gives negative estimates of posterior variance,
then this row is refit using optimization in R.
These alternatives require the degrees of freedom for the prior distribution to be 1,
and will ignore any function provided to the \code{log.lik} argument.
\code{param} should specify the dispersion parameter of a Negative Binomial
(such that Var = mu + param mu^2).}

\item{optim.method}{the method passed to \code{optim}}

\item{bounds}{the bounds for the numeric optimization}
}
\value{
a list of matrices containing the following components:
\itemize{
  \item \code{map}: matrix of MAP estimates, columns for coefficients and rows for features
  \item \code{sd}: matrix of posterior SD, same shape as \code{map}
  \item \code{prior.control}: list with details on the prior
  \item \code{fsr}: vector of the false sign rate for \code{coef}
  \item \code{svalue}: vector of the s-values for \code{coef} 
  \item \code{interval}: matrix of either HPD or credible interval for \code{coef}
  \item \code{thresh}: vector of the posterior probability that the estimated parameter 
is smaller than the threshold value specified in \code{threshold}
when MAP is positive (or greater than
-1 * threshold value when MAP is negative and flip.sign is TRUE)
  \item \code{diag}: matrix of diagnostics
  \item \code{contrast.map}: vector of MAP estimates corresponding to the \code{contrast}
when \code{contrast} is given 
  \item \code{contrast.sd}: vector of posterior SD corresponding to the \code{contrast}
when \code{contrast} is given
  \item \code{ranges}: GRanges or GRangesList with the estimated coefficients,
if \code{Y} was a SummarizedExperiment.
}

Note that all parameters associated with coefficients,
e.g. \code{map}, \code{sd}, etc., are returned on the natural log scale for a log link GLM.
}
\description{
apeglm provides Bayesian shrinkage estimators for effect sizes
in GLM models, using approximation of the posterior for individual coefficients.
}
\details{
\code{prior.control} is a list of parameters that will be passed to determine
the prior distribution. Users are allowed to have a Normal prior on the 
intercept, and a t prior on the non-intercept coefficients (similar
to \code{bayesglm} in the \code{arm} package. The following are defaults:

\itemize{
  \item \code{no.shrink = 1}: index of the coefficient(s) not to shrink 
  \item \code{prior.mean = 0}: mean of t prior
  \item \code{prior.scale = 1}: scale of t prior
  \item \code{prior.df = 1}: df of t prior
  \item \code{prior.no.shrink.mean = 0}: mean of Normal
  \item \code{prior.no.shrink.scale = 15}: scale of Normal
}

So without specifying \code{prior.control}, the following is set inside \code{apeglm}:

\code{prior.control <- list(no.shrink=1,prior.mean=0,prior.scale=1,
      prior.df=1,prior.no.shrink.mean=0,prior.no.shrink.scale=15)}

Note that the prior should be defined on the natural log scale for a log link GLM.
}
\examples{

# Simulate RNA-Seq read counts data

# 5 samples for each of the two groups
# a total of 100 genes
n.per.group <- 5 
n <- n.per.group * 2
m <- 100

# The design matrix includes one column of intercept
# and one column indicating samples that belong to the second group
condition <- factor(rep(letters[1:2], each = n.per.group))
x <- model.matrix(~condition) 

# Specify the standard deviation of beta (LFC between groups)
beta.sd <- 2
beta.cond <- rnorm(m, 0, beta.sd)
beta.intercept <- runif(m, 2, 6)
beta.mat <- cbind(beta.intercept, beta.cond)

# Generate the read counts
mu <- exp(t(x \%*\% t(beta.mat)))
Y <- matrix(rnbinom(m*n, mu=mu, size=1/.1), ncol = n)

# Here we will use the negative binomial log likelihood
# which is an exported function. See 'logLikNB' for details.
# For the NB:
# 'param' is the dispersion estimate (1/size)
# 'offset' can be used to adjust for size factors (log of size factors)
param <- matrix(0.1, nrow = m, ncol = 1)
offset <- matrix(0, nrow = m, ncol = n)

# Shrinkage estimator of betas:
# (for adaptive shrinkage, 'apeglm' requires 'mle' coefficients
# estimated with another software, or by first running 'apeglm'
# setting 'no.shrink=TRUE'.)
res <- apeglm(Y = Y, x = x,
              log.lik = logLikNB,
              param = param,
              offset = offset,
              coef = 2)

head(res$map)
plot(beta.mat[,2], res$map[,2])
abline(0,1)

}
\references{
Adaptive Cauchy prior:

Zhu, A. (2018) Heavy-tailed prior distributions for sequence count data:
removing the noise and preserving large differences. Bioinformatics. doi: 10.1093/bioinformatics/bty895

False sign rate and s-value:

Stephens, M. (2016) False discovery rates: a new deal. Biostatistics, 18:2. doi: 10.1093/biostatistics/kxw041
}
