% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wilcoxonLM.R
\name{wilcoxonLM}
\alias{wilcoxonLM}
\title{Wilcoxon long memory test for a single change in the mean of a long-memory time series.}
\usage{
wilcoxonLM(x, d, tau = 0.15)
}
\arguments{
\item{x}{the univariate numeric vector to be investigated. Missing values are not allowed.}

\item{d}{integer that specifies the long-memory parameter.}

\item{tau}{integer that defines the search area, which is \code{[tau,1-tau]}. Default is \code{tau=0.15} as suggested by Andrews (1993).}
}
\value{
Returns a numeric vector containing the test statistic and the corresponding critical values of the test.
}
\description{
This function performs a Wilcoxon type test for a change-in-mean that is robust under long memory. It applies a consistent estimator of the
long-run variance under long memory and uses a different normalization compared to a standard Wilcoxon test.
The function returns the test statistic as well as critical values.
}
\details{
Note that the critical values are generated for \code{tau=0.15}.
}
\examples{
# set model parameters
T        <- 500
d        <- 0.2

set.seed(410)

# generate a fractionally integrated (long-memory) time series
tseries  <- fracdiff::fracdiff.sim(n=T, d=d)$series

# generate a fractionally integrated (long-memory) time series 
# with a change in mean in the middle of the series
changep  <- c(rep(0,T/2), rep(1,T/2))
tseries2 <- tseries+changep

# estimate the long-memory parameter of both series via local Whittle approach.
# The bandwidth to estimate d is chosen as T^0.65, which is usual in literature
d_est    <- LongMemoryTS::local.W(tseries, m=floor(1+T^0.65))$d
d_est2   <- LongMemoryTS::local.W(tseries2, m=floor(1+T^0.65))$d

# perform the test on both time series
wilcoxonLM(tseries, d=d_est)
wilcoxonLM(tseries2, d=d_est2)
# For the series with no change in mean the test does not reject the null hypothesis
# of a constant mean across time at any reasonable significance level.
# For the series with a change in mean the test rejects the null hypothesis 
# at a 5\% significance level.
}
\references{
Wenger, K. and Leschinski, C. and Sibbertsen, P. (2018): Change-in-mean tests in long-memory time series: a review of recent developments. AStA Advances in Statistical Analysis, 103:2, pp. 237-256.

Dehling, H. and Rooch, A. and Taqqu, M. S. (2012): Non-Parametric Change-Point Tests for Long-Range Dependent Data. Scandinavian Journal of Statistics, 40, pp. 153-173.

Andrews, D. W. K. (1993): Tests for Parameter Instability and Structural Change With Unknown Change Point. Econometrica, 61, pp. 821-856.
}
\seealso{
\code{\link{snwilcoxon}}
}
\author{
Kai Wenger
}
