% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/context.R
\name{request}
\alias{request}
\title{Request over Context (RPC Client for Req/Rep Protocol)}
\usage{
request(
  context,
  data,
  send_mode = c("serial", "raw"),
  recv_mode = c("serial", "character", "complex", "double", "integer", "logical",
    "numeric", "raw", "string"),
  timeout = NULL,
  cv = NULL
)
}
\arguments{
\item{context}{a Context.}

\item{data}{an object (if send_mode = \sQuote{raw}, a vector).}

\item{send_mode}{[default 'serial'] character value or integer equivalent -
either \sQuote{serial} (1L) to send serialised R objects, or \sQuote{raw}
(2L) to send atomic vectors of any type as a raw byte vector.}

\item{recv_mode}{[default 'serial'] character value or integer equivalent -
one of \sQuote{serial} (1L), \sQuote{character} (2L), \sQuote{complex}
(3L), \sQuote{double} (4L), \sQuote{integer} (5L), \sQuote{logical} (6L),
\sQuote{numeric} (7L), \sQuote{raw} (8L), or \sQuote{string} (9L). The
default \sQuote{serial} means a serialised R object; for the other
modes, received bytes are converted into the respective mode.
\sQuote{string} is a faster option for length one character vectors.}

\item{timeout}{[default NULL] integer value in milliseconds or NULL, which
applies a socket-specific default, usually the same as no timeout.}

\item{cv}{(optional) a \sQuote{conditionVariable} to signal when the async
receive is complete, or NULL. If any other value is supplied, this will
cause the pipe connection to be dropped when the async receive is
complete.}
}
\value{
A \sQuote{recvAio} (object of class \sQuote{mirai} and
    \sQuote{recvAio}) (invisibly).
}
\description{
Implements a caller/client for the req node of the req/rep protocol. Sends
    data to the rep node (executor/server) and returns an Aio, which can be
    called for the value when required.
}
\details{
Sending the request and receiving the result are both performed
    async, hence the function will return immediately with a \sQuote{recvAio}
    object. Access the return value at \code{$data}.

    This is designed so that the process on the server can run concurrently
    without blocking the client.

    Optionally use \code{\link{call_aio}} on the \sQuote{recvAio} to call
    (and wait for) the result.

    If an error occured in the server process, a nul byte \code{00} will be
    received. This allows an error to be easily distinguished from a NULL
    return value. \code{\link{is_nul_byte}} can be used to test for a nul
    byte.

    It is recommended to use a new context for each request to ensure
    consistent state tracking. For safety, the context used for the request
    is closed when all references to the returned \sQuote{recvAio} are
    removed and the object is garbage collected.
}
\section{Send Modes}{


    The default mode \sQuote{serial} sends serialised R objects to ensure
    perfect reproducibility within R. When receiving, the corresponding mode
    \sQuote{serial} should be used. Custom serialization and unserialization
    functions for reference objects may be enabled by the function
    \code{\link{serial_config}}.

    Mode \sQuote{raw} sends atomic vectors of any type as a raw byte vector,
    and must be used when interfacing with external applications or raw
    system sockets, where R serialization is not in use. When receiving, the
    mode corresponding to the vector sent should be used.
}

\section{Signalling}{


    By supplying a \sQuote{conditionVariable}, when the receive is complete,
    the \sQuote{conditionVariable} is signalled by incrementing its value by
    1. This happens asynchronously and independently of the R execution
    thread.
}

\examples{
\dontrun{

# works if req and rep are running in parallel in different processes

req <- socket("req", listen = "tcp://127.0.0.1:6546")
rep <- socket("rep", dial = "tcp://127.0.0.1:6546")

reply(.context(rep), execute = function(x) x + 1, timeout = 50)
aio <- request(.context(req), data = 2022)
aio$data

close(req)
close(rep)

# Signalling a condition variable

req <- socket("req", listen = "tcp://127.0.0.1:6546")
ctxq <- context(req)
cv <- cv()
aio <- request(ctxq, data = 2022, cv = cv)
until(cv, 10L)
close(req)

# The following should be run in another process
rep <- socket("rep", dial = "tcp://127.0.0.1:6546")
ctxp <- context(rep)
reply(ctxp, execute = function(x) x + 1)
close(rep)

}

}
