% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom_spatraster.R, R/stat_spatraster.R
\name{geom_spatraster}
\alias{geom_spatraster}
\alias{stat_spatraster}
\title{Visualise SpatRaster objects}
\source{
Based on the \code{layer_spatial()} implementation on ggspatial package.
Thanks to \href{https://github.com/paleolimbot}{Dewey Dunnington} and
\href{https://github.com/paleolimbot/ggspatial/graphs/contributors}{ggspatial contributors}.
}
\usage{
geom_spatraster(
  mapping = aes(),
  data,
  na.rm = TRUE,
  show.legend = NA,
  inherit.aes = FALSE,
  interpolate = FALSE,
  maxcell = 5e+05,
  use_coltab = TRUE,
  ...
)

stat_spatraster(
  mapping = aes(),
  data,
  geom = "raster",
  na.rm = TRUE,
  show.legend = NA,
  inherit.aes = FALSE,
  maxcell = 5e+05,
  ...
)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[ggplot2:aes]{ggplot2::aes()}} or
\code{\link[ggplot2:aes_]{ggplot2::aes_()}}. See \strong{Aesthetics} specially in the use of \code{fill}
aesthetic.}

\item{data}{A SpatRaster object.}

\item{na.rm}{If \code{TRUE}, the default, missing values are silently removed.
If \code{FALSE}, missing values are removed with a warning.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics, rather
than combining with them.}

\item{interpolate}{If \code{TRUE} interpolate linearly, if \code{FALSE}
(the default) don't interpolate.}

\item{maxcell}{positive integer. Maximum number of cells to use for
the plot.}

\item{use_coltab}{Logical. Only applicable to SpatRasters that have a coltab
(\code{\link[terra:colors]{terra::coltab()}}). Should the coltab be used on the plot? See also
\code{\link[=scale_fill_coltab]{scale_fill_coltab()}}.}

\item{...}{Other arguments passed on to \code{\link[ggplot2:layer]{layer()}}. These are
often aesthetics, used to set an aesthetic to a fixed value, like
\code{colour = "red"} or \code{size = 3}. They may also be parameters
to the paired geom/stat.}

\item{geom}{The geometric object to use display the data. Recommended \code{geom}
for SpatRaster are \code{"raster"} (the default), \code{"point"},\code{"text"} and
\code{"label"}.}
}
\value{
A ggplot2 layer
}
\description{
This geom is used to visualise SpatRaster objects (see \code{\link[terra:rast]{terra::rast()}}). The
geom is designed for visualise the object by layers, as \code{\link[terra:plot]{terra::plot()}} does.

For plotting SpatRaster objects as map tiles (i.e. RGB SpatRaster), use
\code{\link[=geom_spatraster_rgb]{geom_spatraster_rgb()}}.

The underlying implementation is based on \code{\link[ggplot2:geom_tile]{ggplot2::geom_raster()}}.

\code{stat_spatraster()} is provided as a complementary function, so the \code{geom}
can be modified.
}
\section{terra equivalent}{


\code{\link[terra:plot]{terra::plot()}}
}

\section{Coords}{


When the SpatRaster does not present a crs (i.e.,
\code{terra::crs(rast) == ""}) the geom does not make any assumption on the
scales.

On SpatRaster that have a crs, the geom uses \code{\link[ggplot2:ggsf]{ggplot2::coord_sf()}} to adjust
the scales. That means that also the \strong{SpatRaster may be reprojected}.
}

\section{Aesthetics}{


\code{geom_spatraster()} understands the following aesthetics:
\itemize{
\item \code{fill}
\item \code{alpha}
}

If \code{fill} is not provided, \code{geom_spatraster()} creates a ggplot2 layer with
all the layers of the SpatRaster object. Use \code{facet_wrap(~lyr)} to display
properly the SpatRaster layers.

If \code{fill} is used, it should contain the name of one layer that is present
on the SpatRaster (i.e.
\verb{geom_spatraster(data = rast, aes(fill = <name_of_lyr>)}). Names of the
layers can be retrieved using \code{names(rast)}.

Using \code{geom_spatraster(..., mapping = aes(fill = NULL))} or
\verb{geom_spatraster(..., fill = <color value(s)>)} would create a layer with no
mapped \code{fill} aesthetic.

\code{fill} can use computed variables.

For \code{alpha} use computed variable. See section \strong{Computed variables}.


\subsection{stat_spatraster()}{

\code{stat_spatraster()} understands the same aesthetics than \code{geom_spatraster()}
when using \code{geom = "raster"} (the default):
\itemize{
\item \code{fill}
\item \code{alpha}
}

When \code{geom = "raster"} the \code{fill} parameter would behave as in
\code{geom_spatraster()}. If another \code{geom} is used \code{stat_spatraster()} would
understand the aesthetics of the required \code{geom} and
\verb{aes(fill = <name_of_lyr>)} would not be applicable.

Note also that mapping of aesthetics \code{x} and \code{y} is provided by default,
so the user does not need to add those aesthetics on \code{aes()}. In all the
cases the aesthetics should be mapped by using computed variables. See
section \strong{Computed variables} and \strong{Examples}.
}
}

\section{Facets}{


You can use \code{ facet_wrap(~lyr)} for creating a faceted plot by each layer of
the SpatRaster object. See \code{\link[ggplot2:facet_wrap]{ggplot2::facet_wrap()}} for details.
}

\section{Computed variables}{


This geom computes internally some variables that are available for use as
aesthetics, using (for example) \code{aes(alpha = after_stat(value))} (see
\code{\link[ggplot2:aes_eval]{ggplot2::after_stat()}}).

\describe{
\item{\code{value}}{Values of the SpatRaster.}
\item{\code{lyr}}{Name of the layer.}
}
}

\examples{
\donttest{
# Avg temperature on spring in Castille and Leon (Spain)
file_path <- system.file("extdata/cyl_temp.tif", package = "tidyterra")

library(terra)
temp_rast <- rast(file_path)

library(ggplot2)

# Display a single layer
names(temp_rast)

ggplot() +
  geom_spatraster(data = temp_rast, aes(fill = tavg_04)) +
  # You can use coord_sf
  coord_sf(crs = 3857) +
  scale_fill_hypso_c()

# Display facets
ggplot() +
  geom_spatraster(data = temp_rast) +
  facet_wrap(~lyr, ncol = 2) +
  scale_fill_hypso_b()


# Non spatial rasters

no_crs <- rast(crs = NA, extent = c(0, 100, 0, 100), nlyr = 1)
values(no_crs) <- seq_len(ncell(no_crs))


ggplot() +
  geom_spatraster(data = no_crs)

# Downsample

ggplot() +
  geom_spatraster(data = no_crs, maxcell = 25)
}

\donttest{
# Using stat_spatraster
# Default
ggplot() +
  stat_spatraster(data = temp_rast) +
  facet_wrap(~lyr)

# Using points
ggplot() +
  stat_spatraster(
    data = temp_rast,
    aes(color = after_stat(value)),
    geom = "point", maxcell = 250
  ) +
  scale_colour_viridis_c(na.value = "transparent") +
  facet_wrap(~lyr)

# Using points and labels

r_single <- temp_rast \%>\% select(1)

ggplot() +
  stat_spatraster(
    data = r_single,
    aes(color = after_stat(value)),
    geom = "point",
    maxcell = 2000
  ) +
  stat_spatraster(
    data = r_single,
    aes(label = after_stat(round(value, 2))),
    geom = "label",
    alpha = 0.85,
    maxcell = 20
  ) +
  scale_colour_viridis_c(na.value = "transparent")
}
}
\seealso{
\code{\link[ggplot2:geom_tile]{ggplot2::geom_raster()}}, \code{\link[ggplot2:ggsf]{ggplot2::coord_sf()}},
\code{\link[ggplot2:facet_wrap]{ggplot2::facet_wrap()}}

Alternative geoms: \code{\link[ggplot2:geom_point]{ggplot2::geom_point()}}, \code{\link[ggplot2:geom_text]{ggplot2::geom_label()}},
\code{\link[ggplot2:geom_text]{ggplot2::geom_text()}}.

Other ggplot2 utils:
\code{\link{autoplot.Spat}},
\code{\link{fortify.Spat}},
\code{\link{geom_spat_contour}},
\code{\link{geom_spatraster_rgb}()},
\code{\link{ggspatvector}},
\code{\link{stat_spat_coordinates}()}
}
\concept{ggplot2.utils}
