% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gcd.R
\name{gcd}
\alias{gcd}
\alias{scm}
\alias{gcd_diff}
\title{Greatest common divisor and smallest common multiple}
\usage{
gcd(
  x,
  tol = sqrt(.Machine$double.eps),
  na_rm = TRUE,
  round = FALSE,
  break_early = TRUE
)

scm(x, tol = sqrt(.Machine$double.eps), na_rm = TRUE, round = FALSE)

gcd_diff(
  x,
  lag = 1L,
  fill = NA,
  tol = sqrt(.Machine$double.eps),
  na_rm = TRUE,
  round = FALSE,
  break_early = TRUE
)
}
\arguments{
\item{x}{A \link{numeric} vector.}

\item{tol}{Tolerance. This must
be a single positive number strictly less than 1.}

\item{na_rm}{If \code{TRUE} the default, \code{NA} values are ignored.}

\item{round}{If \code{TRUE} the output is rounded as
\code{round(gcd, digits)} where digits is
\code{ceiling(abs(log10(tol))) + 1}. \cr
This can potentially reduce floating point errors on
further calculations. \cr
The default is \code{FALSE}.}

\item{break_early}{This is experimental and
applies only to floating-point numbers.
When \code{TRUE} the algorithm will end once \code{gcd > 0 && gcd < 2 * tol}.
This can offer a tremendous speed improvement.
If \code{FALSE} the algorithm finishes once it has gone through all elements of \code{x}.
The default is \code{TRUE}. \cr
For integers, the algorithm always breaks early once \code{gcd > 0 && gcd <= 1}.}

\item{lag}{Lag of differences.}

\item{fill}{Value to initialise the algorithm for \code{gcd_diff()}.}
}
\value{
A number representing the GCD or SCM.
}
\description{
Fast greatest common divisor and smallest common multiple
using the Euclidean algorithm.

\code{gcd()} returns the greatest common divisor. \cr
\code{scm()} returns the smallest common multiple. \cr
\code{gcd_diff()} returns the greatest common divisor of numeric differences.
}
\details{
\subsection{Method}{
\subsection{GCD}{

The GCD is calculated using a binary function that takes input
\code{GCD(gcd, x[i + 1])} where the output of this function is passed as input
back into the same function iteratively along the length of \code{x}.
The first gcd value is \code{x[1]}.

Zeroes are handled in the following way: \cr
\code{GCD(0, 0) = 0} \cr
\code{GCD(a, 0) = a} \cr

This has the nice property that zeroes are essentially ignored.
}

\subsection{SCM}{

This is calculated using the GCD and the formula is: \cr
\code{SCM(x, y) = (abs(x) / GCD(x, y) ) * abs(y)}

If you want to calculate the gcd & lcm for 2 values
instead of a vector of values,
use the internal functions \code{cpp_gcd2} and \code{cpp_lcm2}.
You can then easily write a vectorised method using these.
}

}
}
\examples{
library(timeplyr)
library(bench)
\dontshow{
.n_dt_threads <- data.table::getDTthreads()
.n_collapse_threads <- collapse::get_collapse()$nthreads
data.table::setDTthreads(threads = 2L)
collapse::set_collapse(nthreads = 1L)
}
gcd(c(0, 5, 25))
mark(gcd(c(0, 5, 25)))

x <- rnorm(10^6)
gcd(x)
gcd(x, round = TRUE)
mark(gcd(x))
\dontshow{
data.table::setDTthreads(threads = .n_dt_threads)
collapse::set_collapse(nthreads = .n_collapse_threads)
}
}
