/*
 * Tiny CLI for Mercurial Command Server
 *
 * Copyright (c) 2011 Yuya Nishihara <yuya@tcha.org>
 *
 * This software may be used and distributed according to the terms of the
 * GNU General Public License version 2 or any later version.
 *
 * About Mercurial Command Server:
 * http://mercurial.selenic.com/wiki/CommandServer
 */

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/wait.h>
#include <unistd.h>

#include "hgclient.h"
#include "util.h"

#ifndef UNIX_PATH_MAX
#define UNIX_PATH_MAX (sizeof(((struct sockaddr_un *)NULL)->sun_path))
#endif

static void preparesockdir(const char *sockdir)
{
    int r;
    r = mkdir(sockdir, 0700);
    if (r < 0 && errno != EEXIST)
        abortmsg("cannot create sockdir %s (errno = %d)", sockdir, errno);

    struct stat st;
    r = lstat(sockdir, &st);
    if (r < 0)
        abortmsg("cannot stat %s (errno = %d)", sockdir, errno);
    if (!S_ISDIR(st.st_mode))
        abortmsg("cannot create sockdir %s (file exists)", sockdir);
    if (st.st_uid != geteuid() || st.st_mode & 0077)
        abortmsg("insecure sockdir %s", sockdir);
}

/*
 *    <sockname>                    <pidfile>
 * 1. $CHGSOCKNAME                  $CHGSOCKNAME.pid
 * 2. $TMPDIR/chg$UID/server        $TMPDIR/chg$UID/pid
 */
static void setsockname(char *sockname, char *pidfile)
{
    const char *envsockname = getenv("CHGSOCKNAME");
    if (envsockname) {
        static const char pidsfx[] = ".pid";
        const size_t len = strlen(envsockname);
        if (len + sizeof(pidsfx) > UNIX_PATH_MAX)
            abortmsg("too long CHGSOCKNAME (len = %zu)", len);
        memcpy(sockname, envsockname, len + 1);
        memcpy(pidfile, envsockname, len);
        memcpy(pidfile + len, pidsfx, sizeof(pidsfx));
        return;
    }

    // by default, put socket file in secure directory
    // (permission of socket file may be ignored on some Unices)
    int r;
    char sockdir[256];
    const char *tmpdir = getenv("TMPDIR");
    if (!tmpdir)
        tmpdir = "/tmp";
    r = snprintf(sockdir, sizeof(sockdir), "%s/chg%d", tmpdir, geteuid());
    if (r < 0 || (size_t) r >= sizeof(sockdir))
        abortmsg("too long TMPDIR (r = %d)", r);
    r = snprintf(sockname, UNIX_PATH_MAX, "%s/server", sockdir);
    if (r < 0 || (size_t) r >= UNIX_PATH_MAX)
        abortmsg("too long TMPDIR (r = %d)", r);
    r = snprintf(pidfile, UNIX_PATH_MAX, "%s/pid", sockdir);
    if (r < 0 || (size_t) r >= UNIX_PATH_MAX)
        abortmsg("too long TMPDIR (r = %d)", r);
    preparesockdir(sockdir);
}

static void execcmdserver(const char *sockname, const char *pidfile)
{
    unsetenv("HGPLAIN");
    unsetenv("HGPLAINEXCEPT");
    const char *hgcmd = getenv("CHGHG");
    if (!hgcmd || hgcmd[0] == '\0')
        hgcmd = getenv("HG");
    if (!hgcmd || hgcmd[0] == '\0')
        hgcmd = "hg";
    const char *cmdopts = getenv("CHGCMDSERVEROPTS");
    if (!cmdopts)
        cmdopts = "";

    char cmdline[1024];
    // TODO escape?
    int r = snprintf(cmdline, sizeof(cmdline),
             "'%s' serve --cwd / --cmdserver unix --daemon -a '%s' "
             "--pid-file '%s' "
             "--config extensions.chgsupport= "
             "--config color.mode=chgauto "
             // wrap root ui so that it can be disabled/enabled by config value
             "--config progress.assume-tty=1 "
             "%s", hgcmd, sockname, pidfile, cmdopts);
    if (r < 0 || (size_t) r >= sizeof(cmdline))
        abortmsg("failed to construct cmdline string (r = %d)", r);
    r = execlp("/bin/sh", "/bin/sh", "-c", cmdline, NULL);
    if (r < 0)
        abortmsg("failed to exec cmdserver (errno = %d)", errno);
}

// Spawn new background cmdserver
static void startcmdserver(const char *sockname, const char *pidfile)
{
    debugmsg("start cmdserver at %s", sockname);

    pid_t pid = fork();
    if (pid < 0)
        abortmsg("failed to fork cmdserver process");
    if (pid == 0) {
        // suppress "listening at" message
        int nullfd = open("/dev/null", O_WRONLY);
        if (nullfd >= 0) {
            dup2(nullfd, fileno(stdout));
            close(nullfd);
        }
        execcmdserver(sockname, pidfile);
    } else {
        int status;
        waitpid(pid, &status, 0);
        if (!WIFEXITED(status) || WEXITSTATUS(status) != 0) {
            abortmsg("failed to start cmdserver");
        }
    }
}

static void killcmdserver(const char *pidfile)
{
    FILE *fp = fopen(pidfile, "r");
    if (!fp)
        abortmsg("cannot open %s (errno = %d)", pidfile, errno);
    int pid = 0;
    int n = fscanf(fp, "%d", &pid);
    fclose(fp);
    if (n != 1 || pid <= 0)
        abortmsg("cannot read pid from %s", pidfile);

    if (kill((pid_t)pid, SIGTERM) < 0) {
        if (errno == ESRCH)
            return;
        abortmsg("cannot kill %d (errno = %d)", pid, errno);
    }
}

static hgclient_t *hgc = NULL;

// this isn't signal-safe and may crash, but anyway we're about to abort
static void forwardsignal(int sig)
{
    if (!hgc) return;
    hgc_kill(hgc, sig);
    debugmsg("forward signal %d", sig);
}

static void setupsignalhandler(void)
{
    struct sigaction sa;
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = forwardsignal;
    sa.sa_flags = SA_RESTART;

    sigaction(SIGHUP, &sa, NULL);
    sigaction(SIGINT, &sa, NULL);

    // terminate frontend by double SIGTERM in case of server freeze
    sa.sa_flags |= SA_RESETHAND;
    sigaction(SIGTERM, &sa, NULL);
}

// This implementation is based on hgext/pager.py (pre 369741ef7253)
static void setuppager(hgclient_t *hgc, const char * const args[],
                       size_t argsize)
{
    const char *pagercmd = hgc_getpager(hgc, args, argsize);
    if (!pagercmd)
        return;

    int pipefds[2];
    if (pipe(pipefds) < 0)
        return;
    pid_t pid = fork();
    if (pid < 0)
        goto error;
    if (pid == 0) {
        close(pipefds[0]);
        if (dup2(pipefds[1], fileno(stdout)) < 0)
            goto error;
        if (isatty(fileno(stderr))) {
            if (dup2(pipefds[1], fileno(stderr)) < 0)
                goto error;
        }
        close(pipefds[1]);
        return;
    } else {
        dup2(pipefds[0], fileno(stdin));
        close(pipefds[0]);
        close(pipefds[1]);

        int r = execlp("/bin/sh", "/bin/sh", "-c", pagercmd, NULL);
        if (r < 0) {
            abortmsg("cannot start pager '%s' (errno = %d)", pagercmd, errno);
        }
        return;
    }

error:
    close(pipefds[0]);
    close(pipefds[1]);
    abortmsg("failed to prepare pager (errno = %d)", errno);
}

int main(int argc, const char *argv[], const char *envp[])
{
    if (getenv("CHGDEBUG")) enabledebugmsg();

    char sockname[UNIX_PATH_MAX], pidfile[UNIX_PATH_MAX];
    setsockname(sockname, pidfile);

    if (argc == 2 && strcmp(argv[1], "--kill-chg-daemon") == 0) {
        killcmdserver(pidfile);
        return 0;
    }

    hgc = hgc_open(sockname, envp);
    if (!hgc) {
        // remove dead cmdserver socket and restart it
        if (access(sockname, F_OK) == 0)
            unlink(sockname);
        startcmdserver(sockname, pidfile);
        hgc = hgc_open(sockname, envp);
    }
    if (!hgc) abortmsg("cannot open hg client");

    setupsignalhandler();
    setuppager(hgc, argv + 1, argc - 1);
    int exitcode = hgc_runcommand(hgc, argv + 1, argc - 1);
    hgc_close(hgc);
    hgc = NULL;
    return exitcode;
}
