# chgserver.py - command server extension for cHg
#
# Copyright 2011 Yuya Nishihara <yuya@tcha.org>
#
# This software may be used and distributed according to the terms of the
# GNU General Public License version 2 or any later version.

"""command server extension for cHg

'S' channel (read/write)
    propagate ui.system() request to client

'attachio' command
    attach client's stdio passed by sendmsg()

'chdir' command
    change current directory

'getpager' command
    checks if pager is enabled and which pager should be executed

'setenv' command
    replace os.environ completely

'SIGHUP' signal
    reload configuration files
"""

import errno, os, re, signal, struct, sys
import SocketServer

from mercurial.i18n import _
from mercurial import (
    cmdutil,
    commands,
    commandserver,
    dispatch,
    encoding,
    error,
    extensions,
    i18n,
    scmutil,
    util,
    )

import _chgutil

testedwith = '3.2 3.5 3.6-rc'

_log = commandserver.log

_pagerattended = ['annotate', 'cat', 'diff', 'export', 'glog', 'log', 'qdiff']

# copied from hgext/pager.py:uisetup()
def _setuppagercmd(ui, options, cmd):
    if not ui.formatted():
        return

    p = ui.config("pager", "pager", os.environ.get("PAGER"))
    usepager = False
    always = util.parsebool(options['pager'])
    auto = options['pager'] == 'auto'

    if not p:
        pass
    elif always:
        usepager = True
    elif not auto:
        usepager = False
    else:
        attend = ui.configlist('pager', 'attend', _pagerattended)
        ignore = ui.configlist('pager', 'ignore')
        cmds, _ = cmdutil.findcmd(cmd, commands.table)

        for cmd in cmds:
            var = 'attend-%s' % cmd
            if ui.config('pager', var):
                usepager = ui.configbool('pager', var)
                break
            if (cmd in attend or
                (cmd not in ignore and not attend)):
                usepager = True
                break

    if usepager:
        ui.setconfig('ui', 'formatted', ui.formatted(), 'pager')
        ui.setconfig('ui', 'interactive', False, 'pager')
        return p

_envmodstoreload = [
    (set(['HGENCODING', 'HGENCODINGMODE', 'HGENCODINGAMBIGUOUS',
          'LANG', 'LANGUAGE', 'LC_MESSAGES']), encoding),
    # recreate gettext 't' by reloading i18n
    (set(['LANG', 'LANGUAGE', 'LC_MESSAGES']), i18n),
    ]

def _fixdefaultencoding():
    """Apply new default encoding to commands table"""
    newdefaults = {'encoding': encoding.encoding,
                   'encodingmode': encoding.encodingmode}
    for i, opt in enumerate(commands.globalopts):
        name = opt[1]
        newdef = newdefaults.get(name)
        if newdef is not None:
            commands.globalopts[i] = opt[:2] + (newdef,) + opt[3:]

_envvarre = re.compile(r'\$[a-zA-Z_]+')

def _clearenvaliases(cmdtable):
    """Remove stale command aliases referencing env vars; variable expansion
    is done at dispatch.addaliases()"""
    for name, tab in cmdtable.items():
        cmddef = tab[0]
        if (isinstance(cmddef, dispatch.cmdalias)
            and not cmddef.definition.startswith('!')  # shell alias
            and _envvarre.search(cmddef.definition)):
            del cmdtable[name]

def _renewui(srcui):
    newui = srcui.__class__()
    for a in ['fin', 'fout', 'ferr', 'environ']:
        setattr(newui, a, getattr(srcui, a))
    # stolen from tortoisehg.util.copydynamicconfig()
    for section, name, value in srcui.walkconfig():
        source = srcui.configsource(section, name)
        if ':' in source:
            # path:line
            continue
        if source == 'none':
            # ui.configsource returns 'none' by default
            source = ''
        newui.setconfig(section, name, value, source)
    return newui

class channeledsystem(object):
    """Propagate ui.system() request in the following format:

    payload length (unsigned int),
    cmd, '\0',
    cwd, '\0',
    envkey, '=', val, '\0',
    ...
    envkey, '=', val

    and waits:

    exitcode length (unsigned int),
    exitcode (int)
    """
    def __init__(self, in_, out, channel):
        self.in_ = in_
        self.out = out
        self.channel = channel

    def __call__(self, cmd, environ, cwd):
        args = [util.quotecommand(cmd), cwd or '.']
        args.extend('%s=%s' % (k, v) for k, v in environ.iteritems())
        data = '\0'.join(args)
        self.out.write(struct.pack('>cI', self.channel, len(data)))
        self.out.write(data)
        self.out.flush()

        length = self.in_.read(4)
        length, = struct.unpack('>I', length)
        if length != 4:
            raise error.Abort(_('invalid response'))
        rc, = struct.unpack('>i', self.in_.read(4))
        return rc

class chgcmdserver(commandserver.server):
    def __init__(self, ui, repo, fin, fout, sock):
        super(chgcmdserver, self).__init__(ui, repo, fin, fout)
        self.clientsock = sock
        self.ui._csystem = channeledsystem(fin, fout, 'S')

    def attachio(self):
        """Attach to client's stdio passed via unix domain socket; all
        channels except cresult will no longer be used
        """
        # tell client to sendmsg() with 1-byte payload, which makes it
        # distinctive from "attachio\n" command consumed by client.read()
        self.clientsock.sendall(struct.pack('>cI', 'I', 1))
        clientfds = _chgutil.recvfds(self.clientsock.fileno())
        _log('received fds: %r\n' % clientfds)

        ui = self.ui
        channels = [
            ('cin', ui.fin),
            ('cout', ui.fout),
            ('cerr', ui.ferr),
            ]
        newbufmodefps = [fp for ch, fp in channels[:2]
                         if getattr(self, ch) is not fp]
        for fd, (ch, fp) in zip(clientfds, channels):
            assert fd > 0
            os.dup2(fd, fp.fileno())
            os.close(fd)
            setattr(self, ch, fp)

        # reset to default buffering policy when client is initially attached.
        # as we want to see output immediately on pager, the policy stays
        # unchanged when client re-attached. stderr is unchanged because it
        # should be unbuffered no matter if it is a tty or not.
        for fp in newbufmodefps:
            if fp.isatty():
                m = _chgutil.IOLBF
            else:
                m = _chgutil.IOFBF
            _chgutil.setfilebufmode(fp, m)

        self.cresult.write(struct.pack('>i', len(clientfds)))

    def chdir(self):
        """Change current directory

        Note that the behavior of --cwd option is bit different from this.
        It does not affect --config parameter.
        """
        length = struct.unpack('>I', self._read(4))[0]
        if not length:
            return
        path = self._read(length)
        _log('chdir to %r\n' % path)
        os.chdir(path)

    def getpager(self):
        """Read cmdargs and write pager command to r-channel if enabled

        If pager isn't enabled, this writes '\0' because channeledoutput
        does not allow to write empty data.
        """
        length = struct.unpack('>I', self._read(4))[0]
        if not length:
            args = []
        else:
            args = self._read(length).split('\0')
        try:
            cmd, _func, args, options, _cmdoptions = dispatch._parse(self.ui,
                                                                     args)
        except (error.Abort, error.AmbiguousCommand, error.CommandError,
                error.UnknownCommand):
            cmd = None
            options = {}
        if not cmd or 'pager' not in options:
            self.cresult.write('\0')
            return

        pagercmd = _setuppagercmd(self.ui, options, cmd)
        if pagercmd:
            self.cresult.write(pagercmd)
        else:
            self.cresult.write('\0')

    def setenv(self):
        """Clear and update os.environ

        Note that not all variables can make an effect on the running process.
        """
        length = struct.unpack('>I', self._read(4))[0]
        if not length:
            return
        s = self._read(length)
        try:
            newenv = dict(l.split('=', 1) for l in s.split('\0'))
        except ValueError:
            raise ValueError('unexpected value in setenv request')

        diffkeys = set(k for k in set(os.environ.keys() + newenv.keys())
                       if os.environ.get(k) != newenv.get(k))
        _log('change env: %r\n' % sorted(diffkeys))

        os.environ.clear()
        os.environ.update(newenv)

        for keys, mod in _envmodstoreload:
            if not keys & diffkeys:
                continue
            _log('reload %s module\n' % mod.__name__)
            reload(mod)
            if mod is encoding:
                _fixdefaultencoding()

        if set(['HGPLAIN', 'HGPLAINEXCEPT']) & diffkeys:
            # reload config so that ui.plain() takes effect
            for f in scmutil.rcpath():
                self.ui.readconfig(f, trust=True)

        _clearenvaliases(commands.table)

    capabilities = commandserver.server.capabilities.copy()
    capabilities.update({'attachio': attachio,
                         'chdir': chdir,
                         'getpager': getpager,
                         'setenv': setenv})

# copied from mercurial/commandserver.py
class _requesthandler(SocketServer.StreamRequestHandler):
    def handle(self):
        ui = self.server.ui
        repo = self.server.repo
        # reset time-stamp so that "progress.delay" can take effect
        progbar = getattr(ui, '_progbar', None)
        if progbar:
            progbar.resetstate()
        sv = chgcmdserver(ui, repo, self.rfile, self.wfile, self.connection)
        try:
            try:
                sv.serve()
            # handle exceptions that may be raised by command server. most of
            # known exceptions are caught by dispatch.
            except error.Abort as inst:
                ui.warn(_('abort: %s\n') % inst)
            except IOError as inst:
                if inst.errno != errno.EPIPE:
                    raise
            except KeyboardInterrupt:
                pass
        finally:
            # dispatch._runcatch() does not flush outputs if exception is not
            # handled by dispatch._dispatch()
            ui.flush()

class chgunixservice(commandserver.unixservice):
    def init(self):
        # drop options set for "hg serve --cmdserver" command
        self.ui.setconfig('progress', 'assume-tty', None)
        signal.signal(signal.SIGHUP, self._reloadconfig)
        class cls(SocketServer.ForkingMixIn, SocketServer.UnixStreamServer):
            ui = self.ui
            repo = self.repo
        self.server = cls(self.address, _requesthandler)
        # avoid writing "listening at" message to stdout before attachio
        # request, which calls setvbuf()

    def _reloadconfig(self, signum, frame):
        self.ui = self.server.ui = _renewui(self.ui)

def _dispatch(orig, req):
    uis = [req.ui]
    if req.repo:
        uis.append(req.repo.ui)
    for ui in uis:
        # allow interaction with tty if client's stdio is attached
        if (req.fin is sys.stdin
            and ui.configsource('ui', 'nontty') == 'commandserver'):
            ui.setconfig('ui', 'nontty', None)
    return orig(req)

def _wrapui(ui):
    class chgui(ui.__class__):
        def __init__(self, src=None):
            super(chgui, self).__init__(src)
            if src:
                # src might not have _csystem even if it is a chgui because
                # ui attributes are not copied from lui after uisetup()
                self._csystem = getattr(src, '_csystem', None)
            else:
                self._csystem = None

        def system(self, cmd, environ={}, cwd=None, onerr=None, errprefix=None):
            if not self._csystem:
                return super(chgui, self).system(cmd, environ, cwd,
                                                 onerr, errprefix)
            # copied from mercurial/util.py:system()
            self.flush()
            def py2shell(val):
                if val is None or val is False:
                    return '0'
                if val is True:
                    return '1'
                return str(val)
            env = os.environ.copy()
            env.update((k, py2shell(v)) for k, v in environ.iteritems())
            env['HG'] = util.hgexecutable()
            rc = self._csystem(cmd, env, cwd)
            if rc and onerr:
                errmsg = '%s %s' % (os.path.basename(cmd.split(None, 1)[0]),
                                    util.explainexit(rc)[0])
                if errprefix:
                    errmsg = '%s: %s' % (errprefix, errmsg)
                raise onerr(errmsg)
            return rc

    ui.__class__ = chgui
    ui._csystem = None

def uisetup(ui):
    commandserver._servicemap['chgunix'] = chgunixservice
    extensions.wrapfunction(dispatch, 'dispatch', _dispatch)
    _wrapui(ui)
