/*
 * Tiny CLI for Mercurial Command Server
 *
 * Copyright (c) 2011 Yuya Nishihara <yuya@tcha.org>
 *
 * This software may be used and distributed according to the terms of the
 * GNU General Public License version 2 or any later version.
 *
 * About Mercurial Command Server:
 * http://mercurial.selenic.com/wiki/CommandServer
 */

#include <signal.h>
#include <stdlib.h>
#include <string.h>

#include "hgclient.h"
#include "misc.h"
#include "util.h"

static hgclient_t *hgc = NULL;

// this isn't signal-safe and may crash, but anyway we're about to abort
static void forwardsignal(int sig)
{
    if (!hgc) return;
    hgc_kill(hgc, sig);
    debugmsg("forward signal %d", sig);
}

static void setupsignalhandler(void)
{
    struct sigaction sa;
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = forwardsignal;
    sa.sa_flags = SA_RESTART;

    sigaction(SIGHUP, &sa, NULL);
    sigaction(SIGINT, &sa, NULL);

    // terminate frontend by double SIGTERM in case of server freeze
    sa.sa_flags |= SA_RESETHAND;
    sigaction(SIGTERM, &sa, NULL);
}

int main(int argc, const char *argv[], const char *envp[])
{
    if (getenv("CHGDEBUG")) enabledebugmsg();
    setupsignalhandler();

    hgc = hgc_open(envp);
    if (!hgc) abortmsg("cannot open hg client");

    int exitcode = 0;
    if (argc == 2 && strcmp(argv[1], "--kill-chg-daemon") == 0) {
        hgc_killmaster(hgc, SIGTERM);
    } else {
        setuppager(hgc, argv + 1, argc - 1);
        exitcode = hgc_runcommand(hgc, argv + 1, argc - 1);
    }
    hgc_close(hgc);
    hgc = NULL;
    return exitcode;
}
